/*
 * Copyright (c) 2015 Chimera IoT
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.chimeraiot.android.ble;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.os.Build;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/** BLE devices list scanner. */
@TargetApi(Build.VERSION_CODES.LOLLIPOP)
public class BleScannerLollipop extends ScanCallback implements Runnable, BleScanner {
    /** Logging tag. */
    private static final String TAG = BleScannerLollipop.class.getSimpleName();

    /** Bluetooth adapter. */
    private final BluetoothAdapter bluetoothAdapter;

    /** Scan period. */
    private long scanPeriod = DEFAULT_SCAN_PERIOD;
    /** Scan mode. */
    private int scanMode = ScanSettings.SCAN_MODE_LOW_LATENCY;
    /** Device filter. */
    private List<ScanFilter> filter = new ArrayList<>();
    /** Scan thread. */
    private Thread scanThread;
    /** Indicates whether scanner is running. */
    private volatile boolean isScanning = false;
    /** Scan process listener. */
    private BleDevicesScannerListener listener;

    public BleScannerLollipop(BluetoothAdapter adapter, BleDevicesScannerListener listener) {
        if (adapter == null) {
            throw new IllegalArgumentException("Adapter should not be null");
        }

        this.listener = listener;
        bluetoothAdapter = adapter;
    }

    @Override
    @SuppressWarnings("UnusedDeclaration")
    public synchronized void setScanPeriod(long scanPeriod) {
        this.scanPeriod = scanPeriod < 0 ? PERIOD_SCAN_ONCE : scanPeriod;
    }

    @Override
    public void setScanMode(int scanMode) {
        this.scanMode = scanMode;
    }

    @Override
    public boolean isScanning() {
        return scanThread != null && scanThread.isAlive();
    }

    @Override
    public synchronized void start() {
        start(Collections.<String>emptyList());
    }

    /**
     * Starts scanning for specified devices.
     * @param filter list of device names to search.
     */
    @Override
    public synchronized void start(Collection<String> filter) {
        if (isScanning()) {
            return;
        }

        this.filter.clear();
        for (String name : filter) {
            this.filter.add(new ScanFilter.Builder().setDeviceName(name).build());
        }

        if (scanThread != null) {
            scanThread.interrupt();
        }
        scanThread = new Thread(this);
        scanThread.setName(TAG);
        scanThread.start();
    }

    @Override
    public synchronized void stop() {
        isScanning = false;
        if (scanThread != null) {
            scanThread.interrupt();
            scanThread = null;
        }
        stopScanner();
    }

    private synchronized void stopScanner() {
        final BluetoothLeScanner scanner = bluetoothAdapter.getBluetoothLeScanner();
        if (scanner != null) {
            scanner.stopScan(this);
        }
    }

    @Override
    public void onScanResult(final int callbackType,
            final ScanResult result) {
        super.onScanResult(callbackType, result);
        final byte[] bytes = result.getScanRecord() != null
                ? result.getScanRecord().getBytes() : null;
        listener.onLeScan(result.getDevice(), result.getRssi(), bytes);
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    @Override
    public void run() {
        try {
            isScanning = true;
            listener.onScanStarted();

            do {
                synchronized (this) {
                    final ScanSettings settings = new ScanSettings.Builder()
                            .setScanMode(scanMode)
                            .build();
                    final BluetoothLeScanner scanner = bluetoothAdapter.getBluetoothLeScanner();
                    if (scanner != null) {
                        scanner.startScan(filter, settings, this);
                    }
                }

                if (scanPeriod > 0) {
                    Thread.sleep(scanPeriod);
                } else {
                    Thread.sleep(DEFAULT_SCAN_PERIOD);
                }

                stopScanner();

                if (scanPeriod > 0) {
                    listener.onScanRepeat();
                }
            } while (isScanning && scanPeriod > 0);
            //CHECKSTYLE:OFF
        } catch (InterruptedException ignore) {
        } finally {
            //CHECKSTYLE:ON
            stopScanner();
        }

        listener.onScanStopped();
    }

}
