package com.childcare.common;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.util.AttributeSet;
import android.widget.FrameLayout;

/**
 * CouponView: a special customized view of coupon.
 * <p>setting right round rectangle color {@link #setAccentColor(int)}</p>
 *
 * @author john
 * @since 2017-08-14
 */
public class CouponView extends FrameLayout {
    /**
     * the paint brash
     */
    private Paint mPaint;
    /**
     * mini circles radius and triangle length
     */
    private float mRadius;
    /**
     * percent of the triangles divider
     */
    private float mPercent;
    /**
     * draw triangles count
     */
    private int mTriangleCount;
    /**
     * offset between two triangles
     */
    private float mOffset;
    /**
     * the right round rectangle color and triangles color
     */
    private int mAccentColor;
    /**
     * both mini circle color
     */
    private int mDecentColor;
    /**
     * left round rectangle color
     */
    private int mContentColor;


    public CouponView(Context context) {
        this(context, null);
    }

    public CouponView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public CouponView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        TypedArray a = context.getTheme()
                .obtainStyledAttributes(attrs, R.styleable.CouponView, 0, 0);

        mAccentColor = a.getColor(R.styleable.CouponView_couponAccentColor,
                ContextCompat.getColor(context, R.color.orange));
        mDecentColor = a.getColor(R.styleable.CouponView_couponDecentColor,
                ContextCompat.getColor(context, R.color.grey));
        mContentColor = a.getColor(R.styleable.CouponView_couponContentColor,
                ContextCompat.getColor(context, R.color.white));
        mRadius = a.getDimension(R.styleable.CouponView_couponRadius, 16);
        mPercent = a.getFloat(R.styleable.CouponView_couponPercent, 0.7f);
        a.recycle();
        mPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        setBackgroundColor(Color.TRANSPARENT);
    }

    private void setDecentPaint() {
        mPaint.setColor(mDecentColor);
    }

    private void setAccentPaint() {
        mPaint.setColor(mAccentColor);
    }

    private void setContentPaint() {
        mPaint.setColor(mContentColor);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        mTriangleCount = (int) ((h - 2 * mRadius) / mRadius);
        mOffset = (h - 2 * mRadius) % mRadius / mTriangleCount;
        invalidate();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        float x = mPercent * getWidth();
        drawRoundRectLeft(canvas, x);
        drawRoundRectRight(canvas, x);
        drawCircles(canvas, x);
        drawTriangles(canvas, x);
    }

    /**
     * Draw a jagged triangle
     */
    private void drawTriangles(Canvas canvas, float x) {
        setAccentPaint();
        for (int i = 0; i < mTriangleCount; i++) {
            canvas.drawPath(calculateTriangle(x, mRadius + i * (mRadius + mOffset)), mPaint);
        }

    }

    /**
     * Calculate a single triangular path
     *
     * @param firstPointX The x-coordinate of the first point
     * @param startPointY The Y-coordinate of the first point
     * @return Closed triangle path
     */
    private Path calculateTriangle(float firstPointX, float startPointY) {
        Path path = new Path();
        path.moveTo(firstPointX, startPointY);
        path.lineTo(firstPointX, startPointY + mRadius);
        path.lineTo(firstPointX - mRadius / 2, startPointY + mRadius / 2);
        path.close();
        return path;
    }

    /**
     * Draw right round rectangle
     */
    private void drawRoundRectRight(Canvas canvas, float x) {
        setAccentPaint();
        canvas.drawRoundRect(new RectF(x, 0, getWidth(), getHeight()), mRadius, mRadius, mPaint);
    }

    /**
     * Draw left round rectangle
     */
    private void drawRoundRectLeft(Canvas canvas, float x) {
        setContentPaint();
        canvas.drawRoundRect(new RectF(0, 0, x, getHeight()), mRadius, mRadius, mPaint);
    }

    /**
     * Draw top and bottom circles
     */
    private void drawCircles(Canvas canvas, float x) {
        setDecentPaint();
        canvas.drawCircle(x, 0, mRadius, mPaint);
        canvas.drawCircle(x, getHeight(), mRadius, mPaint);
    }

    /**
     * setting right round rectangle color
     *
     * @param accentColor specified color
     */
    public void setAccentColor(int accentColor) {
        mAccentColor = accentColor;
        invalidate();
    }
}
