package com.bytedance.ttgame.module.netdiagnose.api

import android.content.Context
import com.bytedance.ttgame.framework.module.spi.IModuleApi
import com.bytedance.ttgame.module.netdiagnose.api.data.DnsAddressResult
import com.bytedance.ttgame.module.netdiagnose.api.data.DownloadSpeedResultCallback
import com.bytedance.ttgame.module.netdiagnose.api.data.IpAddressResult
import com.bytedance.ttgame.module.netdiagnose.api.data.PingIPResult

interface INetDiagnoseService : IModuleApi {
    /**
     * 获取当前网络类型
     *
     * @param context Context对象
     * @return [Int] 返回Int类型的值，以下是Int值对应的含义:
     *  - -1    :   未知错误
     *  - 0     :   未知网络
     *  - 1     :   WIFI
     *  - 2     :   2G
     *  - 3     :   3G
     *  - 4     :   4G
     *  - 5     :   5G
     */
    fun getNetworkType(context: Context?): Int

    /**
     * 获取 local ipv4 地址和 ipv6 地址
     *
     * @param context Context对象
     * @return [IpAddressResult] 如果某个地址没有获取到，则返回空字符串[""]
     */
    fun getIPAddress(context: Context?): IpAddressResult

    /**
     * 获取 local 网关地址
     *
     * @param context Context对象
     * @return [String] 如果未获取到网关地址，则返回空字符串[""]
     */
    fun getGatewayAddress(context: Context?): String

    /**
     * 获取 local dns 地址列表
     *
     * @param context Context对象
     * @return [DnsAddressResult] 如果未获取到Dns列表，则返回的[DnsAddressResult]对象中为size为0的列表
     */
    fun getDnsList(context: Context?): DnsAddressResult

    /**
     * ping 方法
     *
     * @param context Context对象
     * @param domain 需要ping的ip地址或者域名
     * @param connectTimes ping的次数
     * @return [PingIPResult] 其中，[PingIPResult]对象中各个字段的含义如下：
     *  - totalPacket	    :	总传输包量
     *  - receivedPacket	:	送达传输包量
     *  - packetLoss	    :	丢包率
     *  - minDelay	        :	最小延迟
     *  - maxDelay	        :	最大延迟
     *  - avgDelay	        :	平均延迟
     *  - stddev	        :	延迟标准差
     */
    fun pingMethod(context: Context?, domain: String?, connectTimes: Int): PingIPResult

    /**
     * 下载测速方法
     *
     * @param context Context对象
     * @param fileUrl 测试下载文件地址
     * @param callback 下载结果回调 [DownloadSpeedResultCallback]
     */
    fun downloadSpeed(context: Context?, fileUrl: String?, callback: DownloadSpeedResultCallback?)

    /**
     * 域名解析方法
     *
     * @param hostName 待解析的域名
     * @return [Array<String>] 解析结果，如果解析发生错误，将会返回length为0的数组
     */
    fun resolveHostName(hostName: String?): Array<String>
}