package com.bytedance.ttgame.module.gpm.api;

import com.bytedance.ttgame.framework.module.spi.IModuleApi;
import com.bytedance.ttgame.framework.module.spi.annotation.InternalApi;

import java.util.Map;

/**
 * GPM相关接口，提供自动化的性能采集和上报能力
 *
 * @author 陈海涛（chenhaitao.jason）
 * @since 10:53 AM
 */
@InternalApi
public interface IGPMService extends IModuleApi {
    /**
     * 自动化采集开始，对应品质度量的通用接口能力
     * 在模块确定的较早位置调用
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param extra     额外附加到该场景下上报的信息
     */
    void logSceneStart(String sceneName, String module, Map<String, Object> extra);

    /**
     * 自动化采集结束，并触发开始上报，对应品质度量的通用接口能力
     * 在模块确定退出后的位置调用
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"，需要和前面调用start处对应
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     */
    void logSceneEnd(String sceneName, String module);

    /**
     * 自动化采集结束，并提供参数决定是否将当前这段数据上报。用于异常场景下丢弃数据，这时可考虑传入upload为false
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"，需要和前面调用start处对应
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param upload    是否上报数据
     */
    void logSceneEnd(String sceneName, String module, boolean upload);

    /**
     * 为模块场景添加自定义信息，在sceneStop前上报，最终在sceneStop调用添加到公参上报
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param key       自定义数据关键字
     * @param value     自定义数据
     */
    void logSceneInfo(String sceneName, String module, String key, int value);

    /**
     * 为模块场景添加自定义信息，在sceneStop前上报，最终在sceneStop调用添加到公参上报
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param key       自定义数据关键字
     * @param value     自定义数据
     */
    void logSceneInfo(String sceneName, String module, String key, String value);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 自行处理上报数据
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param data      附加数据
     */
    void monitorReport(String sceneName, String module, Map<String, Object> data);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 自行计算耗时，这里仅做上报
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param cost      耗时（ms）
     * @param data      附加数据
     */
    void monitorCost(String sceneName, String module, long cost, Map<String, Object> data);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 可自动计算耗时，这里为开始
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param unique    唯一标示（用于比如区分同一场景、同一模块，但是内容不同的事件，比如IM消息的发送），可以传空字串""忽略
     * @param data      附加数据
     */
    void monitorCostStart(String sceneName, String module, String unique, Map<String, Object> data);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 处理多阶段耗时，最终上报多个阶段耗时的数组
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param unique    唯一标示（用于比如区分同一场景、同一模块，但是内容不同的事件，比如IM消息的发送），可以传空字串""忽略
     */
    void monitorCostStage(String sceneName, String module, String unique);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 结束计算并上报
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param unique    唯一标示（用于比如区分同一场景、同一模块，但是内容不同的事件，比如IM消息的发送），可以传空字串""忽略
     * @param data      附加数据
     */
    void monitorCostEnd(String sceneName, String module, String unique, Map<String, Object> data);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 用于成功率计算，这里在某事件成功时调用
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param data      附加数据
     */
    void monitorRateSuccess(String sceneName, String module, Map<String, Object> data);

    /**
     * 完全自定义的上报，对应品质度量的自定义接口
     * 用于成功率计算，这里在某事件结束时调用
     *
     * @param sceneName 场景名，规则"sn_xxx"，例如"sn_douyin_login"
     * @param module    模块名，规则"m_xxx"，例如"m_login"
     * @param data      附加数据
     */
    void monitorRateFailed(String sceneName, String module, Map<String, Object> data);

    /**
     * 触发内存采集
     */
    void logMemoryUsage();
}
