package com.bytedance.ttgame.main.internal.net;

import android.os.Looper;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.WorkerThread;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MediatorLiveData;

import com.bytedance.ttgame.framework.module.util.Objects;

/**
 * 无缓存实现版
 *
 * @param <ResultType>
 */
public abstract class NetworkOnlyBoundResource<ResultType> {


    private final MediatorLiveData<Resource<ResultType>> result = new MediatorLiveData<>();


    public NetworkOnlyBoundResource() {
        if (Looper.getMainLooper().getThread() == Thread.currentThread()) {
            result.setValue(Resource.loading(null));
        } else {
            result.postValue(Resource.loading(null));
        }
        fetchFromNetwork();
    }

    @MainThread
    private void setValue(Resource<ResultType> newValue) {
        if (!Objects.equals(result.getValue(), newValue)) {
            result.setValue(newValue);
        }
    }

    private void fetchFromNetwork() {
        LiveData<ApiResponse<ResultType>> apiResponse = createCall();
        //看到这里一定有疑问？apiResponse是何时进行setValue或者postValue的->答案在 LiveDataCallAdapter
        result.addSource(apiResponse, response -> {
            result.removeSource(apiResponse);
            if (response != null && response.isSuccessful()) {
                setValue(Resource.success(processResponse(response), response.logId));
            } else {
                onFetchFailed();
                setValue(Resource.error(response.errorMessage, null, response.logId));
            }
        });
    }

    protected void onFetchFailed() {
    }

    public LiveData<Resource<ResultType>> asLiveData() {
        return result;
    }

    @WorkerThread
    private ResultType processResponse(ApiResponse<ResultType> response) {
        return response.body;
    }


    @NonNull
    @MainThread
    protected abstract LiveData<ApiResponse<ResultType>> createCall();
}
