/*
 * MIT License
 *
 * Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.graphics.drawable.BitmapDrawable;
import android.os.Build;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;

/**
 * Set image action for {@link ImageLoader}
 */
final class SetImageAction<T> implements Runnable {
    private final BitmapDrawable mBitmapDrawable;
    private final ImageLoader<T> mImageLoader;
    private final LoadImageAction<T> mLoadImageAction;
    private final Callbacks<T> mCallbacks;

    public SetImageAction(@Nullable BitmapDrawable bitmapDrawable,
            @NonNull ImageLoader<T> imageLoader, @Nullable Callbacks<T> callbacks,
            @NonNull LoadImageAction<T> loadImageAction) {
        mBitmapDrawable = bitmapDrawable;
        mImageLoader = imageLoader;
        mCallbacks = callbacks;
        mLoadImageAction = loadImageAction;
    }

    @Override
    @MainThread
    public void run() {
        if (!mLoadImageAction.isCancelled() && !mImageLoader.isExitTasksEarly()) {
            View view = mLoadImageAction.getAttachedView();
            if (mBitmapDrawable == null || view == null) {
                return;
            }
            ImageSource<T> source = mLoadImageAction.getImageSource();
            if (mImageLoader.isImageFadeIn() &&
                    Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                FadeDrawable fadeDrawable =
                        new FadeDrawable(mImageLoader.getPlaceholderDrawable(source),
                                mBitmapDrawable);
                mImageLoader.setDrawable(view, fadeDrawable);
                fadeDrawable.setFadeEndListener(
                        new FadeEndListener<>(mCallbacks, source, mBitmapDrawable, view));
                fadeDrawable.startFade(mImageLoader.getImageFadeInTime());
            } else {
                mImageLoader.setDrawable(view, mBitmapDrawable);
                Callbacks.notifyDisplayed(mCallbacks, source, mBitmapDrawable, view);
            }
        }
    }

    private static final class FadeEndListener<T> implements FadeDrawable.FadeEndListener {
        private final Callbacks<T> mCallbacks;
        private final ImageSource<T> mSource;
        private final BitmapDrawable mDrawable;
        private final View mView;

        public FadeEndListener(@Nullable Callbacks<T> callbacks, @NonNull ImageSource<T> source,
                @NonNull BitmapDrawable drawable, @NonNull View view) {
            mCallbacks = callbacks;
            mSource = source;
            mDrawable = drawable;
            mView = view;
        }

        @Override
        public void onEnd(@NonNull FadeDrawable drawable) {
            Callbacks.notifyDisplayed(mCallbacks, mSource, mDrawable, mView);
        }
    }
}
