/*
 * MIT License
 *
 * Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.ColorFilter;
import android.graphics.ColorMatrix;
import android.graphics.ColorMatrixColorFilter;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.net.Uri;
import android.support.annotation.FloatRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.util.DisplayMetrics;
import android.util.TypedValue;

import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

@SuppressWarnings("SameParameterValue")
public final class ImageLoaderUtils {
    private static final String HASH_ALGORITHM_SHA256 = "SHA-256";
    private static final String DEFAULT_STORAGE_CACHE_DIRECTORY = "image_loader_cache";
    private static final Bitmap.CompressFormat DEFAULT_COMPRESS_FORMAT = Bitmap.CompressFormat.JPEG;
    private static final double DEFAULT_STORAGE_FRACTION = 0.1d;
    private static final float DEFAULT_MEMORY_FRACTION = 0.25f;
    private static final int DEFAULT_COMPRESS_QUALITY = 85;

    private ImageLoaderUtils() {
    }

    /**
     * Fraction of maximum number of bytes heap can expand to
     *
     * @param fraction Fraction
     * @return Number of bytes
     */
    public static int getMaxMemoryFraction(@FloatRange(from = 0.1, to = 0.8) float fraction) {
        if (fraction < 0.1F || fraction > 0.8f) {
            throw new IllegalArgumentException(
                    "Argument \"fraction\" must be between 0.1 and 0.8 (inclusive)");
        }
        return Math.round(fraction * Runtime.getRuntime().maxMemory());
    }

    /**
     * Fraction of available storage space in specified path
     *
     * @param path     Path
     * @param fraction Fraction
     * @return Number of bytes
     */
    public static long getAvailableStorageFraction(@NonNull File path,
            @FloatRange(from = 0.01, to = 1.0) double fraction) {
        if (fraction < 0.01D || fraction > 1.0d) {
            throw new IllegalArgumentException(
                    "Argument \"fraction\" must be between 0.01 and 1.0 (inclusive)");
        }
        return Math.round(InternalUtils.getAvailableBytes(path) * fraction);
    }

    /**
     * Fraction of total storage space in specified path
     *
     * @param path     Path
     * @param fraction Fraction
     * @return Number of bytes
     */
    public static long getTotalStorageFraction(@NonNull File path,
            @FloatRange(from = 0.01, to = 1.0) double fraction) {
        if (fraction < 0.01D || fraction > 1.0d) {
            throw new IllegalArgumentException(
                    "Argument \"fraction\" must be between 0.01 and 1.0 (inclusive)");
        }
        return Math.round(InternalUtils.getTotalBytes(path) * fraction);
    }

    /**
     * Calculate sample size for required size from source size
     * Sample size is the number of pixels in either dimension that
     * correspond to a single pixel
     *
     * @param sourceWidth               Source width
     * @param sourceHeight              Source height
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Sample size
     */
    public static int calculateSampleSize(int sourceWidth, int sourceHeight, int requiredWidth,
            int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        int sampleSize = 1;
        if (sourceWidth > requiredWidth || sourceHeight > requiredHeight) {
            int halfWidth = sourceWidth / 2;
            int halfHeight = sourceHeight / 2;
            while ((halfWidth / sampleSize) > requiredWidth &&
                    (halfHeight / sampleSize) > requiredHeight) {
                sampleSize *= 2;
            }
            if (ignoreTotalNumberOfPixels) {
                return sampleSize;
            }
            long totalPixels = (sourceWidth * sourceHeight) / (sampleSize * sampleSize);
            long totalRequiredPixels = requiredWidth * requiredHeight;
            while (totalPixels > totalRequiredPixels) {
                sampleSize *= 2;
                totalPixels /= 4L;
            }
        }
        return sampleSize;
    }

    /**
     * Load sampled bitmap from uri
     *
     * @param context                   Context
     * @param uri                       Uri
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromUri(@NonNull Context context, @NonNull Uri uri,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) throws
            IOException {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        InputStream inputStream = null;
        try {
            inputStream = InternalUtils.getDataStreamFromUri(context, uri);
            BitmapFactory.decodeStream(inputStream, null, options);
        } finally {
            InternalUtils.close(inputStream);
        }
        options.inJustDecodeBounds = false;
        options.inSampleSize =
                calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                        requiredHeight, ignoreTotalNumberOfPixels);
        inputStream = null;
        try {
            inputStream = InternalUtils.getDataStreamFromUri(context, uri);
            return BitmapFactory.decodeStream(inputStream, null, options);
        } finally {
            InternalUtils.close(inputStream);
        }
    }

    /**
     * Load sampled bitmap from file
     *
     * @param file                      File
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromFile(@NonNull File file, int requiredWidth,
            int requiredHeight, boolean ignoreTotalNumberOfPixels) throws FileNotFoundException {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        InputStream inputStream = null;
        try {
            inputStream = new FileInputStream(file);
            BitmapFactory.decodeStream(inputStream, null, options);
        } finally {
            InternalUtils.close(inputStream);
        }
        options.inJustDecodeBounds = false;
        options.inSampleSize =
                calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                        requiredHeight, ignoreTotalNumberOfPixels);
        inputStream = null;
        try {
            inputStream = new FileInputStream(file);
            return BitmapFactory.decodeStream(inputStream, null, options);
        } finally {
            InternalUtils.close(inputStream);
        }
    }

    /**
     * Load sampled bitmap from file descriptor
     *
     * @param fileDescriptor            File descriptor
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromFileDescriptor(@NonNull FileDescriptor fileDescriptor,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        InputStream inputStream = new FileInputStream(fileDescriptor);
        BitmapFactory.decodeStream(inputStream, null, options);
        InternalUtils.close(inputStream);
        options.inJustDecodeBounds = false;
        options.inSampleSize =
                calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                        requiredHeight, ignoreTotalNumberOfPixels);
        inputStream = new FileInputStream(fileDescriptor);
        Bitmap bitmap = BitmapFactory.decodeStream(inputStream, null, options);
        InternalUtils.close(inputStream);
        return bitmap;
    }

    /**
     * Load sampled bitmap from resource
     *
     * @param resources                 Resources
     * @param resourceId                Resource id
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromResource(@NonNull Resources resources, int resourceId,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        TypedValue typedValue = new TypedValue();
        options.inJustDecodeBounds = true;
        options.inTargetDensity = resources.getDisplayMetrics().densityDpi;
        InputStream inputStream = resources.openRawResource(resourceId, typedValue);
        if (typedValue.density == TypedValue.DENSITY_DEFAULT) {
            options.inDensity = DisplayMetrics.DENSITY_DEFAULT;
        } else if (typedValue.density != TypedValue.DENSITY_NONE) {
            options.inDensity = typedValue.density;
        }
        BitmapFactory.decodeStream(inputStream, null, options);
        InternalUtils.close(inputStream);
        options.inJustDecodeBounds = false;
        options.inSampleSize =
                calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                        requiredHeight, ignoreTotalNumberOfPixels);
        inputStream = resources.openRawResource(resourceId, typedValue);
        Bitmap bitmap = BitmapFactory.decodeStream(inputStream, null, options);
        InternalUtils.close(inputStream);
        return bitmap;

    }

    /**
     * Load sampled bitmap from byte array
     *
     * @param byteArray                 Byte array
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromByteArray(@NonNull byte[] byteArray,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeByteArray(byteArray, 0, byteArray.length, options);
        options.inSampleSize =
                calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                        requiredHeight, ignoreTotalNumberOfPixels);
        options.inJustDecodeBounds = false;
        return BitmapFactory.decodeByteArray(byteArray, 0, byteArray.length, options);
    }

    /**
     * Create new common bitmap loader for uris
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<Uri> newUriBitmapLoader() {
        return new UriBitmapLoader();
    }

    /**
     * Create new common bitmap loader for files
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<File> newFileBitmapLoader() {
        return new FileBitmapLoader();
    }

    /**
     * Create new common bitmap loader for file descriptors
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<FileDescriptor> newFileDescriptorBitmapLoader() {
        return new FileDescriptorBitmapLoader();
    }

    /**
     * Create new common bitmap loader for resources
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<Integer> newResourceBitmapLoader() {
        return new ResourceBitmapLoader();
    }

    /**
     * Create new common bitmap loader for byte arrays
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<byte[]> newByteArrayBitmapLoader() {
        return new ByteArrayBitmapLoader();
    }

    /**
     * Create new common image source that is usable in most cases
     * <br>
     * SHA-256 hash of {@link String#valueOf(Object)} of {@code data} will be used as a key.
     *
     * @param data Source data
     * @return Image source
     */
    @NonNull
    public static <T> ImageSource<T> newImageSource(@NonNull T data) {
        return new ImageSourceImpl<>(data);
    }

    /**
     * Generate SHA-256 hash string with {@link Character#MAX_RADIX} radix
     * for specified {@link String}; usable for keys of {@link ImageSource} implementations
     *
     * @param string Source string
     * @return SHA-256 hash string
     * @see ImageSource#getKey()
     */
    @NonNull
    public static String generateSHA256(@NonNull String string) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance(HASH_ALGORITHM_SHA256);
            messageDigest.update(string.getBytes());
            return new BigInteger(1, messageDigest.digest()).toString(Character.MAX_RADIX);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Create memory image cache with specified maximum size
     *
     * @param maxSize Maximum size in bytes
     * @return Memory image cache
     */
    @NonNull
    public static MemoryImageCache newMemoryImageCache(int maxSize) {
        return new MemoryImageCacheImpl(maxSize);
    }

    /**
     * Create memory image cache with maximum size 25% of
     * total available memory fraction
     *
     * @return Memory image cache
     */
    @NonNull
    public static MemoryImageCache newMemoryImageCache() {
        return newMemoryImageCache(getMaxMemoryFraction(DEFAULT_MEMORY_FRACTION));
    }

    /**
     * Create storage image cache with specified parameters
     *
     * @param directory       Directory
     * @param maxSize         Maximum size
     * @param compressFormat  Compress format
     * @param compressQuality Compress quality
     * @return Storage image cache
     */
    @NonNull
    @SuppressWarnings("ResultOfMethodCallIgnored")
    public static StorageImageCache newStorageImageCache(@NonNull File directory, long maxSize,
            @NonNull Bitmap.CompressFormat compressFormat, int compressQuality) {
        if (!directory.exists()) {
            directory.mkdirs();
        }
        return new StorageImageCacheImpl(directory, maxSize, compressFormat, compressQuality);
    }

    /**
     * Create storage image cache in specified directory with maximum size 10%
     * of total storage size
     *
     * @param directory Cache directory
     * @return Storage image cache
     */
    @NonNull
    @SuppressWarnings("ResultOfMethodCallIgnored")
    public static StorageImageCache newStorageImageCache(@NonNull File directory) {
        if (!directory.exists()) {
            directory.mkdirs();
        }
        return newStorageImageCache(directory,
                getTotalStorageFraction(directory, DEFAULT_STORAGE_FRACTION),
                DEFAULT_COMPRESS_FORMAT, DEFAULT_COMPRESS_QUALITY);
    }

    /**
     * Create storage image cache in default application cache directory with maximum size 10%
     * of total storage size
     *
     * @param context Context
     * @return Storage image cache
     */
    @NonNull
    public static StorageImageCache newStorageImageCache(@NonNull Context context) {
        return newStorageImageCache(getDefaultStorageCacheDirectory(context));
    }

    /**
     * Default storage image cache directory
     *
     * @param context Context
     * @return Cache directory
     */
    @NonNull
    public static File getDefaultStorageCacheDirectory(@NonNull Context context) {
        File cacheDir = context.getExternalCacheDir();
        if (cacheDir == null) {
            cacheDir = context.getCacheDir();
        }
        return new File(cacheDir, DEFAULT_STORAGE_CACHE_DIRECTORY);
    }

    /**
     * Invert image colors
     *
     * @param image Source image
     * @return Inverted image
     */
    @NonNull
    public static Bitmap invertColors(@NonNull Bitmap image) {
        return applyColorFilter(image, new ColorMatrixColorFilter(
                new float[]{-1, 0, 0, 0, 255, 0, -1, 0, 0, 255, 0, 0, -1, 0, 255, 0, 0, 0, 1, 0}));
    }

    /**
     * Convert image colors to gray-scale
     *
     * @param image Source image
     * @return Converted image
     */
    @NonNull
    public static Bitmap convertToGrayScale(@NonNull Bitmap image) {
        ColorMatrix colorMatrix = new ColorMatrix();
        colorMatrix.setSaturation(0);
        return applyColorFilter(image, new ColorMatrixColorFilter(colorMatrix));
    }

    /**
     * Apply color filter to the specified image
     *
     * @param image       Source image
     * @param colorFilter Color filter
     * @return Filtered image
     */
    @NonNull
    public static Bitmap applyColorFilter(@NonNull Bitmap image, @NonNull ColorFilter colorFilter) {
        Paint paint = new Paint(Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG);
        paint.setColorFilter(InternalUtils.requireNonNull(colorFilter));
        Bitmap bitmap =
                Bitmap.createBitmap(image.getWidth(), image.getHeight(), Bitmap.Config.ARGB_8888);
        bitmap.setDensity(image.getDensity());
        new Canvas(bitmap).drawBitmap(image, 0, 0, paint);
        return bitmap;
    }

    /**
     * Crop center of image in proportions of {@code resultWidth} and {@code resultHeight}
     * and, if needed, resize it to {@code resultWidth} x {@code resultHeight} size.
     * If specified {@code resultWidth} and {@code resultHeight} are the same as the current
     * width and height of the source image, the source image will be returned.
     *
     * @param image        Source image
     * @param resultWidth  Result width
     * @param resultHeight Result height
     * @return Cropped (and/or resized) image or source image
     */
    @NonNull
    public static Bitmap cropCenter(@NonNull Bitmap image, int resultWidth, int resultHeight) {
        int sourceWidth = image.getWidth();
        int sourceHeight = image.getHeight();
        if (sourceWidth == resultWidth && sourceHeight == resultHeight) {
            return image;
        }
        int sourceDivisor = greatestCommonDivisor(sourceWidth, sourceHeight);
        int sourceRatioWidth = sourceWidth / sourceDivisor;
        int sourceRatioHeight = sourceHeight / sourceDivisor;
        int resultDivisor = greatestCommonDivisor(resultWidth, resultHeight);
        int resultRatioWidth = resultWidth / resultDivisor;
        int resultRatioHeight = resultHeight / resultDivisor;
        if (sourceRatioWidth == resultRatioWidth && sourceRatioHeight == resultRatioHeight) {
            return Bitmap.createScaledBitmap(image, resultWidth, resultHeight, true);
        }
        Bitmap cropped;
        int cropWidth = resultRatioWidth * sourceHeight / resultRatioHeight;
        if (cropWidth > sourceWidth) {
            int cropHeight = resultRatioHeight * sourceWidth / resultRatioWidth;
            cropped = Bitmap.createBitmap(image, 0, (sourceHeight - cropHeight) / 2, sourceWidth,
                    cropHeight);
            if (cropHeight == resultHeight && sourceWidth == resultWidth) {
                return cropped;
            }
        } else {
            cropped = Bitmap.createBitmap(image, (sourceWidth - cropWidth) / 2, 0, cropWidth,
                    sourceHeight);
            if (cropWidth == resultWidth && sourceHeight == resultHeight) {
                return cropped;
            }
        }
        Bitmap scaled = Bitmap.createScaledBitmap(cropped, resultWidth, resultHeight, true);
        if (cropped != image && cropped != scaled) {
            cropped.recycle();
        }
        return scaled;
    }

    /**
     * Fit image to specified frame ({@code resultWidth} x {@code resultHeight},
     * image will be scaled if needed.
     * If specified {@code resultWidth} and {@code resultHeight} are the same as the current
     * width and height of the source image, the source image will be returned.
     *
     * @param image        Source image
     * @param resultWidth  Result width
     * @param resultHeight Result height
     * @return Frame image with source image drawn in center of it or original image
     * or scaled image
     */
    @NonNull
    public static Bitmap fitCenter(@NonNull Bitmap image, int resultWidth, int resultHeight) {
        int sourceWidth = image.getWidth();
        int sourceHeight = image.getHeight();
        if (sourceWidth == resultWidth && sourceHeight == resultHeight) {
            return image;
        }
        int sourceDivisor = greatestCommonDivisor(sourceWidth, sourceHeight);
        int sourceRatioWidth = sourceWidth / sourceDivisor;
        int sourceRatioHeight = sourceHeight / sourceDivisor;
        int resultDivisor = greatestCommonDivisor(resultWidth, resultHeight);
        int resultRatioWidth = resultWidth / resultDivisor;
        int resultRatioHeight = resultHeight / resultDivisor;
        if (sourceRatioWidth == resultRatioWidth && sourceRatioHeight == resultRatioHeight) {
            return Bitmap.createScaledBitmap(image, resultWidth, resultHeight, true);
        }
        Bitmap result = Bitmap.createBitmap(resultWidth, resultHeight, Bitmap.Config.ARGB_8888);
        result.setDensity(image.getDensity());
        Canvas canvas = new Canvas(result);
        int fitWidth = sourceRatioWidth * resultHeight / sourceRatioHeight;
        if (fitWidth > resultWidth) {
            int fitHeight = sourceRatioHeight * resultWidth / sourceRatioWidth;
            int top = (resultHeight - fitHeight) / 2;
            canvas.drawBitmap(image, null, new Rect(0, top, resultWidth, top + fitHeight),
                    new Paint(Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG));
        } else {
            int left = (resultWidth - fitWidth) / 2;
            canvas.drawBitmap(image, null, new Rect(left, 0, left + fitWidth, resultHeight),
                    new Paint(Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG));
        }
        return result;
    }

    /**
     * Scale image to fit specified frame ({@code resultWidth} x {@code resultHeight}).
     * If specified {@code resultWidth} and {@code resultHeight} are the same as or smaller than
     * the current width and height of the source image, the source image will be returned.
     *
     * @param image        Source image
     * @param resultWidth  Result width
     * @param resultHeight Result height
     * @return Scaled image or original image
     */
    @NonNull
    public static Bitmap scaleToFit(@NonNull Bitmap image, int resultWidth, int resultHeight) {
        return scaleToFit(image, resultWidth, resultHeight, false);
    }

    /**
     * Scale image to fit specified frame ({@code resultWidth} x {@code resultHeight}).
     * If specified {@code resultWidth} and {@code resultHeight} are the same as the current
     * width and height of the source image, the source image will be returned.
     *
     * @param image        Source image
     * @param resultWidth  Result width
     * @param resultHeight Result height
     * @param upscale      Upscale image if it is smaller than the frame
     * @return Scaled image or original image
     */
    @NonNull
    public static Bitmap scaleToFit(@NonNull Bitmap image, int resultWidth, int resultHeight,
            boolean upscale) {
        int sourceWidth = image.getWidth();
        int sourceHeight = image.getHeight();
        if (sourceWidth == resultWidth && sourceHeight == resultHeight) {
            return image;
        }
        if (!upscale && sourceWidth < resultWidth && sourceHeight < resultHeight) {
            return image;
        }
        int sourceDivisor = greatestCommonDivisor(sourceWidth, sourceHeight);
        int sourceRatioWidth = sourceWidth / sourceDivisor;
        int sourceRatioHeight = sourceHeight / sourceDivisor;
        int resultDivisor = greatestCommonDivisor(resultWidth, resultHeight);
        int resultRatioWidth = resultWidth / resultDivisor;
        int resultRatioHeight = resultHeight / resultDivisor;
        if (sourceRatioWidth == resultRatioWidth && sourceRatioHeight == resultRatioHeight) {
            return Bitmap.createScaledBitmap(image, resultWidth, resultHeight, true);
        }
        int fitWidth = sourceRatioWidth * resultHeight / sourceRatioHeight;
        if (fitWidth > resultWidth) {
            if (sourceWidth == resultWidth) {
                return image;
            } else {
                int fitHeight = sourceRatioHeight * resultWidth / sourceRatioWidth;
                return Bitmap.createScaledBitmap(image, resultWidth, fitHeight, true);
            }
        } else {
            if (sourceHeight == resultHeight) {
                return image;
            } else {
                return Bitmap.createScaledBitmap(image, fitWidth, resultHeight, true);
            }
        }
    }

    /**
     * Round image corners with specified corner radius
     *
     * @param image        Source image
     * @param cornerRadius Corner radius
     * @return Image with rounded corners
     */
    @NonNull
    public static Bitmap roundCorners(@NonNull Bitmap image, float cornerRadius) {
        Paint paint = new Paint(Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG);
        paint.setColor(0xff424242);
        int width = image.getWidth();
        int height = image.getHeight();
        Rect rect = new Rect(0, 0, width, height);
        RectF rectF = new RectF(rect);
        Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        bitmap.setDensity(image.getDensity());
        Canvas canvas = new Canvas(bitmap);
        canvas.drawARGB(0, 0, 0, 0);
        canvas.drawRoundRect(rectF, cornerRadius, cornerRadius, paint);
        paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_IN));
        canvas.drawBitmap(image, rect, rect, paint);
        return bitmap;
    }

    /**
     * Mirror image horizontally
     *
     * @param image Source image
     * @return Mirrored image
     */
    @NonNull
    public static Bitmap mirrorHorizontally(@NonNull Bitmap image) {
        Matrix matrix = new Matrix();
        matrix.setScale(-1, 1);
        return Bitmap.createBitmap(image, 0, 0, image.getWidth(), image.getHeight(), matrix, true);
    }

    /**
     * Mirror image vertically
     *
     * @param image Source image
     * @return Mirrored image
     */
    @NonNull
    public static Bitmap mirrorVertically(@NonNull Bitmap image) {
        Matrix matrix = new Matrix();
        matrix.setScale(1, -1);
        return Bitmap.createBitmap(image, 0, 0, image.getWidth(), image.getHeight(), matrix, true);
    }

    /**
     * Rotate image by specified amount of degrees
     *
     * @param image         Source image
     * @param rotationAngle Amount of degrees
     * @return Rotated image
     */
    @NonNull
    public static Bitmap rotate(@NonNull Bitmap image, float rotationAngle) {
        Matrix matrix = new Matrix();
        matrix.setRotate(rotationAngle);
        return Bitmap.createBitmap(image, 0, 0, image.getWidth(), image.getHeight(), matrix, true);
    }

    private static int greatestCommonDivisor(int a, int b) {
        while (a > 0 && b > 0) {
            if (a > b) {
                a %= b;
            } else {
                b %= a;
            }
        }
        return a + b;
    }
}
