/*
 * MIT License
 *
 * Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;
import android.widget.ImageView;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * {@link ImageLoader} is a universal tool for loading bitmaps efficiently in Android, which
 * provides automatic memory and storage caching. {@link ImageLoader} is usable without caches,
 * with one of them, or both (without caches, caching is not available). Also, {@link ImageLoader}
 * is usable without {@link BitmapLoader} (loading new bitmaps is not available in this case).
 */
public class ImageLoader<T> {
    private final Lock mPauseLoadingLock = new ReentrantLock();
    private final Condition mPauseLoadingCondition = mPauseLoadingLock.newCondition();
    private final Context mContext;
    private volatile BitmapLoader<T> mBitmapLoader;
    private volatile MemoryImageCache mMemoryImageCache;
    private volatile StorageImageCache mStorageImageCache;
    private volatile PlaceholderProvider<T> mPlaceholderProvider;
    private volatile boolean mImageFadeIn = true;
    private volatile boolean mExitTasksEarly;
    private volatile boolean mLoadingPaused;
    private volatile int mImageFadeInTime = 250;

    /**
     * ImageLoader without any cache or bitmap loader
     * <br>
     * Use <b>application context</b> to avoid memory leaks.
     *
     * @param context Context
     * @see #setBitmapLoader(BitmapLoader)
     * @see #setMemoryImageCache(MemoryImageCache)
     * @see #setStorageImageCache(StorageImageCache)
     * @see #setPlaceholderProvider(PlaceholderProvider)
     * @see Context#getApplicationContext()
     */
    public ImageLoader(@NonNull Context context) {
        mContext = InternalUtils.requireNonNull(context);
    }

    /**
     * Load image to imageView from imageSource
     *
     * @param source    Image source
     * @param view      Image view
     * @param callbacks Optional callbacks
     * @see #makeCallbacks()
     */
    @MainThread
    public void loadImage(@NonNull ImageSource<T> source, @NonNull View view,
            @Nullable Callbacks<T> callbacks) {
        MemoryImageCache memoryImageCache = getMemoryImageCache();
        BitmapDrawable drawable = null;
        if (memoryImageCache != null) {
            drawable = memoryImageCache.get(source.getKey());
        }
        if (drawable != null) {
            setDrawable(view, drawable);
            Callbacks.notifyDisplayed(callbacks, source, drawable, view);
        } else if (cancelPotentialWork(source, view)) {
            LoadImageAction<T> loadAction =
                    new LoadImageAction<>(source, view, callbacks, this, mPauseLoadingLock,
                            mPauseLoadingCondition);
            PlaceholderDrawable placeholderDrawable =
                    new PlaceholderDrawable(getPlaceholderDrawable(source), loadAction);
            setDrawable(view, placeholderDrawable);
            loadAction.execute();
        }
    }

    /**
     * Load image to imageView from imageSource
     *
     * @param source Image source
     * @param view   Image view
     */
    @MainThread
    public void loadImage(@NonNull ImageSource<T> source, @NonNull View view) {
        loadImage(source, view, null);
    }

    /**
     * Delete cached image for specified {@link ImageSource}
     */
    public void invalidate(@NonNull ImageSource<T> source) {
        String key = source.getKey();
        MemoryImageCache memoryImageCache = getMemoryImageCache();
        if (memoryImageCache != null) {
            memoryImageCache.remove(key);
        }
        StorageImageCache storageImageCache = getStorageImageCache();
        if (storageImageCache != null) {
            storageImageCache.remove(key);
        }
    }

    /**
     * Create new {@link Callbacks} object
     *
     * @see Callbacks#load(LoadCallback)
     * @see Callbacks#error(ErrorCallback)
     * @see Callbacks#display(DisplayCallback)
     */
    @NonNull
    public Callbacks<T> makeCallbacks() {
        return new Callbacks<>();
    }

    /**
     * Current {@link BitmapLoader} implementation
     * <br>
     * {@link BitmapLoader} is used for loading new bitmaps
     * if there are no cached images with the same key
     */
    @Nullable
    public BitmapLoader<T> getBitmapLoader() {
        return mBitmapLoader;
    }

    /**
     * Current {@link BitmapLoader} implementation
     * <br>
     * {@link BitmapLoader} is used for loading new bitmaps
     * if there are no cached images with the same key
     */
    @NonNull
    public ImageLoader<T> setBitmapLoader(@Nullable BitmapLoader<T> loader) {
        mBitmapLoader = loader;
        return this;
    }

    /**
     * Current {@link MemoryImageCache} implementation
     * <br>
     * {@link MemoryImageCache} is used for caching images in memory
     */
    @Nullable
    public MemoryImageCache getMemoryImageCache() {
        return mMemoryImageCache;
    }

    /**
     * Current {@link MemoryImageCache} implementation
     * <br>
     * {@link MemoryImageCache} is used for caching images in memory
     */
    @NonNull
    public ImageLoader<T> setMemoryImageCache(@Nullable MemoryImageCache cache) {
        mMemoryImageCache = cache;
        return this;
    }

    /**
     * Current {@link StorageImageCache} implementation
     * <br>
     * {@link StorageImageCache} is used for caching images in storage
     */
    @Nullable
    public StorageImageCache getStorageImageCache() {
        return mStorageImageCache;
    }

    /**
     * Current {@link StorageImageCache} implementation
     * <br>
     * {@link StorageImageCache} is used for caching images in storage
     */
    @NonNull
    public ImageLoader<T> setStorageImageCache(@Nullable StorageImageCache cache) {
        mStorageImageCache = cache;
        return this;
    }

    /**
     * Placeholder provider, provides placeholders for images that are currently loading
     *
     * @see PlaceholderProvider
     * @see #setPlaceholderImage(Bitmap)
     * @see #setPlaceholderImage(int)
     */
    @NonNull
    public ImageLoader<T> setPlaceholderProvider(@Nullable PlaceholderProvider<T> provider) {
        mPlaceholderProvider = provider;
        return this;
    }

    /**
     * Placeholder provider, provides placeholders for images that are currently loading
     *
     * @see PlaceholderProvider
     */
    @Nullable
    public PlaceholderProvider<T> getPlaceholderProvider() {
        return mPlaceholderProvider;
    }

    /**
     * Placeholder image
     * <br>
     * Displayed while image is loading
     *
     * @param image Image bitmap
     */
    @NonNull
    public ImageLoader<T> setPlaceholderImage(@NonNull Bitmap image) {
        mPlaceholderProvider = new PlaceholderProviderImpl<>(getContext().getResources(), image);
        return this;
    }

    /**
     * Convenience method to set placeholder image from resource with image data.
     * <br>
     * Displayed while image is loading
     *
     * @param resourceId Image resource identifier
     */
    @NonNull
    public ImageLoader<T> setPlaceholderImage(int resourceId) {
        Resources resources = getContext().getResources();
        mPlaceholderProvider = new PlaceholderProviderImpl<>(resources,
                BitmapFactory.decodeResource(resources, resourceId));
        return this;
    }

    /**
     * Whether to use fade effect to display images
     *
     * @see #getImageFadeInTime()
     * @see #setImageFadeInTime(int)
     */
    public boolean isImageFadeIn() {
        return mImageFadeIn;
    }

    /**
     * Whether to use fade effect to display images
     *
     * @see #getImageFadeInTime()
     * @see #setImageFadeInTime(int)
     */
    @NonNull
    public ImageLoader<T> setImageFadeIn(boolean fadeIn) {
        mImageFadeIn = fadeIn;
        return this;
    }

    /**
     * Fade effect duration if that effect is enabled
     *
     * @see #isImageFadeIn()
     * @see #setImageFadeIn(boolean)
     */
    public int getImageFadeInTime() {
        return mImageFadeInTime;
    }

    /**
     * Fade effect duration if that effect is enabled
     *
     * @see #isImageFadeIn()
     * @see #setImageFadeIn(boolean)
     */
    @NonNull
    public ImageLoader<T> setImageFadeInTime(int time) {
        mImageFadeInTime = time;
        return this;
    }

    /**
     * Check if image loading is paused
     *
     * @see #setPauseLoading(boolean)
     */
    public boolean isLoadingPaused() {
        return mLoadingPaused;
    }

    /**
     * Whether to pause image loading. If this method is invoked with {@code true} parameter,
     * all loading actions will be paused until it will be invoked with {@code false}.
     */
    public void setPauseLoading(boolean pause) {
        mPauseLoadingLock.lock();
        try {
            mLoadingPaused = pause;
            if (!pause) {
                mPauseLoadingCondition.signalAll();
            }
        } finally {
            mPauseLoadingLock.unlock();
        }
    }

    /**
     * Whether to exit all image loading tasks before start of image loading
     */
    public boolean isExitTasksEarly() {
        return mExitTasksEarly;
    }

    /**
     * Whether to exit all image loading tasks before start of image loading
     */
    public void setExitTasksEarly(boolean exit) {
        mExitTasksEarly = exit;
        if (exit) {
            mPauseLoadingLock.lock();
            try {
                mPauseLoadingCondition.signalAll();
            } finally {
                mPauseLoadingLock.unlock();
            }
        }
    }

    /**
     * Clear all caches
     *
     * @see #getMemoryImageCache()
     * @see #setMemoryImageCache(MemoryImageCache)
     * @see #getStorageImageCache()
     * @see #setStorageImageCache(StorageImageCache)
     */
    public void clearCache() {
        MemoryImageCache memoryImageCache = getMemoryImageCache();
        if (memoryImageCache != null) {
            memoryImageCache.clear();
        }
        StorageImageCache storageImageCache = getStorageImageCache();
        if (storageImageCache != null) {
            storageImageCache.clear();
        }
    }

    /**
     * Context of this {@link ImageLoader} instance
     */
    @NonNull
    protected Context getContext() {
        return mContext;
    }

    /**
     * Set {@code drawable) to {@code view}
     */
    protected void setDrawable(@NonNull View view, @Nullable Drawable drawable) {
        if (view instanceof ImageView) {
            ((ImageView) view).setImageDrawable(drawable);
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                view.setBackground(drawable);
            } else {
                //noinspection deprecation
                view.setBackgroundDrawable(drawable);
            }
        }
    }

    /**
     * Get {@code drawable) from {@code view}
     */
    @Nullable
    protected Drawable getDrawable(@NonNull View view) {
        if (view instanceof ImageView) {
            return ((ImageView) view).getDrawable();
        } else {
            return view.getBackground();
        }
    }

    /**
     * Cancel loading process, associated with the specified {@code view}
     */
    @MainThread
    protected void cancelWork(@Nullable View view) {
        LoadImageAction<?> loadImageAction = getLoadImageAction(view);
        if (loadImageAction != null) {
            loadImageAction.cancel();
        }
    }

    /**
     * Cancel loading process, associated with the specified {@code view},
     * if image sources are different
     */
    @MainThread
    protected boolean cancelPotentialWork(@NonNull ImageSource<?> source, @Nullable View view) {
        LoadImageAction<?> loadImageAction = getLoadImageAction(view);
        if (loadImageAction != null) {
            if (!InternalUtils.equals(loadImageAction.getImageSource().getKey(), source.getKey())) {
                loadImageAction.cancel();
            } else {
                return false;
            }
        }
        return true;
    }

    /**
     * Get {@link LoadImageAction} instance, associated with {@code view}
     */
    @Nullable
    @MainThread
    protected LoadImageAction<?> getLoadImageAction(@Nullable View view) {
        if (view != null) {
            Drawable drawable = getDrawable(view);
            if (drawable instanceof PlaceholderDrawable) {
                return ((PlaceholderDrawable) drawable).getLoadImageAction();
            }
        }
        return null;
    }

    /**
     * Placeholder drawable for concrete image source
     *
     * @see PlaceholderProvider
     */
    @NonNull
    protected Drawable getPlaceholderDrawable(@NonNull ImageSource<T> source) {
        PlaceholderProvider<T> provider = getPlaceholderProvider();
        if (provider == null) {
            return new ColorDrawable(Color.TRANSPARENT);
        } else {
            return provider.get(getContext(), source.getData());
        }
    }
}
