/*
 * MIT License
 *
 * Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.content.Context;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.StatFs;
import android.support.annotation.AnyThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

final class InternalUtils {
    private static final String URI_SCHEME_HTTP = "http";
    private static final String URI_SCHEME_HTTPS = "https";
    private static final String URI_SCHEME_FTP = "ftp";
    private static final Lock MAIN_THREAD_HANDLER_LOCK = new ReentrantLock();
    private static final Lock LOADER_EXECUTOR_LOCK = new ReentrantLock();
    private static final Lock CACHE_EXECUTOR_LOCK = new ReentrantLock();
    private static volatile Handler sMainThreadHandler;
    private static volatile ThreadPoolExecutor sImageLoaderExecutor;
    private static volatile ThreadPoolExecutor sStorageCacheExecutor;

    private InternalUtils() {
    }

    @SuppressWarnings("deprecation")
    public static long getAvailableBytes(@NonNull File path) {
        StatFs statFs = new StatFs(path.getAbsolutePath());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            return statFs.getAvailableBytes();
        } else {
            return statFs.getAvailableBlocks() * statFs.getBlockSize();
        }
    }

    @SuppressWarnings("deprecation")
    public static long getTotalBytes(@NonNull File path) {
        StatFs statFs = new StatFs(path.getAbsolutePath());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            return statFs.getTotalBytes();
        } else {
            return statFs.getBlockCount() * statFs.getBlockSize();
        }
    }

    public static boolean equals(@Nullable Object a, @Nullable Object b) {
        return a == b || a != null && a.equals(b);
    }

    @NonNull
    public static <T> T requireNonNull(@Nullable T value) {
        if (value == null) {
            throw new NullPointerException();
        }
        return value;
    }

    @Nullable
    public static InputStream getDataStreamFromUri(@NonNull Context context,
            @NonNull Uri uri) throws IOException {
        String scheme = uri.getScheme();
        if (URI_SCHEME_HTTP.equalsIgnoreCase(scheme) || URI_SCHEME_HTTPS.equalsIgnoreCase(scheme) ||
                URI_SCHEME_FTP.equalsIgnoreCase(scheme)) {
            return new URL(uri.toString()).openConnection().getInputStream();
        } else {
            return context.getContentResolver().openInputStream(uri);
        }
    }

    public static void close(@Nullable Closeable closeable) {
        if (closeable == null) {
            return;
        }
        try {
            closeable.close();
        } catch (IOException ignored) {
        }
    }

    @AnyThread
    public static void runOnMainThread(@NonNull Runnable action) {
        getMainThreadHandler().post(action);
    }

    @NonNull
    public static ThreadPoolExecutor getImageLoaderExecutor() {
        ThreadPoolExecutor executor = sImageLoaderExecutor;
        if (executor == null) {
            LOADER_EXECUTOR_LOCK.lock();
            try {
                executor = sImageLoaderExecutor;
                if (executor == null) {
                    executor = new ImageLoaderExecutor(
                            Math.round(Runtime.getRuntime().availableProcessors() * 1.5f),
                            Thread.MIN_PRIORITY);
                    sImageLoaderExecutor = executor;
                }
            } finally {
                LOADER_EXECUTOR_LOCK.unlock();
            }
        }
        return executor;
    }

    @NonNull
    public static ThreadPoolExecutor getStorageCacheExecutor() {
        ThreadPoolExecutor executor = sStorageCacheExecutor;
        if (executor == null) {
            CACHE_EXECUTOR_LOCK.lock();
            try {
                executor = sStorageCacheExecutor;
                if (executor == null) {
                    executor = new ImageLoaderExecutor(1, Thread.MIN_PRIORITY);
                    sStorageCacheExecutor = executor;
                }
            } finally {
                CACHE_EXECUTOR_LOCK.unlock();
            }
        }
        return executor;
    }

    @NonNull
    private static Handler getMainThreadHandler() {
        Handler handler = sMainThreadHandler;
        if (handler == null) {
            MAIN_THREAD_HANDLER_LOCK.lock();
            try {
                handler = sMainThreadHandler;
                if (handler == null) {
                    handler = new Handler(Looper.getMainLooper());
                    sMainThreadHandler = handler;
                }
            } finally {
                MAIN_THREAD_HANDLER_LOCK.unlock();
            }
        }
        return handler;
    }
}
