package com.boanda.android.sync;

import android.content.Context;
import android.os.Environment;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.widget.Toast;

import com.szboanda.android.platform.db.DbHelper;
import com.szboanda.android.platform.db.SQLiteDao;
import com.szboanda.android.platform.http.HttpTask;
import com.szboanda.android.platform.http.ParamsWrapper;
import com.szboanda.android.platform.http.ResponseProcessor;
import com.szboanda.android.platform.util.BeanUtil;
import com.szboanda.android.platform.util.FileUtils;
import com.szboanda.android.platform.util.StringUtils;

import org.json.JSONArray;
import org.json.JSONObject;
import org.xutils.db.Selector;
import org.xutils.db.sqlite.WhereBuilder;
import org.xutils.db.table.TableEntity;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.reactivex.Flowable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;
import io.reactivex.schedulers.Schedulers;

public class SyncSubmitManager {
	
	public final static String RELATIVE_DIR = "/tempSubmitData";
	
	/**数据库表数据约定后缀名*/
	public final static String TABLE_DATA_FILE_SUFFIX = ".tabd";
	
	/**同步内容类型-文件*/
	private final static String CONTENT_TYPE_FILE = "file";

    private final static String IDENTIFY_SERVICE_PATH = "@path:";

	private static Object mLock = new Object();
	
	private static SyncSubmitManager INSTANCE; 
	
	private Context context;
	
	private Handler handler;
	
	private SyncSubmitManager(){
		try{
			DbHelper.createTableIfNotExist(SyncSubmitItem.class);
		}catch (Exception e) {
			e.printStackTrace();
		}
	}

	public static SyncSubmitManager getInstance(){
		if(INSTANCE == null){
			synchronized (mLock) {
				if(INSTANCE == null){
					INSTANCE = new SyncSubmitManager();
				}
			}
		}
		return INSTANCE;
	}
	
	public <E> void appendSubmitItem(List<E> entitys){
		if(entitys != null){
			for(E e:entitys){
				appendSubmitItem(e);
			}
		}
	}
	
	public <E> boolean  appendSubmitItem(E entity){
		SyncSubmitItem item = new SyncSubmitItem();
        try{
            Object idValue = DbHelper.getDao().getTable(entity.getClass()).getId().getColumnValue(entity);
            item.setTable(entity.getClass().getName());
            //这里使用加一个空串转换id的值为字符串是因为使用强转方式时，如果
            //id的值是数字，会出现转换异常
            item.setPkValue(idValue + "");
            SyncConfig config = DataSyncManager.getInstance().getParamConfig();
            String userId = config.getUserId();
            item.setCjr(userId);
            item.setXgr(userId);
            item.setCjsj(new Date());
            item.setXgsj(new Date());
            item.setOrgid(config.getOrgid());
            item.setSubmited(false);
        }catch(Exception e){
            e.printStackTrace();
        }
		return addSyncItemToQueue(item);
	}

    /**
     * 批量添加文件到同步队列
     * @param files
     */
	public void  appendSubmitFile(List<File> files){
		if(files != null){
			for(File f:files){
				appendSubmitFile(f);
			}
		}
	}

    /**
     * 添加文件到同步队列
     * @param file
     * @return
     */
	public boolean  appendSubmitFile(File file){
		return appendSubmitFile(file, null);
	}

    /**
     * 添加文件到同步队列
     * @param filePath
     * @return
     */
	public boolean  appendSubmitFile(String filePath){
		return appendSubmitFile(new File(filePath), null);
	}

    /**
     * 添加文件到同步队列
     * @param filePath
     * @param servicePath
     * @return
     */
	public boolean  appendSubmitFile(String filePath, String servicePath){
		return appendSubmitFile(new File(filePath), servicePath);
	}

    /**
     * 添加文件到同步队列
     * @param file 需要同步的文件
     * @param servicePath 文件存储到服务器上相对于服务应用的根目录的路径，假设服务应用的绝对路径是D://Ydzf/webapp/,
     *                    servicePath的值为/test/，第一个参数file的文件名称为test.png，则在服务器上存储的路径为D://Ydzf/webapp/test/test.png。
     *                    如果servicePath的值为空则存储路径为D://Ydzf/webapp/WEB-INF/ + 参数file的路径replace掉{@link DataSyncManager#getCachePath()}
     *                    组成的路径
     * @return
     */
	public boolean  appendSubmitFile(File file, String servicePath){
		SyncSubmitItem item = new SyncSubmitItem();
		item.setTable(CONTENT_TYPE_FILE);
        if(TextUtils.isEmpty(servicePath)) {
            item.setPkValue(file.getAbsolutePath());
        }else{
            item.setPkValue(file.getAbsolutePath() + IDENTIFY_SERVICE_PATH + servicePath);
        }
        SyncConfig config = DataSyncManager.getInstance().getParamConfig();
        String userId = config.getUserId();
        item.setCjr(userId);
        item.setXgr(userId);
        item.setCjsj(new Date());
        item.setXgsj(new Date());
        item.setOrgid(config.getOrgid());
		item.setSubmited(false);
		return addSyncItemToQueue(item);
	}

	private boolean addSyncItemToQueue(SyncSubmitItem item){
		boolean isSuccess = false;
		try{
			SyncSubmitItem findItem = isItemExist(item);
			if(findItem == null){
				DbHelper.getDao().save(item);
			}else{
				item.setId(findItem.getId());
				DbHelper.getDao().update(item, "tab_name", "pk_value", "submited", "xgr", "xgsj");
			}
			isSuccess = true;
		}catch (Exception e) {
			e.printStackTrace();
		}
		return isSuccess;
	}
	
	public void submitSyncData(Class<?> tableCls){
        if(tableCls == null) {
            submitSyncData(SyncAll.class, null);
        }else{
            submitSyncData(tableCls, null);
        }
	}

	private void submitSyncData(Class<?> tableCls, final String filePath){
        SyncConfig config = DataSyncManager.getInstance().getParamConfig();
		if(!TextUtils.isEmpty(config.getUserId())){
            Flowable.just(tableCls)
                    .subscribeOn(Schedulers.io())
                    .map(new Function<Class<?>, List<String>>() {
                        @Override
                        public List<String> apply(Class<?> cls) throws Exception {
                            return createTempSubmitFile(cls);
                        }
                    })
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Consumer<List<String>>() {
                        @Override
                        public void accept(List<String> paths) throws Exception {
                            submitSyncDataFile(paths, filePath);
                        }
                    });
		}
	}

    /**
     * 同步所有未提交的表数据及文件
     */
	public void submitSyncData(){
		submitSyncData(null);
	}

    /**
     * 同步指定文件
     * @param filePath 同步指定的文件
     */
    public void submitSyncFile(String filePath){
        submitSyncData(null, filePath);
    }
	
	private File getSubmitDataTempDir(){
		String dirPath = DataSyncManager.getInstance().getCachePath() + RELATIVE_DIR;
		return new File(dirPath);
	}
	
	/**
	 * 创建临时的提交数据文件.为防止数据量过大无法提交，使用先把数据转换成文件，
	 * 以文件的形式提交到服务器
	 */
	private List<String> createTempSubmitFile(Class<?> indentifyTableCls){
		File dir = getSubmitDataTempDir();
		if(dir.exists()){
			File[] childFiles = dir.listFiles();
			if(childFiles != null){
				for(File cf:childFiles){
					cf.delete();
				}
			}
		}else{
			dir.mkdirs();
		}
		String sql;
		if(indentifyTableCls != null && !indentifyTableCls.equals(SyncAll.class)){
			String tableName = indentifyTableCls.getName();
			sql = "SELECT TAB_NAME FROM T_SUBMIT_ITEMS "
					+"WHERE TAB_NAME = '" + tableName + "'"
					+ " AND SUBMITED <> 1 "
					+" AND TAB_NAME <> 'file' GROUP BY TAB_NAME";
		}else{
			sql = "SELECT TAB_NAME FROM T_SUBMIT_ITEMS "
					+"WHERE SUBMITED <> 1 "
					+" AND TAB_NAME <> 'file' GROUP BY TAB_NAME";
		}
        List<String> submitTableFiles = new ArrayList<>();
        JSONArray tables;
		try{
            SQLiteDao dao = DbHelper.getDao();
            tables = dao.getListValue(sql);
			if(tables != null){
				for(int i=0; i<tables.length(); i++){
					JSONObject item = tables.optJSONObject(i);
					String tableClsStr = item.optString("TAB_NAME");
					Class<?> tableCls = BeanUtil.loadClass(tableClsStr);
                    TableEntity table = dao.getTable(tableCls);
                    String tn = table.getName();
					String idCol = table.getId().getName();
					String dataSql = "SELECT MT.* FROM " + tn + " MT"
							+ " INNER JOIN T_SUBMIT_ITEMS ST ON MT." + idCol + " = ST.pk_value";
					JSONArray submitDataItems = dao.getListValue(dataSql);
					if(submitDataItems != null){
						String submitFileName = tn + TABLE_DATA_FILE_SUFFIX;
						boolean success = FileUtils.createFileByContent(dir.getAbsolutePath(), submitFileName, submitDataItems.toString());
                        if(success) {
                            submitTableFiles.add(dir.getAbsolutePath() + File.separator + submitFileName);
                        }
					}
				}
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		return submitTableFiles;
	}
	
	private void submitSyncDataFile(List<String> tableDataFiles, String filePath){
        boolean hasSyncData = false;
        ParamsWrapper params = new DataSyncParams();
        params.setMultipart(true);
        params.addQueryStringParameter("service", SyncServiceEnum.SUBMIT_TABLE_FILE.name());
        if(tableDataFiles != null){
            hasSyncData = tableDataFiles.size() > 0;
            for(String path : tableDataFiles){
                String fn = StringUtils.parseFileName(path);
                params.addBodyParameter(fn, new File(path));
            }
        }
        final List<String> syncFiles = findSyncFiles(filePath);
        if(syncFiles != null){
            hasSyncData = hasSyncData || syncFiles.size() > 0;
            String sdCardPath = Environment.getExternalStorageDirectory().getAbsolutePath();
            for(String path : syncFiles){
                File f = new File(path.split(IDENTIFY_SERVICE_PATH)[0]);
                if(f.exists()){
                    params.addBodyParameter(path.replace(sdCardPath, ""), f);
                }
            }
        }
        if(hasSyncData){
            HttpTask submitTask = context != null ? new HttpTask(context,
                    "正在同步数据") : new HttpTask();
            ResponseProcessor processor = new ResponseProcessor<JSONObject>() {
                @Override
                public void onSuccessTyped(JSONObject result) {
                    deleteTempSubmitFile();
                    if(result != null){
                        boolean isSuccess = result.optBoolean("isSuccess");
                        if(handler != null){
                            Message msg = new Message();
                            msg.what = 0;
                            msg.arg1 = isSuccess ? 0 : 1;
                            handler.sendMessage(msg);
                        }
                        if(isSuccess){
                            String submitTables = result.optString("tables");
                            try{
                                Map<String, String> tableClsMap = findTableClassMap();
                                if(!TextUtils.isEmpty(submitTables) && tableClsMap != null){
                                    String[] tables = submitTables.split(";");
                                    for(String t:tables){
                                        String clsName = tableClsMap.get(t);
                                        if(!TextUtils.isEmpty(clsName)){
                                            WhereBuilder wb = WhereBuilder.b("tab_name", "=", clsName);
                                            DbHelper.getDao().delete(SyncSubmitItem.class, wb);
                                        }
                                    }
                                }
                                String successedFiles = result.optString("files");
                                if(!TextUtils.isEmpty(successedFiles)){
                                    String[] paths = successedFiles.split(",");
                                    for(String sp: paths){
                                        for(String lp : syncFiles){
                                            if(lp.endsWith(sp) || lp.equals(sp)){
                                                WhereBuilder wb = WhereBuilder.b("pk_value", "=", lp);
                                                DbHelper.getDao().delete(SyncSubmitItem.class, wb);
                                                System.out.println("delete:" + lp);
                                            }
                                        }
                                    }
                                }
                            }catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }

                @Override
                public void onError(Throwable ex, boolean isOnCallback) {
                    if(context != null){
                        Toast.makeText(context, "提交失败", Toast.LENGTH_SHORT).show();
                    }
                    if(handler != null){
                        Message msg = new Message();
                        msg.what = 0;
                        msg.arg1 = 1;//1代表失败
                        handler.sendMessage(msg);
                    }
                }
            };
            submitTask.executePost(params, context != null, processor);
        }
	}
	
	private Map<String, String> findTableClassMap(){
        Map<String, String> tableClassMap = null;
        try{
            String tableName = DbHelper.getDao().getTable(SyncSubmitItem.class).getName();
            String sql = "SELECT TAB_NAME FROM "
                + tableName
                +" WHERE SUBMITED <> 1 "
                +" AND TAB_NAME <> 'file' GROUP BY TAB_NAME";
            SQLiteDao dao = DbHelper.getDao();
            JSONArray tables = dao.getListValue(sql);
            if(tables != null){
                tableClassMap = new HashMap<String, String>();
                for(int i=0; i<tables.length(); i++){
                    JSONObject tb = tables.optJSONObject(i);
                    String clsName = tb.optString("TAB_NAME");
                    Class<?> cls = BeanUtil.loadClass(clsName);
                    String tn = dao.getTable(cls).getName();
                    tableClassMap.put(tn, clsName);
                }
            }
        }catch(Exception e){
            e.printStackTrace();
        }
		return tableClassMap;
	}
	
	private List<String> findSyncFiles(String filePath){
		List<String> syncFiles = null;
        try{
            Selector s = DbHelper.getDao().selector(SyncSubmitItem.class);
            s.where("tab_name", "=", CONTENT_TYPE_FILE);
            s.and("submited", "=", false);
            if(!TextUtils.isEmpty(filePath)){
                s.and("pk_value", "like", filePath + "%");
            }
            List<SyncSubmitItem> items = s.findAll();
			if(items != null){
				syncFiles = new ArrayList<String>();
				for(SyncSubmitItem si:items){
					String path = (String)si.getPkValue();
					File f = new File(path.split(IDENTIFY_SERVICE_PATH)[0]);
					if(f.exists()){
						syncFiles.add(path);
					}
				}
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		return syncFiles;
	}
	
	/**
	 * 删除临时生成的提交数据文件
	 */
	private void deleteTempSubmitFile(){
		File dir = getSubmitDataTempDir();
		File[] childFiles = dir.listFiles();
		if(childFiles != null){
			for(File cf:childFiles){
				cf.delete();
			}
		}
	}
	
	/**
	 * 设置context来显示进度条
	 ***/
	public void setContext(Context context){
		this.context = context;
	}
	
	/**
	 * 设置context来显示进度条
	 ***/
	public void setHandler(Handler handler){
		this.handler = handler;
	}
	
	private SyncSubmitItem isItemExist(SyncSubmitItem item){
		SyncSubmitItem findItem = null;
		try{
			findItem = DbHelper.getDao().selector(SyncSubmitItem.class)
                .where("tab_name", "=", item.getTable())
                .and("pk_value", "=", item.getPkValue())
                .findFirst();
		}catch (Exception e) {
			e.printStackTrace();
		}
		return findItem;
	}
	
}
