package com.boanda.android.sync;

import android.app.Dialog;
import android.content.Context;
import android.text.TextUtils;

import com.szboanda.android.platform.db.DbHelper;
import com.szboanda.android.platform.db.SQLiteDao;
import com.szboanda.android.platform.http.HttpTask;
import com.szboanda.android.platform.http.ParamsWrapper;
import com.szboanda.android.platform.http.impl.IProgressChangeView;
import com.szboanda.android.platform.util.JsonUtils;
import com.szboanda.android.platform.util.NetworkUtils;
import com.szboanda.android.platform.util.StringUtils;

import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import io.reactivex.Flowable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;
import io.reactivex.schedulers.Schedulers;

/**
 * <p>Description: 数据库同步管理类</p>
 * @author    Siyi Lu    Create Date: 2015-8-18
 */
public class DataSyncManager {

	/**终端提交的同步表的键*/
	public final static String KEY_UNION_TABLES = "UNION_TABLES";
	/**表最后一次同步时间的键*/
	public final static String KEY_UNION_SYNC_TIMES = "UNION_SYNC_TIMES";
	/**同步数据返回格式*/
	public final static String KEY_PROTCOL_SUPPORT = "protcol_support";
	
	private static Object mLock = new Object();
	
	private volatile static DataSyncManager mInstance = null;

    private static JSONObject SYNC_FREEZE_INTERVAL = new JSONObject();

    private Context mContext;

    private SyncConfig mParamConfig;

    //防止被使用new方式实例化
	private DataSyncManager(){}

	public static DataSyncManager getInstance(){
		if(mInstance == null){
			synchronized(mLock){
				if(mInstance == null){
					mInstance = new DataSyncManager();
				}
			}
		}
		return mInstance;
	}

	public void init(Context context, SyncConfig config){
        mContext = context;
        mParamConfig = config;
    }

    public SyncConfig getParamConfig(){
        return mParamConfig;
    }
	/**
	 * 异步执行同步任务
	 * @param context
	 * @param tables
	 */
	public void syncTableAsync(Context context, final String... tables){
		syncTableAsync(context, null, tables);
	}
	
	/**
	 * 异步执行同步任务
	 * @param context
     * @param progress
     * @param tables
	 */
	public void syncTableAsync(Context context, IProgressChangeView progress, final String... tables){
		syncTableAsync(context, progress, null, tables);
	}
	
	/**
	 * 异步执行同步任务
	 * @param context
     * @param progress
     * @param callback
     * @param tables
	 */
	public void syncTableAsync(Context context, final IProgressChangeView progress
			,final Runnable callback, final String... tables){
        if (progress != null && progress instanceof Dialog){
            ((Dialog)progress).show();
        }
        Flowable.just(tables)
                .subscribeOn(Schedulers.io())
                .map(new Function<String[], Boolean>() {
                    @Override
                    public Boolean apply(String[] tables) throws Exception {
                        syncTable(tables);
                        return true;
                    }
                })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<Boolean>() {
                    @Override
                    public void accept(Boolean aBoolean) throws Exception {
                        if (progress != null && progress instanceof Dialog && ((Dialog)progress).isShowing()){
                            ((Dialog)progress).dismiss();
                        }
                        if(callback != null){
                            callback.run();
                        }
                    }
                });
	}
	
	/**
	 *	同步执行同步任务,调用者需要自己启动子线程
	 * @param tableName
	 */
	public synchronized void syncTable(String tableName){
		long start;
		long end1;
		long end2;
		if(NetworkUtils.isNetworkOpened(mContext)){
			ParamsWrapper params = new DataSyncParams();
			params.addQueryStringParameter("service", SyncServiceEnum.DATASYNC_REQUEST.toString());
			params.addQueryStringParameter("lastSyncTime", getLastSyncTime(tableName));
			params.addQueryStringParameter("table", tableName);
			HttpTask syncTask = new HttpTask(null);
			start = System.currentTimeMillis();
            DataSyncResponse response = syncTask.sendRequest(params, DataSyncResponse.class);
		}
	}
	
	public void syncTable(String... tables){
		String[] tableArray = (String[])tables;
		List<String> tableList = Arrays.asList(tableArray);
		syncTables(tableList);
	}
	
	public void syncTable(Class<?>... entitys){
		Class<?>[] entityArray = (Class<?>[])entitys;
		List<String> tableList = new ArrayList<String>();
        SQLiteDao dao = DbHelper.getDao();
		for(Class<?> cls:entityArray){
            try{
                String name = dao.getTable(cls).getName();
                tableList.add(name);
            }catch(Exception e){
                e.printStackTrace();
            }
		}
		syncTables(tableList);
	}
	
	public void syncTable(List<Class<?>> entitys){
		List<String> tableList = new ArrayList<String>();
        SQLiteDao dao = DbHelper.getDao();
		for(Class<?> cls : entitys){
            try{
                tableList.add(dao.getTable(cls).getName());
            }catch(Exception e){
                e.printStackTrace();
            }
		}
		syncTables(tableList);
	}
	
	public synchronized void syncTables(List<String> tables){
		long start;
		long end1;
		long end2;
		if(NetworkUtils.isNetworkOpened(mContext)){
			ParamsWrapper params = new DataSyncParams();
			String unionTables = StringUtils.joinStringItem(tables, ",");
			List<String> syncTimeList = new ArrayList<String>();
			for(String tableName:tables){
				syncTimeList.add(getLastSyncTime(tableName));
			}
			String unionSyncTimes = StringUtils.joinStringItem(syncTimeList, ",");
			params.addBodyParameter("service", SyncServiceEnum.SYNC_TABLES.toString());
			params.addBodyParameter(KEY_UNION_TABLES, unionTables);
			params.addBodyParameter(KEY_UNION_SYNC_TIMES, unionSyncTimes);
			//设置支持返回的数据格式：json OR xml
			params.addBodyParameter(KEY_PROTCOL_SUPPORT, "json");
			HttpTask syncTask = new HttpTask(null);
			syncTask.setTimeOut(30 * 1000);
            params.setReadTimeout(Integer.MAX_VALUE);
			start = System.currentTimeMillis();
			syncTask.sendRequest(params, DataSyncResponse.class);
		}
	}
	
	/**
	 * 获取表最后同步时间
	 * @return
	 */
	public String getLastSyncTime(String tableName){
		return getLastSyncTime(null, tableName);
	}
	
	/**
	 * 获取表最后同步时间
	 * @return 
	 */
	public String getLastSyncTime(String dbName, String tableName){
		String sql = "SELECT ifnull(MAX(XGSJ),'') as LASTSYNC FROM " + tableName;
        SQLiteDao dao = DbHelper.getDao(dbName);
        JSONObject last = dao.getSingleValue(sql);
		String lastSyncTime = (last == null ? "" : last.optString("LASTSYNC"));
		return TextUtils.isEmpty(lastSyncTime) ? "-1":lastSyncTime;
	}

	public void setTableSyncFreezeInterval(String table, int sec){
        JsonUtils.put(SYNC_FREEZE_INTERVAL, table, sec);
    }

    public String getCachePath(){
        File cacheFile = mContext.getExternalCacheDir();
        if(cacheFile == null){
            cacheFile = mContext.getCacheDir();
        }
        String cachePath = cacheFile.getAbsolutePath() + "/datasync";
        return cachePath;
    }
}
