package com.szboanda.android.platform.view;

import android.content.Context;
import android.graphics.Canvas;
import android.util.AttributeSet;
import android.view.View;
import android.widget.BaseAdapter;
import android.widget.ListAdapter;

/**
 * <p>Description: 顶部悬浮组标题列表</p>
 * @author    Siyi Lu    Create Date: 2015-7-24
 */
public class PinnedHeaderListView extends XListView {
	
	private static final int MAX_ALPHA = 255;

	private PinnedHeaderAdapter mAdapter;
	/** 悬浮的在头部View*/
	private View mHeaderView;
	
	private boolean mHeaderViewVisible;
	
	private int mHeaderViewWidth;
	
	private int mHeaderViewHeight;

	public PinnedHeaderListView(Context context) {
		super(context);
	}

	public PinnedHeaderListView(Context context, AttributeSet attrs) {
		super(context, attrs);
	}

	public PinnedHeaderListView(Context context, AttributeSet attrs,
			int defStyle) {
		super(context, attrs, defStyle);
	}

	public void setPinnedHeaderView(View view) {
		mHeaderView = view;
		if (mHeaderView != null) {
			setFadingEdgeLength(0);
		}
	}

	@Override
	public void setAdapter(ListAdapter adapter) {
		super.setAdapter(adapter);
		mAdapter = (PinnedHeaderAdapter) adapter;
	}

	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		super.onMeasure(widthMeasureSpec, heightMeasureSpec);
		if (mHeaderView != null) {
			measureChild(mHeaderView, widthMeasureSpec, heightMeasureSpec);
			mHeaderViewWidth = mHeaderView.getMeasuredWidth();
			mHeaderViewHeight = mHeaderView.getMeasuredHeight();
		}
	}

	@Override
	protected void onLayout(boolean changed, int left, int top, int right,
			int bottom) {
		super.onLayout(changed, left, top, right, bottom);
		if (mHeaderView != null) {
			mHeaderView.layout(0, 0, mHeaderViewWidth, mHeaderViewHeight);
			layoutHeaderView(getFirstVisiblePosition());
		}
	}

	@Override
	protected void dispatchDraw(Canvas canvas) {
		super.dispatchDraw(canvas);
		if (mHeaderViewVisible && mHeaderView != null) {
			drawChild(canvas, mHeaderView, getDrawingTime());
		}
	}
	
	/**
	 *根据状态值{@link PinnedHeaderAdapter}返回的悬浮头部状态值布局头部视图
	 * @param firstVisableItemPosition 列表第一条可见条目的索引
	 */
	public void layoutHeaderView(int firstVisableItemPosition) {
		if (mHeaderView == null) {
			return;
		}
		int state = mAdapter.getPinnedHeaderState(firstVisableItemPosition);
		switch (state) {
			case PinnedHeaderAdapter.PINNED_HEADER_GONE: {
				mHeaderViewVisible = false;
				break;
			}
	
			case PinnedHeaderAdapter.PINNED_HEADER_VISIBLE: {
				mAdapter.refreshHeaderView(mHeaderView, firstVisableItemPosition, MAX_ALPHA);
				if (mHeaderView.getTop() != 0) {  
	                mHeaderView.layout(0, 0, mHeaderViewWidth, mHeaderViewHeight);  
	            }  
	            mHeaderViewVisible = true;  
				break;
			}
	
			case PinnedHeaderAdapter.PINNED_HEADER_PUSHED_UP: {
				View firstView = getChildAt(0);
				int bottom = firstView.getBottom();
				int headerHeight = mHeaderView.getHeight();
				int y;
				int alpha;
				if (bottom < headerHeight) {
					y = (bottom - headerHeight);
					alpha = MAX_ALPHA * (headerHeight + y) / headerHeight; 
				} else {
					y = 0;
					alpha = MAX_ALPHA;
				}
				//在前一个悬浮顶部视图被下一个组标题顶出界面时，需要第一个显示的item不为组条目才才生效
				IPinnedHeaderListItem firstItem = (IPinnedHeaderListItem)((BaseAdapter)mAdapter).getItem(firstVisableItemPosition);
				if(!firstItem.isGroupItem()){
					mAdapter.refreshHeaderView(mHeaderView, firstVisableItemPosition, alpha);
				}
				if (mHeaderView.getTop() != y) {
					mHeaderView.layout(0, y, mHeaderViewWidth, mHeaderViewHeight + y);
				}
				mHeaderViewVisible = true;
				break;
			}
		}
	}

	
	public interface PinnedHeaderAdapter {
		
		/**不显示列表顶部悬浮视图*/
		public static final int PINNED_HEADER_GONE = 0;
		
		/**显示列表顶部悬浮视图*/
		public static final int PINNED_HEADER_VISIBLE = 1;
		
		/**下一个顶部悬浮视图推上一个*/
		public static final int PINNED_HEADER_PUSHED_UP = 2;

		/**获取列表顶部悬浮视图的更新状态*/
		int getPinnedHeaderState(int firstVisableItemPosition);

		/**更新列表顶部悬浮视图*/
		void refreshHeaderView(View header, int position, int alpha);
	}
}