package com.szboanda.android.platform.nfc;

import android.app.Activity;
import android.app.PendingIntent;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Bundle;

/**
 * NFC操作接口,提供涉及到NFC操作需要的基本方法与几个自定义方法接口.
 * 该接口有一个抽象实现类{@link NfcBaseOperation}实现了几个主要方法,
 * 具体协议的读写目前只实现了NFC论坛标准协议
 * NDEF(NFC Data Exchange Format),实现类为{@link NdefOperation}.
 *
 * 使用该接口只有在主线程中处于Active状态(正在与用户交互)的Activity在调用了{@link #enableForegroundDispatch()}
 * 方法后才能接收到NFC的标签分发系统在扫描到NFC标签后根据过滤的数据类型而分到的{@link android.nfc.Tag}
 *
 * 请在项目中的manifest中添加 uses-permission android:name="android.permission.NFC"权限
 * 在{@link Activity#onCreate(Bundle)}中调用{@link #prepareForegroundDispatch()}
 * 在{@link Activity#onResume()}中调用{@link #enableForegroundDispatch()}
 * 在{@link Activity#onPause()}中调用{@link #disableForegroundDispatch()}
 *
 * 具体使用Demo请参照{@link com.szboanda.android.platform.nfc.demo.NdefDemoActivity}
 * --修订时间:16/4/14
 *
 *
 * Created by zhanggenhua on 16/4/14.
 */
public interface INfcOperation {

    /**
     * 在Activity中接收NFC的标签分发系统分发{@link android.nfc.Tag}的准备方法
     */
    void prepareForegroundDispatch();

    /**
     * 准备过程中需要判断设备是否打开NFC,此方法需要在你的Activity的
     * {@link android.app.Activity#onActivityResult(int, int, Intent)}中调用
     * @param requestCode {@link android.app.Activity#onActivityResult(int, int, Intent)} 参数中的requestCode
     */
    void onPrepareForegroundDispatch(int requestCode);

    /**
     * 请参照{@link android.nfc.NfcAdapter#enableForegroundDispatch(Activity, PendingIntent, IntentFilter[], String[][])}
     * 参数在{@link NfcBaseOperation}中有处理,不用担心
     */
    void enableForegroundDispatch();

    /**
     * 请参照{@link android.nfc.NfcAdapter#disableForegroundDispatch(Activity)}
     * 参数在{@link NfcBaseOperation}中有处理,不用担心
     */
    void disableForegroundDispatch();

    /**
     * 在你的Activity的{@link Activity#onNewIntent(Intent)}中调用以接收{@link android.nfc.Tag}
     * @param intent
     */
    void onNewIntent(Intent intent);

    /**
     * @return 从标签中解析出的payload
     */
    String read();

    /**
     * @param content 被写入的内容
     * @return 如果成功返回true, 否则false
     */
    boolean write(String content);

    /**
     * 设置是否可以在NFC标签切近时自动读取内容自动读取, 读取回调为
     * {@link INfcListener#onRead(String)}, 可通过{@link #setNfcListener(INfcListener)}
     * 处理你得到回调
     * @param autoRead 是否自动读取
     */
    void enableAutoRead(boolean autoRead);

    /**
     * 设置自动读取回调,需要{@link #enableAutoRead(boolean)}为true时才会有回调产生
     * @param l
     */
    void setNfcListener(INfcListener l);

}
