package com.szboanda.android.platform.http;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.widget.RemoteViews;

import com.szboanda.android.platform.R;
import com.szboanda.android.platform.activity.PdfViewActivity;
import com.szboanda.android.platform.http.impl.IProgressChangeView;
import com.szboanda.android.platform.util.FileUtils;

import org.xutils.common.Callback;
import org.xutils.ex.HttpException;

import java.io.File;

/**
 * <p>Description: 文件下载响应处理器</p>
 * @author    Siyi Lu    Create Date: 2015-4-14
 */
public abstract class DownloadResponseProcessor 
	implements Callback.ProgressCallback<File> {
	
	private Context mContext;
	
	private boolean isDownloadBackgroud = false;
	
	/**下载文件本地存储路径(包含文件名)*/
	private String mTargetPath;
	
	/**下载文件文件名*/
	private String mDownloadFileName;
	
	/**进度更新视图*/
	private IProgressChangeView mProgressView;
	
	private boolean isAutoRename = false;

	public IProgressChangeView getProgressView() {
		return mProgressView;
	}

	public void setProgressView(IProgressChangeView progressView) {
		this.mProgressView = progressView;
	}
	
	public void setAutoRename(boolean autoRename){
		this.isAutoRename = autoRename; 
	}
	
	public boolean isAutoRename(){
		return isAutoRename;
	}

	@Override
	public void onSuccess(File downloadFile) {
		String suffix = FileUtils.parseFileSuffix(downloadFile.getAbsolutePath());
		if("dltemp".equals(suffix)){
			//因为下载文件下载过程中，存储的文件使用了临时下载文件后缀名"dltemp"
			//下载成功后需要还原文件名
            downloadFile.renameTo(new File(mTargetPath));
		}
		if(isAutoRename){
			mTargetPath = downloadFile.getAbsolutePath();
		}
		
		if(mProgressView != null){
			mProgressView.onFinish();
		}
		onDownloadSuccess(new File(mTargetPath));
	}
	
	@Override
	public void onError(Throwable ex, boolean isOnCallback) {
		if(mProgressView != null){
			mProgressView.onFinish();
		}
        IErroResolver resolver = ErrorResolverFactory.newResolver(mContext, ex);
        if(resolver != null){
            resolver.resolve();
        }
    }
	
	private NotificationManager mNoticeManager;
	private Notification mNotice;
	@Override
	public void onLoading(long total, long current, boolean isUploading) {
		if(isDownloadBackgroud && mContext != null){
			if(mNoticeManager == null){
				mNoticeManager = (NotificationManager)mContext.getSystemService(Service.NOTIFICATION_SERVICE);
			}
			noticeOnStatuBar(total, current, isUploading);
		}
		if(mProgressView != null){
			mProgressView.initProgressMax(total);
			int ratio = (int)(Double.valueOf(current + "")/Double.valueOf(total + "")*100);
			mProgressView.onProgressChanged(total, current, ratio);
		}
	}
	
	public Context getContext() {
		return mContext;
	}

	public void setContext(Context mContext) {
		this.mContext = mContext;
	}

	public boolean isDownloadBackgroud() {
		return isDownloadBackgroud;
	}
	
	public String getTargetPath() {
		return mTargetPath;
	}

	public void setTargetPath(String targetPath) {
		this.mTargetPath = targetPath;
		this.mDownloadFileName = FileUtils.parseFileName(targetPath);
	}

	public void setDownloadBackgroud(boolean isDownloadBackgroud) {
		this.isDownloadBackgroud = isDownloadBackgroud;
	}
	
	RemoteViews rv = null;
	/**
	 * 状态栏通知下载进度
	 * @param total
	 * @param current
	 * @param isUploading
	 */
	private void noticeOnStatuBar(long total, long current, boolean isUploading){
		
		int ratio = (int)(Double.valueOf(current + "")/Double.valueOf(total + "")*100);
		if(mNotice == null){
			mNotice = new Notification();
			mNotice.flags |= Notification.FLAG_AUTO_CANCEL;
			mNotice.tickerText = mDownloadFileName;
			int iconId = FileUtils.getFileIconId(mContext, mDownloadFileName);
			mNotice.icon = iconId;
			rv = new RemoteViews(mContext.getPackageName(), R.layout.viewjar_progress_notice); 
			rv.setTextViewText(R.id.title, mDownloadFileName);
			rv.setImageViewResource(R.id.notice_icon, iconId);
			mNotice.contentView = rv;
		}
		rv.setTextViewText(R.id.percent, ratio + "%");
		rv.setProgressBar(R.id.progress, 100, ratio, false);
		Intent i = new Intent(mContext, PdfViewActivity.class);
		PendingIntent p=PendingIntent.getActivity(mContext, 0, i, 0);
		mNotice.contentIntent = p;
		mNoticeManager.notify(mTargetPath.hashCode(), mNotice);
	}

    @Override
    public void onWaiting() {

    }

    @Override
    public void onStarted() {

    }

    @Override
    public void onCancelled(CancelledException cex) {

    }

    @Override
    public void onFinished() {

    }

    /**
	 * 下载文件成功后的调用
	 * @param downloadFile
	 */
	public abstract void onDownloadSuccess(File downloadFile);
	
	/**
	 * 下载出错时调用
	 * @param e
	 */
	public abstract void onDownloadFailure(Throwable e);
		
	
	
}
