package com.szboanda.android.platform.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.lang.reflect.Field;
import java.math.BigInteger;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.zip.CRC32;
import java.util.zip.CheckedInputStream;
import java.util.zip.ZipInputStream;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Environment;
import android.provider.MediaStore.Images.ImageColumns;
import android.text.TextUtils;

import com.szboanda.android.platform.activity.OpenEncryptBridgeActivity;
import com.szboanda.android.platform.activity.PdfViewActivity;
import com.szboanda.android.platform.util.ResourceUtils.ResoureType;

public class FileUtils {

	public final static String INVALID_CHAR = "/\\:*?<>|\"";

	public final static String MIME_TYPE_APK = "application/vnd.android.package-archive";
	public final static String MIME_TYPE_HTML = "text/html";
	public final static String MIME_TYPE_IMAGE = "image/*";
	public final static String MIME_TYPE_TXT = "text/plain";
	public final static String MIME_TYPE_WORD = "application/msword";
	public final static String MIME_TYPE_EXCEL = "application/vnd.ms-excel";
	public final static String MIME_TYPE_PDF = "application/pdf";
	public final static String MIME_TYPE_POWER_POINT = "application/vnd.ms-powerpoint";
	public final static String MIME_TYPE_VIDEO = "video/*";
	public final static String MIME_TYPE_AUDIO = "audio/*";
	public final static String MIME_TYPE_CHM = "application/x-chm";

	public static Intent openFile(String filePath){

		File file = new File(filePath);
		if(!file.exists())
			return null;

		/* 取得扩展名 */
		String name = file.getName();
		int lastDotIndex = name.lastIndexOf(".");
		if(lastDotIndex == -1){
			return null;
		}
		//文件后缀名
		String suffix = name.substring( lastDotIndex+1).toLowerCase(Locale.getDefault());
		/* 依扩展名的类型决定MimeType */
		if("m4a".equals(suffix)||"mp3".equals(suffix)||"mid".equals(suffix)||
				"xmf".equals(suffix)||"ogg".equals(suffix)||"wav".equals(suffix)
				||"amr".equals(suffix)||"3gpp".equals(suffix)){
			return getAudioFileIntent(filePath);
		}else if(suffix.equals("3gp")||suffix.equals("mp4")){
			return getVideoFileIntent(filePath);
		}else if("jpg".equals(suffix)||"gif".equals(suffix)||"png".equals(suffix)||
				"jpeg".equals(suffix)||"bmp".equals(suffix)){
			return getImageFileIntent(filePath);
		}else if("apk".equals(suffix)){
			return getApkFileIntent(filePath);
		}else if("ppt".equals(suffix)||"pptx".equals(suffix)){
			return getPptFileIntent(filePath);
		}else if("xls".equals(suffix)||"xlsx".equals(suffix)){
			return getExcelFileIntent(filePath);
		}else if("docx".equals(suffix)||"doc".equals(suffix)){
			return getWordFileIntent(filePath);
		}else if("pdf".equals(suffix)){
			return getPdfFileIntent(filePath);
		}else if("chm".equals(suffix)){
			return getChmFileIntent(filePath);
		}else if("txt".equals(suffix)){
			return getTextFileIntent(filePath,false);
		}else if("htm".equals(suffix)||"html".equals(suffix)||"php".equals(suffix)||"jsp".equals(suffix)){
			return getHtmlFileIntent(filePath);
		}else{
			return null;
		}
	}

	/**Android获取一个用于打开APK文件的intent*/
	public static Intent getApkFileIntent(String filePath) {
		return getFileIntent(filePath, MIME_TYPE_APK);
	}

	/**Android获取一个用于打开VIDEO文件的intent*/
	public static Intent getVideoFileIntent(String filePath) {

		Intent intent = getFileIntent(filePath, MIME_TYPE_VIDEO);
		intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
		intent.putExtra("oneshot", 0);
		intent.putExtra("configchange", 0);
		return intent;
	}

	/**Android获取一个用于打开AUDIO文件的intent*/
	public static Intent getAudioFileIntent(String filePath){

		Intent intent = getFileIntent(filePath, MIME_TYPE_AUDIO);
		intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
		intent.putExtra("oneshot", 0);
		intent.putExtra("configchange", 0);
		return intent;
	}

	/**Android获取一个用于打开Html文件的intent*/
	public static Intent getHtmlFileIntent( String param ){
		Uri uri = Uri.parse(param ).buildUpon().encodedAuthority("com.android.htmlfileprovider").scheme("content").encodedPath(param ).build();
		Intent intent = new Intent(Intent.ACTION_VIEW);
		intent.setDataAndType(uri, MIME_TYPE_HTML);
		return intent;
	}

	/**Android获取一个用于打开图片文件的intent*/
	public static Intent getImageFileIntent(String filePath) {
		return getFileIntent(filePath, MIME_TYPE_IMAGE);
	}

	/**Android获取一个用于打开PPT文件的intent*/
	public static Intent getPptFileIntent(String filePath){
		return getFileIntent(filePath, MIME_TYPE_POWER_POINT);
	}

	/**Android获取一个用于打开Excel文件的intent*/
	public static Intent getExcelFileIntent(String filePath){
		return getFileIntent(filePath, MIME_TYPE_EXCEL);
	}

	/**Android获取一个用于打开Word文件的intent*/
	public static Intent getWordFileIntent(String filePath){
		return getFileIntent(filePath, MIME_TYPE_WORD);
	}

	/**Android获取一个用于打开CHM文件的intent*/
	public static Intent getChmFileIntent( String filePath ){
		return getFileIntent(filePath, MIME_TYPE_CHM);
	}

	//Android获取一个用于打开文本文件的intent
	public static Intent getTextFileIntent( String param, boolean paramBoolean){

		Intent intent = new Intent(Intent.ACTION_VIEW);
		intent.addCategory(Intent.CATEGORY_DEFAULT);
		if (paramBoolean){
			Uri uri1 = Uri.parse(param );
			intent.setDataAndType(uri1, MIME_TYPE_TXT);
		}else{
			Uri uri2 = Uri.fromFile(new File(param ));
			intent.setDataAndType(uri2, "text/plain");
		}
		return intent;
	}

	/**Android获取一个用于打开PDF文件的intent*/
	public static Intent getPdfFileIntent(String filePath){
		Intent intent = new Intent(PdfViewActivity.ACTION_VIEW_PDF);
		Uri uri = Uri.fromFile(new File(filePath));
		intent.setDataAndType(uri, MIME_TYPE_PDF);
		return intent;
	}

	public static Intent getFileIntent(String filePath, String mimeType){
		Intent intent = new Intent(Intent.ACTION_VIEW);
		intent.addCategory(Intent.CATEGORY_DEFAULT);
		Uri uri = Uri.fromFile(new File(filePath));
		intent.setDataAndType(uri, mimeType);
		return intent;
	}

	public static boolean fileExist(String filePath){
		return new File(filePath).exists() ? true:false;
	}

	/**
	 * 复制文件
	 * @param filePath
	 * @param newPath
	 */
	public static void copyFile(String filePath,String newPath){
		if (!TextUtils.isEmpty(filePath) && !TextUtils.isEmpty(newPath)) {
			BufferedOutputStream bops = null;
			BufferedInputStream bips = null;
			FileInputStream ins = null;
			File outFile = new File(newPath);
			try {
				if(!outFile.getParentFile().isDirectory()){
					outFile.getParentFile().mkdirs();
				}else{
					if(!outFile.exists()){
						outFile.createNewFile();
					}
				}
				ins = new FileInputStream(filePath);
				bips = new BufferedInputStream(ins);
				bops = new BufferedOutputStream(new FileOutputStream(outFile));
				byte[] buffer = new byte[2048];
				int readLen = -1;
				while ((readLen = bips.read(buffer)) != -1) {
					bops.write(buffer, 0, readLen);
				}
				bops.flush();
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("create file to [" + newPath + "] error:", e);
			} finally {
				closeInputStream(ins);
				closeInputStream(bips);
				closeOutputStream(bops);
			}
		}
	}

	/**
	 * 将文件流存储成文件
	 * @param ins
	 * @param filePath
	 */
	public static void createFile(InputStream ins,String filePath){
		if(ins !=null && !TextUtils.isEmpty(filePath)){
			BufferedOutputStream bops 	= null;
			BufferedInputStream  bips 	= null;
			try{
				bips = new BufferedInputStream(ins);
				bops = new BufferedOutputStream(new FileOutputStream(filePath));
				byte[] buffer = new byte[2048];
				int len;
				while((len =bips.read()) !=-1){
					bops.write(buffer, 0, len);
				}
				bops.flush();
			}catch(Exception e){
				e.printStackTrace();
			}finally{
				closeInputStream(bips);
				closeOutputStream(bops);

			}
		}
	}


	/**
	 * 从输入流读取totalBytes个字节，写入到指定的文件
	 * <br>不关闭输入流
	 * @param path
	 * @param in
	 * @param totalBytes 从输入流读取内容长度
	 */
	public static File createFile(String path, InputStream in, int totalBytes){
		File file = new File(path);
		try {
			if(file.exists())
				file.delete();
			else
				file.createNewFile();
		} catch (IOException e) {
			e.printStackTrace();
		}
		if(totalBytes > 0 ){
			try {
				FileOutputStream fos = new FileOutputStream(path);
				int len = -1;
				byte[] buffer = getBuffer(totalBytes);
				while(totalBytes > 0 && (len = in.read(buffer)) != -1){
					fos.write(buffer, 0, len);
					totalBytes -= len;
					buffer = getBuffer(totalBytes);//保证最后一次读取的时候，不能多读
				}
				fos.flush();
				fos.close();
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return file;
	}

	/**计算并构建从输入流读信息时的缓存*/
	private static byte[] getBuffer(int totalBytes){
		int len = 8192;//以8192作为最小读取长度
		if(totalBytes > 0 && totalBytes < len)
			len = totalBytes;
		return new byte[len];
	}

	private static File createLogFile(){
		String currentDate = DateUtils.formatDate(new Date(), DateUtils.FORMAT_DATE_NONE);
		String savePath = Environment.getExternalStorageDirectory()+"/Boanda";
		String filePath = savePath + "/"+currentDate+".txt";
		File dir = new File(savePath);
		if(!dir.exists()){
			dir.mkdirs();
		}
		File file = new File(filePath);
		if(!file.exists()){
			try {
				file.createNewFile();
			} catch (IOException e) {
				e.printStackTrace();
				file = null;
			}
		}
		return file;
	}

	/**保存日志，路径Boanda/20151231.txt*/
	public static boolean saveLog(String log){
		File logFile = createLogFile();
		if(logFile == null){
			return false;
		}
		String currentTime = DateUtils.formatDate(new Date(), DateUtils.FORMAT_TIME_HMS);
		log =  currentTime + "\t" + log + "\n";
		try{
			PrintWriter printWriter = new PrintWriter(new FileOutputStream(logFile, true));
			printWriter.write(log);
			printWriter.flush();
			printWriter.close();
			return true;
		}catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	/**保存日志，路径Boanda/20151231.log*/
	synchronized public static boolean saveError(Throwable error){
		File logFile = createLogFile();
		if(logFile == null){
			return false;
		}
		try{
			PrintStream printStream = new PrintStream(new FileOutputStream(logFile, true));
			printStream.println(DateUtils.formatDate(new Date(), DateUtils.FORMAT_DATE_TIME_S));
			error.printStackTrace(printStream);
			printStream.flush();
			printStream.close();
			return true;
		}catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * @param filePath
	 */
	public static void deleteFiles(String filePath){
		File file = new File(filePath);
		if(file.exists()){
			file.delete();
		}
	}

	public static void deleteFile(String matchStr, String unMatchStr,String dirPath){
		File dir = new File(dirPath);
		File[] fileList = dir.listFiles();
		for (File file : fileList) {
			String tempFileName = file.getName();
			if (tempFileName.contains(matchStr)
					&& !tempFileName.contains(unMatchStr)) {
				file.delete();
			}
		}
	}

	/**
	 * 删除文件夹,可递归删除子文件
	 * @param file
	 */
	public static void deleteDirectory(File file){
		if(file.isDirectory()){
			File[] fs = file.listFiles();
			for(int i=0;i<fs.length;i++){
				deleteDirectory(fs[i]);
			}
		}else{
			file.delete();
		}
	}

	/**
	 * 得到不带后缀的文件名
	 *
	 * @param name
	 * @return
	 */
	public static String getSuffixName(String name) {
		return name.substring(0, name.lastIndexOf("."));
	}

	/**
	 * 验证文件名的合法性
	 * @param fileName
	 */
	public static boolean validFileName(String fileName){
		String regex = "["+ INVALID_CHAR +"]+";
		if(fileName.matches(regex)){
			return true;
		}
		return false;

	}

	/**
	 * 把字符串存储到txt文件当中
	 * @param savePath    保存路径
	 * @param name          存储文件名
	 * @param content      字符串
	 * @return
	 */
	public static boolean createFileByContent(String savePath, String name, String content){
		return createFileByContent(savePath, name, content, false);
	}

	/**
	 * 把字符串存储到txt文件当中
	 * @param savePath    保存路径
	 * @param name          存储文件名
	 * @param content      字符串
	 * @param append      是否追加
	 * @return
	 */
	public static boolean createFileByContent(String savePath, String name, String content, boolean append){
		try{
			FileOutputStream fOut = null;
			File dir = new File(savePath);
			if(!dir.exists()){
				dir.mkdirs();
			}
			File file = new File(savePath + File.separator + name);
			fOut = new FileOutputStream(file, append);
			if(content == null){
				content = "";
			}
			fOut.write(content.getBytes());
			fOut.flush();
			fOut.close();
			return true;
		}catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * 将bitmap存储成文件，以JPEG方式压缩保存
	 * @param savePath
	 * @param name
	 * @param bitmap
	 * @param quality
	 * @return boolean true if no exception occurred, else return false
	 */
	public static boolean saveBitmap(String savePath, String name, Bitmap bitmap, int quality){
		FileOutputStream fOut = null;
		try{
			File dir = new File(savePath);
			if(!dir.exists()){
				dir.mkdirs();
			}
			File file = new File(savePath + File.separator + name);
			fOut = new FileOutputStream(file);
			bitmap.compress(Bitmap.CompressFormat.JPEG, quality, fOut);
			fOut.flush();
			fOut.close();
			return true;
		}catch (Exception e) {
			e.printStackTrace();
			closeOutputStream(fOut);
			return false;
		}
	}

	/**
	 * 将bitmap存储成文件
	 * @param savePath
	 * @param name
	 * @param bitmap
	 * @return
	 */
	public static boolean saveBitmap(String savePath, String name, Bitmap bitmap){
		FileOutputStream fOut = null;
		try{
			File dir = new File(savePath);
			if(!dir.exists()){
				dir.mkdirs();
			}
			File file = new File(savePath + File.separator + name);
			fOut = new FileOutputStream(file);
			bitmap.compress(Bitmap.CompressFormat.PNG, 100, fOut);
			fOut.flush();
			fOut.close();
			return true;
		}catch (Exception e) {
			e.printStackTrace();
			closeOutputStream(fOut);
			return false;
		}
	}

	/**
	 * 打开使用默认方式加密(See {@link AESUtils#defaultAesEncrypt(Context, String)})的文件
	 * @param context
	 * @param filePath
	 * @param isEncrypt
	 */
	public static void openFile(Context context, String filePath, boolean isEncrypt){
		Intent intent = null;
		if(isEncrypt){
			intent = new Intent(context, OpenEncryptBridgeActivity.class);
			intent.putExtra(OpenEncryptBridgeActivity.P_REQUIRE_PATH, filePath);
		}else{
			intent = openFile(filePath);
		}
		context.startActivity(intent);
	}

	/**
	 * 关闭输入流
	 * */
	public static void closeInputStream(InputStream... ins){
		if(ins != null && ins.length > 0){
			for(InputStream in : ins){
				if(in != null){
					try {
						in.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}

	/**
	 * 关闭输入通道
	 * */
	public static void closeInputChannel(FileChannel... inChannels){
		if(inChannels != null && inChannels.length > 0){
			for(FileChannel in : inChannels){
				if(in != null){
					try {
						in.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}

	/**
	 * 关闭输出流
	 * */
	public static void closeOutputStream(OutputStream... ous){
		if(ous != null && ous.length > 0){
			for(OutputStream out : ous){
				if(out != null){
					try {
						out.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}

	/**
	 * 关闭输出通道
	 * */
	public static void closeOutputChannel(FileChannel... outChannels){
		if(outChannels != null && outChannels.length > 0){
			for(FileChannel out : outChannels){
				if(out != null){
					try {
						out.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}

	public static String parseFileName(String path){
		if(TextUtils.isEmpty(path)){
			return "";
		}else{
			String repath = path.replace("\\", "/").replace("\\\\", "/").replace("//", "/");
			return new File(repath).getName();
		}
	}

	/**
	 * 由路径解析出文件后缀
	 * @param path
	 * @return path为空或是文件夹时返回空串
	 */
	public static String parseFileSuffix(String path){
		if(!TextUtils.isEmpty(path)){
			int dotIndex = path.lastIndexOf(".");
			if(dotIndex >0){
				return path.substring(dotIndex +1);
			}
		}
		return "";
	}

	/**
	 * 获取文件图标drawable/id
	 * @param context
	 * @param path 文件路径
	 * @return
	 */
	public static int getFileIconId(Context context, String path) {
		String iconName = "";
		try{
			if(!TextUtils.isEmpty(path)){
				String prefix = "libicon_file_";//基础包约定的文件标识图标前缀
				if(new File(path).isDirectory()
						|| TextUtils.isEmpty(parseFileSuffix(path))
						|| !parseFileSuffix(path).matches("^[0-9a-zA-Z]+$")){
					iconName = prefix + "folder";
				}else{
					String suffix = parseFileSuffix(path).toLowerCase(Locale.getDefault());
					String packageName = context.getPackageName();
					Class<?> drawableCls = Class.forName(packageName + ".R$drawable");
					Field[] fields = drawableCls.getDeclaredFields();
					if(fields != null && fields.length>0){
						for(Field f:fields){
							String name = f.getName();
							if(name.startsWith(prefix)){
								//切除多余的约定前缀及文件后缀
								String trimName = name.replace(prefix, "").replace(".png", "");
								List<String> supportSuffixs = Arrays.asList(trimName.split("_"));
								if(supportSuffixs.contains(suffix)){
									iconName = name;
									break;
								}
							}
						}
					}
				}
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		if(TextUtils.isEmpty(iconName)){
			iconName = "libicon_file_unknow";
		}
		int resultId = ResourceUtils.getResourcesId(context, ResoureType.DRAWABLE, iconName);
		return resultId;
	}

	/**
	 * 使用Apache提供的接口解压文件,这种方式支持中文名的压缩文件名或压缩文件内部子文件
	 * @param zipFilePath 压缩文件路径
	 * @param destDir 存储解压文件的目录路径
	 */
	@SuppressWarnings("unchecked")
	public static boolean decompressZipFileByApacheApi(String zipFilePath, String destDir){
		boolean isSuccess = false;
		BufferedInputStream bi;
		ZipFile zf = null;
		BufferedOutputStream bos = null;
		try{
			zf = new ZipFile(zipFilePath, "GBK");
			Enumeration<ZipEntry> e = zf.getEntries();
			while (e.hasMoreElements()) {
				ZipEntry ze = (ZipEntry) e.nextElement();
				String entryName = ze.getName();
				String entryPath = destDir + "/" + entryName;
				if(ze.isDirectory()){
					//创建子文件夹
					File childDir = new File(entryPath);
					if(!childDir.exists()){
						childDir.mkdirs();
					}
				}else{
					String filePath = entryPath.substring(0, entryPath.lastIndexOf("/"));
					File fileDir = new File(filePath);
					if(!fileDir.exists()){
						fileDir.mkdirs();
					}
					FileOutputStream fos = new FileOutputStream(destDir + "/" + entryName);
					bos = new BufferedOutputStream(fos);
					bi = new BufferedInputStream(zf.getInputStream(ze));
					byte[] buffer = new byte[2048];
					int readCount = bi.read(buffer);
					while (readCount != -1) {
						bos.write(buffer, 0, readCount);
						readCount = bi.read(buffer);
					}
					bos.flush();
					bos.close();
				}
			}
			isSuccess = true;
		}catch (Exception e) {
			e.printStackTrace();
		}finally{
			try{
				bos.close();
				zf.close();
			}catch (Exception e) {
				e.printStackTrace();
			}
		}
		return isSuccess;
	}

	/**
	 * 使用Jave提供的原生接口解压文件,这种方式不支持中文名的压缩文件名或压缩文件内部子文件
	 * @param srcPath 压缩文件路径
	 * @param destDir 存储解压文件的目录路径
	 */
	public static boolean decompressZipFileByOriginlApi(String srcPath, String destDir){
		boolean isSuccess = false;
		BufferedInputStream bis = null;
		BufferedOutputStream bos = null;
		try{
			FileInputStream fis = new FileInputStream(srcPath);
			CheckedInputStream cis = new CheckedInputStream(fis, new CRC32());
			ZipInputStream zis = new ZipInputStream(cis);
			bis = new BufferedInputStream(zis);
			java.util.zip.ZipEntry ze = null;
			while ((ze = zis.getNextEntry()) != null) {
				String entryName = ze.getName();
				String entryPath = destDir + "/" + entryName;
				if(ze.isDirectory()){
					File childDir = new File(entryPath);
					if(!childDir.exists()){
						childDir.mkdirs();
					}
				}else{
					FileOutputStream out = new FileOutputStream(entryPath);
					bos = new BufferedOutputStream(out);
					byte[] buffer = new byte[2048];
					int readCount = bis.read(buffer);
					while (readCount != -1) {
						bos.write(buffer, 0, readCount);
						readCount = bis.read(buffer);
					}
					bos.flush();
					bos.close();
				}
			}
			isSuccess = true;
		}catch (Exception e) {
			e.printStackTrace();
		}finally{
			closeOutputStream(bos);
			closeInputStream(bis);
		}
		return isSuccess;
	}


	/**
	 * Try to return the absolute file path from the given Uri
	 *
	 * @param context
	 * @param uri
	 * @return the file path or null
	 */
	public static String getRealFilePath(final Context context, final Uri uri) {
		if (null == uri)
			return null;
		final String scheme = uri.getScheme();
		String path = null;
		if (scheme == null)
			path = uri.getPath();
		else if (ContentResolver.SCHEME_FILE.equals(scheme)) {
			path = uri.getPath();
		} else if (ContentResolver.SCHEME_CONTENT.equals(scheme)) {
			Cursor cursor = context.getContentResolver().query(uri, new String[] { ImageColumns.DATA }, null, null,
					null);
			if (null != cursor) {
				if (cursor.moveToFirst()) {
					int index = cursor.getColumnIndex(ImageColumns.DATA);
					if (index > -1) {
						path = cursor.getString(index);
					}
				}
				cursor.close();
			}
		}
		return path;
	}

	public static File getLastestFile(File[] files){
		if(files == null){
			return null;
		}else{
			return getLastestFile(Arrays.asList(files));
		}
	}

	public static File getLastestFile(List<File> files){
		File lastestFile = null;
		if(files != null && files.size() > 0){
			Collections.sort(files, new LastestFileComparator());
			lastestFile = files.get(0);
		}
		return lastestFile;
	}

	/**较新文件比较器*/
	public static class LastestFileComparator implements Comparator<File>{
		@Override
		public int compare(File lhs, File rhs) {
			return lhs.lastModified() < rhs.lastModified() ? 1 : -1;
		}
	}

	public static class PictureFileSelector implements FileFilter{
		@Override
		public boolean accept(File f) {
			if(f.isDirectory()){
				return false;
			}else{
				String suffix = FileUtils.parseFileSuffix(f.getName())
						.toLowerCase(Locale.getDefault());
				return suffix.equals("png") || suffix.equals("jpg")
						|| suffix.equals("gif") || suffix.equals("jpeg")|| suffix.equals("bmp");
			}
		}
	}

	//FIXME 这几个内部类不应该在这个文件工具类定义
	public static abstract class LetterOrder<T> implements Comparator<T>{

		@Override
		public abstract int compare(T lhs, T rhs);

		public int compareName(String lhs, String rhs){
	        String firstName = lhs;
	        if(firstName.getBytes().length != firstName.length())
	        	firstName = PinyinUtils.getPinyinFirstChars(firstName);
	        firstName = firstName.toLowerCase(Locale.getDefault());
	        String anotherName = rhs;
	        if(anotherName.getBytes().length != anotherName.length())
	        	anotherName = PinyinUtils.getPinyinFirstChars(anotherName);
	        anotherName = anotherName.toLowerCase(Locale.getDefault());
	        return firstName.compareTo(anotherName);
		}

	}

	/**
     * 获取文件夹中文件的MD5值，可用于判断文件更新或一致性
     * @param file
     * @param algorithm 所请求算法的名称  for example: MD5, SHA1, SHA-256, SHA-384, SHA-512 etc.
     * @return
     */
    public static String getFileMD5(File file,String algorithm) {
        if (!file.isFile()) {
            return null;
        }
        MessageDigest digest = null;
        FileInputStream in = null;
        byte buffer[] = new byte[1024];
        int len;
        try {
            digest = MessageDigest.getInstance(algorithm);
            in = new FileInputStream(file);
            while ((len = in.read(buffer, 0, 1024)) != -1) {
                digest.update(buffer, 0, len);
            }
            in.close();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        BigInteger bigInt = new BigInteger(1, digest.digest());
        return bigInt.toString(16);
    }

    /**
     * 获取文件夹中文件的MD5值
     *
     * @param dirFile
     * @param algorithm 所请求算法的名称  for example: MD5, SHA1, SHA-256, SHA-384, SHA-512 etc.
     * @param listChild 是否递归子目录中的文件
     * @return
     */
    public static Map<String, String> getDirMD5(File dirFile,String algorithm, boolean listChild) {
        if (!dirFile.isDirectory()) {
            return null;
        }
        // <filepath,algCode>
        Map<String, String> pathAlgMap = new HashMap<String, String>();
        String algCode;
        File files[] = dirFile.listFiles();

        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            if (file.isDirectory() && listChild) {
                pathAlgMap.putAll(getDirMD5(file, algorithm, listChild));
            } else {
                algCode = getFileMD5(file, algorithm);
                if (algCode != null) {
                    pathAlgMap.put(file.getPath(), algCode);
                }
            }
        }
        return pathAlgMap;
    }

    /**
	 * 获取文件大小
	 */
	public static String getFileSize(File file){
		long length = 0l;
		if(file != null && file.exists()){
			length = file.length();
		}
		return formatFileSize(length);
	}

	/**
	 * 获取文件大小
	 * @param length 单位B
	 * @return
	 */
	public static String formatFileSize(long length) {
		DecimalFormat df = new DecimalFormat("#.00");
		UnitType unit = UnitType.B;
		String size = "0";
		if (length < 1024) {
			size = df.format(length);
		} else if (length < 1048576) {
			size = df.format((double) length / 1024);
			unit = UnitType.KB;
		} else if (length < 1073741824) {
			size = df.format((double) length / 1048576);// 1024*1024
			unit = UnitType.MB;
		} else {
			size = df.format((double) length / 1073741824);// 1024*1024*1024
			unit = UnitType.GB;
		}
		return size + " " + unit.name();
	}

	enum UnitType{
		B,KB,MB,GB,TB
	}

	/**
	* 判断图片是否损毁
	* @param filePath
	* @return boolean    返回类型
	 */
	public static boolean ifDestoryImg(String filePath){
		BitmapFactory.Options options = null;
		if (options == null) options = new BitmapFactory.Options();
		     options.inJustDecodeBounds = true;

		BitmapFactory.decodeFile(filePath, options); //filePath代表图片路径
        if (options.mCancel || options.outWidth == -1 || options.outHeight == -1) {
               //表示图片已损毁
        	return true;
        }
        return false;
	}

	public static String readContent(String path){
		return readContent(path, "utf-8");
	}

	public static String readContent(String path, String charsetName){
		File file = new File(path);
		if(file.exists()){
			StringBuffer buffer = new StringBuffer();
			BufferedReader reader = null;
			try{
				reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), charsetName));
				String temp = "";
				while ((temp = reader.readLine()) != null) {
					buffer.append(temp);
				}
			}catch (Exception e) {
				e.printStackTrace();
			}finally{
				try {
					reader.close();
				} catch (Exception fe) {
					fe.printStackTrace();
				}
			}
			return buffer.toString();
		}else{
			return null;
		}
	}

    public static String readFile(InputStream fileStream, String charsetName){
        StringBuffer buffer = new StringBuffer();
        BufferedReader reader = null;
        try{
            reader = new BufferedReader(new InputStreamReader(fileStream, charsetName));
            String temp = "";
            while ((temp = reader.readLine()) != null) {
                buffer.append(temp);
            }
        }catch (Exception e) {
            e.printStackTrace();
        }finally{
            try {
                reader.close();
            } catch (Exception fe) {
                fe.printStackTrace();
            }
        }
        return buffer.toString();
    }
}
