package com.szboanda.android.platform.util;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.support.annotation.NonNull;
import android.view.View;
import android.view.ViewGroup;

import java.lang.reflect.Method;

/** 
 * 类说明： 
 * @author 苏浩
 * @date 2015-7-9
 * @version 1.0
 */

public class ViewUtils {
	
	@SuppressWarnings("unchecked")
	public static <E> E findViewAutoConvert(View rv, int id){
		View v = rv.findViewById(id);
		return v == null ? null : (E)v;
	}

	/** 测量视图宽高 
	 * @return [width,height]*/
	public static int[] measureView(View v) {
		int[] size = new int[2];
		if(v != null){
			int width = 0;
			int height = 0;
			int wmode = View.MeasureSpec.UNSPECIFIED;
			int hmode = View.MeasureSpec.UNSPECIFIED;
			ViewGroup.LayoutParams params = v.getLayoutParams();
			if(params != null){
				if(params.width == ViewGroup.LayoutParams.WRAP_CONTENT){
					wmode = View.MeasureSpec.AT_MOST;
					width = (1<<30)-1;
				}else if(params.width > 0 ){
					wmode = View.MeasureSpec.EXACTLY;
					width = params.width;
				}
				if(params.height == ViewGroup.LayoutParams.WRAP_CONTENT){
					hmode = View.MeasureSpec.AT_MOST;
					height = (1<<30)-1;
				}else if(params.height > 0 ){
					hmode = View.MeasureSpec.EXACTLY;
					height = params.height;
				}
			}
			width = View.MeasureSpec.makeMeasureSpec(width, wmode);
			height = View.MeasureSpec.makeMeasureSpec(height, hmode);
			v.measure(width, height);//测量宽高
			size[0] = v.getMeasuredWidth();
			size[1] = v.getMeasuredHeight();
		}
		return size;
	}
	
	/**View转Bitmap
	 * @see #createSnapshot(View)
	 * @param view 当是ActionBarView实例，布局非match_parent时抛异常*/
	public static Bitmap getViewBitmap(View view) {
		if(view == null)
			return null;
		view.setDrawingCacheEnabled(true);
		view.measure(View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED),
				View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED));
		view.layout(0, 0, view.getMeasuredWidth(), view.getMeasuredHeight());
		view.buildDrawingCache();
		Bitmap cacheBitmap = view.getDrawingCache();
		Bitmap bitmap = Bitmap.createBitmap(cacheBitmap);
		view.setDrawingCacheEnabled(false);
		return bitmap;
	}
	
	/**
	 * 反射使用View.createSnapshot创建视图快照/截图
	 * <br/>参阅相关源码
	 * @param view
	 * @return
	 */
	public static Bitmap createSnapshot(View view) {
		Bitmap bitmap = null;
		if (view != null) {
			Class<?> cls = view.getClass();
			while (!cls.getName().equals("android.view.View")) {
				cls = cls.getSuperclass();
			}
			try {
				Method method = cls.getDeclaredMethod("createSnapshot", Bitmap.Config.class, int.class, boolean.class);
				method.setAccessible(true);
				bitmap = (Bitmap) method.invoke(view, Bitmap.Config.ARGB_8888, Color.WHITE, false);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return bitmap;
	}

	/**
	 * 渐变动画
	 * @param inView
	 * @param duration
     */
	public static void crossFadeIn(@NonNull View inView,long duration){
		if (inView != null) {
			inView.setAlpha(0f);
			inView.setVisibility(View.VISIBLE);

			inView.animate()
					.alpha(1f)
					.setDuration(duration)
					.setListener(null);
		}
	}

	/**
	 * 渐变动画
	 * @param outView
	 * @param duration
     */
	public static void crossFadeOut(@NonNull final View outView, long duration){
		if (outView != null) {
			outView.animate()
					.alpha(0f)
					.setDuration(duration)
					.setListener(new AnimatorListenerAdapter() {
						@Override
						public void onAnimationEnd(Animator animation) {
							outView.setVisibility(View.GONE);
						}
					});
		}
	}

	/**
	 * 渐变动画
	 * @param inView
	 * @param outView
	 * @param duration
     */
	public static void crossFade(@NonNull View inView, @NonNull final View outView, long duration){
		if (inView != null) {
			inView.setAlpha(0f);
			inView.setVisibility(View.VISIBLE);

			inView.animate()
					.alpha(1f)
					.setDuration(duration)
					.setListener(null);
		}

		if (outView != null) {
			outView.animate()
					.alpha(0f)
					.setDuration(duration)
					.setListener(new AnimatorListenerAdapter() {
						@Override
						public void onAnimationEnd(Animator animation) {
							outView.setVisibility(View.GONE);
						}
					});
		}
	}
	
}
