package com.szboanda.android.platform.util;

import java.io.File;

import android.app.Activity;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.CursorLoader;
import android.content.Intent;
import android.database.Cursor;
import android.net.Uri;
import android.provider.ContactsContract.CommonDataKinds.Organization;
import android.provider.ContactsContract.CommonDataKinds.Phone;
import android.provider.ContactsContract.CommonDataKinds.StructuredName;
import android.provider.ContactsContract.RawContacts;
import android.provider.ContactsContract.RawContacts.Data;
import android.provider.MediaStore;
import android.provider.MediaStore.Audio.Media;
import android.text.TextUtils;
import android.widget.Toast;

/**
 * 系统功能帮助类，主要用于拍照、相册、录音、视频
 * @author Lu Siyi
 * @version : 1.0 
 * @Create Date : 2013-10-27
 */
public class SystemFunctionHelper {
	
	/**
	 * 启动照相机进行拍照，使用默认的存储路径
	 */
	public static void startupCamera(Activity activity, int requestCode){
		startupCameraWithPath(activity,requestCode, null);
	}
	
	/**
	 * 启动照相机进行拍照，并指定存储路径
	 * @param extraPath  存储路径
	 */
	public static void startupCameraWithPath(Activity activity, int requestCode, String extraPath){
		
		Intent intent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
		if(extraPath != null){
			String name = new File(extraPath).getName();
			String dirPath = extraPath.replace(name, "");
			File dir = new File(dirPath);
			if(!dir.exists()){
				dir.mkdirs();
			}
			File saveFile = new File(extraPath);
			Uri saveFileUri = Uri.fromFile(saveFile);
			intent.putExtra(MediaStore.EXTRA_OUTPUT, saveFileUri);
		}
		activity.startActivityForResult(intent, requestCode);
	}
	
	/**
	 * 调用系统录音功能
	 * <p>使用方法：
	 * <br/>1、调用当前方法
	 * <br/>2、onActivityResult中用getFileFromUri方法从返回的Intent对象中拿到录音文件路径地址<p/>
	 * @param activity 调用该方法的Activity
	 * @param requestCode  onActivityResult接收返回值时的标识码
	 */
	public static void startupRecordSound(Activity activity, int requestCode){
		try {
			Intent intent = new Intent(Media.RECORD_SOUND_ACTION);
			activity.startActivityForResult(intent, requestCode);
		} catch (Exception e) {
			Toast.makeText(activity, "打开系统录音失败", Toast.LENGTH_SHORT).show();
		}
	}
	
	/**
	 * 打开系统录音界面
	 * @param activity
	 * @param requestCode
	 */
	public static void startupSoundRecorder(Activity activity, int requestCode){
		Intent intent = new Intent(Intent.ACTION_MAIN);
		intent.addCategory(Intent.CATEGORY_LAUNCHER);
		intent.addCategory(Intent.CATEGORY_DEFAULT);
		intent.setPackage("com.android.soundrecorder");
		activity.startActivityForResult(intent, requestCode);
	}
	
	/**
	 * 调用系统摄像功能
	 * <p>使用方法：
	 * <br/>1、调用当前方法
	 * <br/>2、onActivityResult中用getFileFromUri方法从返回的Intent对象中拿到摄像视频文件路径地址<p/>
	 * @param activity 调用该方法的Activity
	 * @param requestCode  onActivityResult接收返回值时的标识码
	 */
	public static void startupVideo(Activity activity, int requestCode){
		Intent intent = new  Intent(MediaStore.ACTION_VIDEO_CAPTURE);
		activity.startActivityForResult(intent, requestCode);
	}
	
	/**
	 * 调用系统摄像功能，并指定画质
	 * <p>使用方法：
	 * <br/>1、调用当前方法
	 * <br/>2、onActivityResult中用getFileFromUri方法从返回的Intent对象中拿到摄像视频文件路径地址<p/>
	 * @param activity 调用该方法的Activity
	 * @param requestCode  onActivityResult接收返回值时的标识码
	 * @param videoQuality  0：低画质  1：高画质
	 */
	public static void startupVideo(Activity activity, int requestCode,int videoQuality){
		Intent intent = new  Intent(MediaStore.ACTION_VIDEO_CAPTURE);
		intent.putExtra(MediaStore.EXTRA_VIDEO_QUALITY,videoQuality);
		activity.startActivityForResult(intent, requestCode);
	}
	
	
	
	/**从返回的Intent对象中的Uri里获取拍照、录音、摄像的存储的路径地址*/
	public static String getFileFromUri(Activity activity, Intent data) {
		if (data != null) {
			Uri uri = data.getData();
			String[] proj = { MediaStore.Images.Media.DATA };
			CursorLoader cursorLoader = new CursorLoader(activity, uri, proj, null, null, null);
			Cursor imageCursor = cursorLoader.loadInBackground();
			int columnIndex = imageCursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA);
			imageCursor.moveToFirst();
			String path = imageCursor.getString(columnIndex);
			return path;
		}
		return "";
	}
	
	/**
	 * 启动图图库获取图片
	 */
	public static void startupImageStoreForPath(Activity activity, int requestCode){
		
		Intent intent = new Intent(Intent.ACTION_PICK, null);
		intent.setDataAndType(MediaStore.Images.Media.EXTERNAL_CONTENT_URI, "image/*");
		activity.startActivityForResult(intent, requestCode);
		
	}
	
	/**
	 * 裁剪图片（自由裁剪）
	 * <p>使用方法：请根据onActivityResult的返回结果，然后从保存路径拿到裁剪后的图片<p/>
	 * @param cropImg 需要进行裁剪的图片；如果是文件路径请用：Uri.fromFile(new File(path))
	 * @param extraPath 裁剪后的图片保存路径
	 * @param requestCode onActivityResult接收返回值时的标识码
	 */
	public static void cropImg(Activity activity,Uri cropImg,String extraPath,int requestCode){
		cropImg(activity,cropImg,extraPath,-1,-1,requestCode);
	}
	
	/**
	 * 裁剪图片
	 * <p>使用方法：请根据onActivityResult的返回结果，然后从保存路径拿到裁剪后的图片<p/>
	 * @param cropImg 需要进行裁剪的图片；如果是文件路径请用：Uri.fromFile(new File(path))
	 * @param extraPath 裁剪后的图片保存路径
	 * @param width 裁剪后的图片宽度（大于0时才有效）
	 * @param height 裁剪后的图片高度（大于0时才有效）
	 * @param requestCode onActivityResult接收返回值时的标识码
	 */
	public static void cropImg(Activity activity,Uri cropImg,String extraPath,int width,int height,int requestCode) {
		Intent intent = new Intent("com.android.camera.action.CROP"); 
		intent.setDataAndType(cropImg, "image/*");
		intent.putExtra("crop", "true");
		if(width>0&&height>0){
			// aspectX aspectY 是宽高的比例
			intent.putExtra("aspectX", width);
			intent.putExtra("aspectY", height);
			// outputX outputY 是裁剪图片宽高
			intent.putExtra("outputX", width);
			intent.putExtra("outputY", height);
		}
		intent.putExtra("noFaceDetection", true); //取消人脸识别功能
//		intent.putExtra("return-data", true);//当裁剪后图片过大时，该种方式不能正常返回
		intent.putExtra("output", Uri.fromFile(new File(extraPath)));
		activity.startActivityForResult(intent, requestCode);
	}

	/**
	 * 打开拨号
	 * @param context
	 * @param number
	 */
	public static void startDialNumber(Context context,String number){
		Intent intent = new Intent(Intent.ACTION_DIAL);
		intent.setData(Uri.parse("tel:"+number));
		context.startActivity(intent);
	}
	
	/**
	 * 发送信息
	 * @param context
	 * @param number
	 * @param message
	 */
	public static void startSendMessage(Context context,String number,String message){
		Intent intent = new Intent(Intent.ACTION_SENDTO);
		intent.setData(Uri.parse("smsto:"+number));
		if(!TextUtils.isEmpty(message))
			intent.putExtra("sms_body", message);
		context.startActivity(intent);
	}
	
	/**
	 * 发送邮件
	 * @param context
	 * @param address
	 */
	public static void startSendMail(Context context,String address){
		Intent intent = new Intent(Intent.ACTION_SENDTO);
		intent.setData(Uri.parse("mailto:"+address));
		context.startActivity(intent);
	}
	
	/**
	 * 发送文件
	 * @param context
	 * @param data
	 */
	public static void startShare(Context context,Uri data){
		Intent sendIntent = new Intent();  
		sendIntent.setAction(Intent.ACTION_SEND);  
		sendIntent.setType("*/*");
		sendIntent.putExtra(Intent.EXTRA_STREAM, data);
		context.startActivity(sendIntent); 
	}
	
	/**插入联系人*/
	public static void addContact(Context context, String name, String bmmc, String phone, String bgdh) {

		ContentValues values = new ContentValues();  
        //首先向RawContacts.CONTENT_URI执行一个空值插入，目的是获取系统返回的rawContactId   
        Uri rawContactUri = context.getContentResolver().insert(RawContacts.CONTENT_URI, values);  
        if(rawContactUri==null){
        	Toast.makeText(context, "请检查SIM卡", Toast.LENGTH_LONG).show();
        	return ;
        }
        //获取id  
        long rawContactId = ContentUris.parseId(rawContactUri);  
        //往data表插入姓名数据  
        values.clear();  
        values.put(Data.RAW_CONTACT_ID, rawContactId); //添加id  
        values.put(Data.MIMETYPE, StructuredName.CONTENT_ITEM_TYPE);//添加内容类型（MIMETYPE）  
        values.put(StructuredName.GIVEN_NAME, name);//添加名字，添加到first name位置  
        context.getContentResolver().insert(android.provider.ContactsContract.Data.CONTENT_URI, values);  
        //往data表插入手机数据  
        values.clear();  
        values.put(Data.RAW_CONTACT_ID, rawContactId);  
        values.put(Data.MIMETYPE, Phone.CONTENT_ITEM_TYPE);  
        values.put(Phone.NUMBER, phone);  
        values.put(Phone.TYPE, Phone.TYPE_MOBILE);  
        context.getContentResolver().insert(android.provider.ContactsContract.Data.CONTENT_URI, values);
        //往data表插入电话数据  
        values.clear();  
        values.put(Data.RAW_CONTACT_ID, rawContactId);  
        values.put(Data.MIMETYPE, Phone.CONTENT_ITEM_TYPE);  
        values.put(Phone.NUMBER, bgdh);  
        values.put(Phone.TYPE, Phone.TYPE_WORK);  
        context.getContentResolver().insert(android.provider.ContactsContract.Data.CONTENT_URI, values);
        //往data表插入电话数据  
        values.clear();  
        values.put(Data.RAW_CONTACT_ID, rawContactId);  
        values.put(Data.MIMETYPE, Organization.CONTENT_ITEM_TYPE);  
        values.put(Organization.DEPARTMENT, bmmc);  
        values.put(Organization.TYPE, Organization.TYPE_WORK);  
        context.getContentResolver().insert(android.provider.ContactsContract.Data.CONTENT_URI, values);
        
	}

    public static void backToHomeDesktop(Context context){
        Intent home = new Intent(Intent.ACTION_MAIN);
        home.addCategory(Intent.CATEGORY_HOME);
        home.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
        context.startActivity(home);
    }
	
}
