package com.szboanda.android.platform.util;

import java.util.HashSet;
import java.util.Set;

import com.baidu.location.BDGeofence;
import com.baidu.location.BDLocation;
import com.baidu.location.BDLocationListener;
import com.baidu.location.LocationClient;
import com.baidu.location.LocationClientOption;
import com.baidu.location.LocationClientOption.LocationMode;

import android.content.Context;
import android.location.LocationManager;
import android.util.Log;

/**
 * <p>Description: 网络定位帮助类</p>
 * @Company    深圳市博安达信息技术股份有限公司
 * @author    Siyi Lu    Create Date: 2015-4-30
 * @modify  
 */
public class LocationHelper {
	
	public static final int DEFAULT_SCANSPAN = 5*1000;
	private static final String TAG = "LocationHelper.java";
	
	/**百度KEY错误代码值*/
	static Set<Integer> CodeKeyError = new HashSet<Integer>();
	/**有效位置代码值*/
	static Set<Integer> CodeEftLoc = new HashSet<Integer>();
	static{
		//以下添加的代码值代表的意见请参见百度官网API
		CodeEftLoc.add(61);
		CodeEftLoc.add(65);
		CodeEftLoc.add(66);
		CodeEftLoc.add(68);
		CodeEftLoc.add(161);
		
		CodeKeyError.add(501);
		CodeKeyError.add(502);
		CodeKeyError.add(505);
		CodeKeyError.add(601);
		CodeKeyError.add(602);
		CodeKeyError.add(700);
	}
	
	private volatile static LocationHelper INSTANCE = null;
	
	private static Object mLock = new Object(); 
	
	private Context mContext;
	
	private LocationClient mClient;
	
	private LocationClientOption mOption;
	
	private PinLocationListener mListener;
	
	private boolean isRegisteredListener = false;
	
	private int scanSpan = 0;
	
	public static LocationHelper getInstance(){
		if(INSTANCE == null){
			synchronized (mLock) {
				if(INSTANCE == null){
					INSTANCE = new LocationHelper();
				}
			}
		}
		return INSTANCE;
	}
	
	public void initContext(Context context){
		mContext = context;
		this.scanSpan = DEFAULT_SCANSPAN;
	}
	
	public void setScanSpan(int scanSpan){
		this.scanSpan = scanSpan;
		if(mOption != null){
			mOption.setScanSpan(scanSpan);
		}
	}
	
	public int getScanSpan(){
		return scanSpan;
	}
	
	/**
	 * 默认判断网络
	 */
	public void startLocation(){
		startLocation(NetworkUtils.isNetworkOpened(mContext), null);
	}
	
	/**
	 * 指定网络状态
	 * @param isNetConnected
	 */
	public void startLocation(boolean isNetConnected){
		startLocation(isNetConnected, null);
	}
	
	/**
	 * 默认判断网络，并指定监听器
	 * @param listener
	 */
	public void startLocation(PinLocationListener listener){
		startLocation(NetworkUtils.isNetworkOpened(mContext), listener);
	}
	
	public void startLocation(boolean isNetConnected, PinLocationListener listener){
		if(mClient == null){
			boolean inited = initOptions(isNetConnected);
			if(!inited){
				return;
			}
			mClient = new LocationClient(mContext);
			mClient.setLocOption(mOption);
		}
		//listener==null时，mClient未注册任何监听器，定位时，会报错
		if(listener != null){
			mListener = listener;
			mClient.unRegisterLocationListener(mBDListener);
			mClient.registerLocationListener(mBDListener);
			isRegisteredListener = true;
		}
		if(mClient.isStarted()){
			if(isNetConnected)
				mClient.requestLocation();
			else
				mClient.requestOfflineLocation();
		}else{
			mClient.start();
		}
	}
	
	/**
	 * 初始化定位参数
	 * @param isNetConnected
	 * @return 网络未连接且GPS没打开情况下不定位
	 */
	private boolean initOptions(boolean isNetConnected){
		if(mOption == null){
			mOption = new LocationClientOption();
			mOption.setCoorType(BDGeofence.COORD_TYPE_BD09LL);
			if(isNetConnected){//网络打开优先使用网络定位
				if(isGpsOpened(mContext)){
					mOption.setLocationMode(LocationMode.Hight_Accuracy);
					mOption.setOpenGps(true);
				}else{
					mOption.setLocationMode(LocationMode.Battery_Saving);
				}
			}else if(isGpsOpened(mContext)){
				mOption.setOpenGps(true);
				mOption.setProdName(LocationManager.GPS_PROVIDER);
				mOption.setLocationMode(LocationMode.Device_Sensors);
			}else{
				return false;
			}
			mOption.setScanSpan(scanSpan);
			mOption.setIsNeedAddress(true);
			mOption.setNeedDeviceDirect(true);
			mOption.setTimeOut(10*1000);
			return true;
		}else{
			return true;
		}
	}
	
	public void stopLocation(){
		if(isRegisteredListener){
			mClient.unRegisterLocationListener(mBDListener);
			isRegisteredListener = false;
		}
		if(mClient != null){
			mClient.stop();
		}
	}
	
	/**
	 * 判断GPS是否开启
	 * @return
	 */
	public static boolean isGpsOpened(Context context){
		boolean isOpen = false;
		LocationManager locMag = (LocationManager)context.getSystemService(Context.LOCATION_SERVICE);
		if(locMag.isProviderEnabled(LocationManager.GPS_PROVIDER)){
			isOpen = true;
		}
		return isOpen;
	}
	
	public void setOption(LocationClientOption option){
		mOption = option;
	}
	
	public boolean isStoped(){
		return mClient == null || !mClient.isStarted();
	}
	
	private BDLocationListener mBDListener = new BDLocationListener(){
		@Override
		public void onReceiveLocation(BDLocation location) {
			if(scanSpan<1000)
				stopLocation();
			int locType = location.getLocType();
			if(CodeEftLoc.contains(locType)){
				mListener.onSucess(location);
			}else if(CodeKeyError.contains(locType)){
				Log.e(TAG, "请正确配置百度KEY, 错误代码：" + locType);
			}else{
				mListener.onFailure(locType);
			}
		}
	};
	
	public interface PinLocationListener{
		
		public void onSucess(BDLocation location);
		
		public void onFailure(int errorCode);
		
	}

}
