package com.szboanda.android.platform.util;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONObject;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;

import android.text.TextUtils;
import android.util.Log;

public class BeanUtil {
	
	public static final Class<?> loadClass(String className) {
		Class<?> c = null;
		if(!TextUtils.isEmpty(className)){
			try {
				c = Class.forName(className);
			} catch (Exception e) {
				e.printStackTrace();
				c = null;
			}
		}
		return c;
	}
	
	public static final <E> E loadInstance(String className){
		return loadInstance(loadClass(className));
	}

	@SuppressWarnings("unchecked")
	public static <E> E loadInstance(Class<?> cls) {
		E c = null;
		if(cls != null){
			try {
				Object instance = cls.newInstance();
				c = (E)instance;
			} catch (Exception e) {
				e.printStackTrace();
				c = null;
			}
		}
		return c;
	}
	
	/**
	 * 直接设置对象属性值,无视private/protected修饰符,不经过setter函数.
	 */
	public static void setFieldValue(final Object object,
			final String fieldName, final Object value) {
		Field field = getDeclaredField(object, fieldName);

		if (field == null)
			throw new IllegalArgumentException("Could not find field ["
					+ fieldName + "] on target [" + object + "]");
		setFieldValue(object, field, value);
	}
	
	/**
	 * 直接设置对象属性值,无视private/protected修饰符,不经过setter函数.
	 */
	public static void setFieldValue(final Object object, Field field, final Object value) {
		makeAccessible(field);
		try {
			field.set(object, value);
		} catch (IllegalAccessException e) {
			Log.e("zbkc", "", e);
		}
	}
	
	/**
	 * 循环向上转型,获取对象的DeclaredField.
	 */
	protected static Field getDeclaredField(final Object object,
			final String fieldName) {
		return getDeclaredField(object.getClass(), fieldName);
	}
	
	/**
	 * 循环向上转型,获取类的DeclaredField.
	 */
	protected static Field getDeclaredField(final Class<?> clazz,
			final String fieldName) {
		for (Class<?> superClass = clazz; superClass != Object.class; superClass = superClass
				.getSuperclass()) {
			try {
				return superClass.getDeclaredField(fieldName);
			} catch (NoSuchFieldException e) {
				// Field不在当前类定义,继续向上转型
			}
		}
		return null;
	}
	
	/**
	 * 强制转换fileld可访问.
	 */
	public static void makeAccessible(Field field) {
		if (!Modifier.isPublic(field.getModifiers())
				|| !Modifier.isPublic(field.getDeclaringClass().getModifiers())) {
			field.setAccessible(true);
		}
	}
	
	public static List<Field> getDeclaredFields(Class<?> cls){
		List<Field> result = new ArrayList<Field>();
		getDeclaredFields(cls, result);
		return result;
	}
	
	private static void getDeclaredFields(Class<?> cls, List<Field> fields){
		Field[] temp = cls.getDeclaredFields();
		for(Field f:temp){
			fields.add(f);
		}
		if(cls.getSuperclass() != null) {
			getDeclaredFields(cls.getSuperclass(), fields);
		}
	}
	
	/**
	 * 将JSON字符串转换为对象实体
	 * @param jsonStr
	 * @param c
	 * @return
	 */
	public static <E> E  convertJsonStr2Entity(String jsonStr, Type c){
		E result = null;
		if(!TextUtils.isEmpty(jsonStr) && c != null){
			Gson convert = new GsonBuilder()
					.registerTypeAdapter(Date.class, new DateTypeAdapter())
					.create();
			result = convert.fromJson(jsonStr, c);
		}
		return result;
	}
	
	/**
	 * 将JSON数组字符串转化为实体集合
	 * @param arrayStr
	 * @param type 如：new TypeToken&lt;List&lt;User&gt;&gt;(){}.getType()
	 * @return
	 */
	public static <E> List<E> convertArrayStr2Entitys(String arrayStr, Type type){
		List<E> result = null;
		if(!TextUtils.isEmpty(arrayStr) && type != null){
			Gson convert = new GsonBuilder()
					.registerTypeAdapter(Date.class, new DateTypeAdapter())
					.create();
			result = convert.fromJson(arrayStr, type);
		}
		return result;
	}
 
	
	public static JSONObject entitys2JsonObj(Object c ){
		JSONObject obj = null;
		try {
			if(c != null){
				Gson convert = new GsonBuilder()
						.registerTypeAdapter(Date.class, new DateTypeAdapter())
						.create();
				String jsonStr = convert.toJson(c);
				obj = new JSONObject(jsonStr);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return obj;
	}
	
	@SuppressWarnings("unchecked")
	public static <T, S> List<T> convertImplToEntity(List<S> source){
		List<T> target = null;
		if(source != null && source.size() > 0){
			target = new ArrayList<T>();
			for(S s:source){
				target.add((T)s);
			}
		}
		return target;
	} 

	/**
	 * 将对象集合转化为JsonArr
	 * @param <T>
	 * @param entitys
	 * @param type 如：new TypeToken&lt;List&lt;User&gt;&gt;(){}.getType()
	 * @return
	 */
	public static <T> JSONArray convertEntitys2JsonArr(List<T> entitys, Type type){
		JSONArray resultArr = new JSONArray();
		try {
			if(entitys!=null&&entitys.size()>0){
				Gson convert = new GsonBuilder()
						.registerTypeAdapter(Date.class, new DateTypeAdapter())
						.create();
				String jsonArrStr = convert.toJson(entitys, type);
				resultArr = new JSONArray(jsonArrStr);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return resultArr;
	}
	
	public static class DateTypeAdapter implements JsonSerializer<Date>, JsonDeserializer<Date>{

		@Override
		public Date deserialize(JsonElement json, Type type, JsonDeserializationContext jsc)
				throws JsonParseException {
			if (!(json instanceof JsonPrimitive)) {      
	             throw new JsonParseException("The date should be a string value");      
	         } 
			return DateUtils.parseDate(json.getAsString(), DateUtils.FORMAT_DATE_TIME_M);
		}

		@Override
		public JsonElement serialize(Date date, Type type, JsonSerializationContext jsc) {
			return new JsonPrimitive(DateUtils.formatDate(date, DateUtils.FORMAT_DATE_TIME_M));
		}
		
	}
}
