package com.szboanda.android.platform.util;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import android.content.Context;
import android.util.Log;

/**
 * AES加密解密工具类
 * @Company 深圳市博安达软件开发有限公司
 * @author  HuangYuSong Create Date：2014-10-31
 */
public class AESUtils {
	public static final String TAG = AESUtils.class.getSimpleName();
	
	private boolean randomKey = false;//默认不是用随机值，packageName.getBytes();
	/**
	 * 默认使用系统包名做密钥对文件进行AES加密，加密文件存储路径与明文文件一致，<br/>
	 *文件名使用{@link MessageDigestHelper#getHexStringDigest(String)}方法进行转换
	 * @param explicitFilePath
	 * @return return true if success, otherwise false
	 */
	public boolean defaultAesEncrypt(Context context, String explicitFilePath){
		String packageName = context.getPackageName();
		File explicitFile = new File(explicitFilePath);
		String explicitPath = explicitFile.getPath();
		String explicitName = explicitFile.getName();
		String digestName = new MessageDigestHelper().getHexStringDigest(explicitName);
		String encryptPath = explicitPath.replace(explicitName, digestName);
		File encryptFile = new File(encryptPath);
		if(encryptFile.exists() && (encryptFile.length()==explicitFile.length())){
			return true;
		}
		return randomKey?aesCipherRandom(Cipher.ENCRYPT_MODE, explicitFilePath, encryptPath, packageName):aesCipherStable(Cipher.ENCRYPT_MODE, explicitFilePath, encryptPath, packageName);
	}
	
	/**
	 * 默认使用系统包名做密钥对文件进行AES加密
	 * @param sourceFilePath
	 * @param targetFilePath
	 * @return return true if success, otherwise false
	 */
	public boolean defaultAesEncrypt(Context context, String sourceFilePath, String targetFilePath){
		String packageName = context.getPackageName();
		return randomKey?aesCipherRandom(Cipher.ENCRYPT_MODE, sourceFilePath, targetFilePath, packageName):aesCipherStable(Cipher.ENCRYPT_MODE, sourceFilePath, targetFilePath, packageName);
	}
	
	/**
	 * 默认使用系统包名做密钥对文件进行AES解密,加密文件存储路径与明文文件一致，<br/>
	 *文件名使用{@link MessageDigestHelper#getHexStringDigest(String)}方法进行转换
	 * @param explicitFilePath 注意，这里传进来的是明文文件路径，会根据它找到加密文件进行解密
	 * @return return true if success, otherwise false
	 */
	public boolean defaultAesDecrypt(Context context,String explicitFilePath){
		String packageName = context.getPackageName();
		File explicitFile = new File(explicitFilePath);
		String explicitPath = explicitFile.getPath();
		String explicitName = explicitFile.getName();
		String digestName = new MessageDigestHelper().getHexStringDigest(explicitName);
		String encryptPath = explicitPath.replace(explicitName, digestName);
		File encryptFile = new File(encryptPath);
		if(explicitFile.exists() && (explicitFile.length()==encryptFile.length())){
			return true;
		}
		return randomKey?aesCipherRandom(Cipher.DECRYPT_MODE, encryptPath, explicitPath, packageName):aesCipherStable(Cipher.DECRYPT_MODE, encryptPath, explicitPath, packageName);
	}
	
	/**
	 * 默认使用系统包名做密钥对文件进行AES解密
	 * @param sourceFilePath
	 * @param targetFilePath
	 * @return return true if success, otherwise false
	 */
	public boolean defaultAesDecrypt(Context context, String sourceFilePath, String targetFilePath){
		String packageName = context.getPackageName();
		return randomKey?aesCipherRandom(Cipher.DECRYPT_MODE, sourceFilePath, targetFilePath, packageName):aesCipherStable(Cipher.DECRYPT_MODE, sourceFilePath, targetFilePath, packageName);
	}

	public boolean aesCipherStable(int cipherMode, String sourceFilePath, String targetFilePath, String seed) {
		byte[] rawkey = new byte[16];
		byte[] temp = seed.getBytes();
		if(temp.length >= rawkey.length){
			rawkey = Arrays.copyOf(temp, rawkey.length);
		}else{
			for(int i=0;i<rawkey.length;i++){
				rawkey[i] = temp[i%temp.length];
			}
		}
		return aesCipher(cipherMode, sourceFilePath, targetFilePath, rawkey);
	}
	
	public boolean aesCipherRandom(int cipherMode, String sourceFilePath, String targetFilePath, String seed) {
		try {
			byte[] rawkey = getRawKey(seed.getBytes());
			return aesCipher(cipherMode, sourceFilePath, targetFilePath, rawkey);
		} catch (Exception e) {
			return false;
		}
	}
	
	/**
	 * 对文件进行加密解密
	 * @param cipherMode 加密或解密<br/>Cipher.ENCRYPT_MODE：加密<br/>Cipher.DECRYPT_MODE：解密
	 * @param sourceFilePath 要进行操作(加密/解密)的文件路径
	 * @param targetFilePath 操作(加密/解密)之后保存的文件路径
	 * @param seed 加密或解密的 密钥
	 * @return true：操作成功  false：操作失败
	 */
	protected boolean aesCipher(int cipherMode, String sourceFilePath, String targetFilePath, byte[] rawkey) {
		boolean result = false;
		FileChannel sourceFC = null;
		FileChannel targetFC = null;

		try {
			if (cipherMode != Cipher.ENCRYPT_MODE&& cipherMode != Cipher.DECRYPT_MODE) {
				Log.d(TAG,"未指明是加密还是解密操作");
				return false;
			}

			Cipher mCipher = Cipher.getInstance("AES/CFB/NoPadding");

//			byte[] rawkey = getRawKey(seed.getBytes());
			
			File sourceFile = new File(sourceFilePath);
			File targetFile = new File(targetFilePath);

			sourceFC = new RandomAccessFile(sourceFile, "r").getChannel();
			targetFC = new RandomAccessFile(targetFile, "rw").getChannel();

			SecretKeySpec secretKey = new SecretKeySpec(rawkey, "AES");

			mCipher.init(cipherMode, secretKey, new IvParameterSpec(new byte[mCipher.getBlockSize()]));

			ByteBuffer byteData = ByteBuffer.allocate(1024);
			while (sourceFC.read(byteData) != -1) {
				// 通过通道读写交叉进行。
				// 将缓冲区准备为数据传出状态
				byteData.flip();

				byte[] byteList = new byte[byteData.remaining()];
				byteData.get(byteList, 0, byteList.length);
				//此处，若不使用数组加密解密会失败，因为当byteData达不到1024个时，加密方式不同对空白字节的处理也不相同，从而导致成功与失败。 
				byte[] bytes = mCipher.doFinal(byteList);
				targetFC.write(ByteBuffer.wrap(bytes));
				byteData.clear();
			}

			result = true;
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			try {
				if (sourceFC != null) {
					sourceFC.close();
				}
				if (targetFC != null) {
					targetFC.close();
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		return result;
	}
	
	/**
	 * 加密字符串
	 * @param seed 密钥
	 * @param source 要进行加密的字符串
	 */
	public String encrypt(String seed, String source) {
		byte[] result = null;
		try {
			byte[] rawkey = getRawKey(seed.getBytes());
			result = encrypt(rawkey, source.getBytes());
		} catch (Exception e) {
			e.printStackTrace();
		}
		String content = toHex(result);
		return content;

	}

	/**
	 * 使用指定的密钥
	 * @param rawkey 自定义密钥
	 * @param source 需要加密的字符串
	 * @return
	 */
	public String encrypt(byte[] rawkey, String source) {
		byte[] result = null;
		try {
			result = encrypt(rawkey, source.getBytes());
		} catch (Exception e) {
			e.printStackTrace();
		}
		String content = toHex(result);
		return content;
	}
	
	/**
	 * 从指定种子得到长度16的字节数组
	 * <br/> 如果seed.getBytes()长度大于16，取前16，否则循环添加
	 * @param seed 
	 * @return
	 */
	public byte[] getRawKey(String seed){
		byte[] rawkey = new byte[16];
		byte[] temp = seed.getBytes();
		if(temp.length >= rawkey.length){
			rawkey = Arrays.copyOf(temp, rawkey.length);
		}else{
			for(int i=0;i<rawkey.length;i++){
				rawkey[i] = temp[i%temp.length];
			}
		}
		return rawkey;
	}
	
	/**
	 * 解密字符串
	 * @param seed 密钥
	 * @param encrypted 要进行解密的字符串
	 */
	public String decrypt(byte[] rawKey, String encrypted) {
		try {
			byte[] enc = toByte(encrypted);
			byte[] result = decrypt(rawKey, enc);
			String coentn = new String(result);
			return coentn;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}
	
	/**
	 * 解密字符串
	 * @param seed 密钥
	 * @param encrypted 要进行解密的字符串
	 */
	public String decrypt(String seed, String encrypted) {
		byte[] rawKey;
		try {
			rawKey = getRawKey(seed.getBytes());
			byte[] enc = toByte(encrypted);
			byte[] result = decrypt(rawKey, enc);
			String coentn = new String(result);
			return coentn;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * 使用一个安全的随机数来产生一个密匙,密匙加密使用的
	 * 
	 * @param seed
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private byte[] getRawKey(byte[] seed) throws NoSuchAlgorithmException {
		// 获得一个随机数，传入的参数为默认方式。
		SecureRandom sr = null;
		try{
			if(android.os.Build.VERSION.SDK_INT >= 17){
				sr = SecureRandom.getInstance("SHA1PRNG", "Crypto");
			}else{
				sr = SecureRandom.getInstance("SHA1PRNG");
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		// 设置一个种子,一般是用户设定的密码
		sr.setSeed(seed);
		// 获得一个key生成器（AES加密模式）
		KeyGenerator keyGen = KeyGenerator.getInstance("AES");
		// 设置密匙长度128位
		keyGen.init(128, sr);
		// 获得密匙
		SecretKey key = keyGen.generateKey();
		// 返回密匙的byte数组供加解密使用
		byte[] raw = key.getEncoded();
		return raw;
	}

	/**
	 * 结合密钥生成加密后的密文
	 * @param raw
	 * @param input
	 * @return
	 * @throws Exception
	 */
	private byte[] encrypt(byte[] raw, byte[] input) throws Exception {
		// 根据上一步生成的密匙指定一个密匙
		SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
		// Cipher cipher = Cipher.getInstance("AES");
		// 加密算法，加密模式和填充方式三部分或指定加密算
		Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
		// 初始化模式为加密模式，并指定密匙
		cipher.init(Cipher.ENCRYPT_MODE, skeySpec, new IvParameterSpec(
				new byte[cipher.getBlockSize()]));
		byte[] encrypted = cipher.doFinal(input);
		return encrypted;
	}

	/**
	 * 根据密钥解密已经加密的数据
	 * @param raw
	 * @param encrypted
	 * @return
	 * @throws Exception
	 */
	private byte[] decrypt(byte[] raw, byte[] encrypted) throws Exception {
		SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
		Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding");
		cipher.init(Cipher.DECRYPT_MODE, skeySpec, new IvParameterSpec(
				new byte[cipher.getBlockSize()]));
		byte[] decrypted = cipher.doFinal(encrypted);
		return decrypted;
	}

	public String toHex(String txt) {
		return toHex(txt.getBytes());
	}

	public String fromHex(String hex) {
		return new String(toByte(hex));
	}

	public byte[] toByte(String hexString) {
		int len = hexString.length() / 2;
		byte[] result = new byte[len];
		for (int i = 0; i < len; i++)
			result[i] = Integer.valueOf(hexString.substring(2 * i, 2 * i + 2),
					16).byteValue();
		return result;
	}

	public String toHex(byte[] buf) {
		if (buf == null || buf.length <= 0)
			return "";
		StringBuffer result = new StringBuffer(2 * buf.length);
		for (int i = 0; i < buf.length; i++) {
			appendHex(result, buf[i]);
		}
		return result.toString();
	}

	private void appendHex(StringBuffer sb, byte b) {
		final String HEX = "0123456789ABCDEF";
		sb.append(HEX.charAt((b >> 4) & 0x0f)).append(HEX.charAt(b & 0x0f));
	}
	
	public void setRandomKey(boolean randomKey){
		this.randomKey = randomKey;
	}
	
	public boolean getRandomKey(){
		return this.randomKey;
	}
}