package com.szboanda.android.platform.db;

import android.content.Context;
import android.util.Log;

import com.szboanda.android.platform.util.ResourceUtils;
import com.szboanda.android.platform.util.ResourceUtils.ResoureType;

import org.xutils.DbManager;
import org.xutils.db.DbManagerImpl;
import org.xutils.db.table.TableEntity;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * <p>Description: 数据升级监听器</p>
 * @Company    深圳市博安达软件开发有限公司
 * @author    Siyi Lu    Create Date: 2015-4-22
 * @modify  
 */
public class SQLiteUpgradeListener implements DbManager.DbUpgradeListener{
	
	private static final String TAG = "SQLiteUpgradeListener";

	private String mDbName;
	
	private Context mContext;
	
	/**配置数据库升级时删除表的标志字符串*/
	private static final String SYMBOL_DROP_TABLE = "_upgrade_drop_table_";
	/**配置数据库升级时添加字段的标志字符串*/
	private static final String SYMBOL_ADD_COLUMN = "_upgrade_add_col_";
	/**配置数据库升级时执行语句的标志字符串*/
	private static final String SYMBOL_SQL = "_upgrade_sql_";
	
	private DbManager mDbManager;
	
	public SQLiteUpgradeListener(Context context,String dbName){
		mContext = context;
		mDbName = dbName;
	}

    @Override
    public void onUpgrade(DbManager db, int oldVersion, int newVersion) {
        upgradeDbByConfig(oldVersion, newVersion);
    }

	/**
	 * 通过配置文件升级数据库
	 * @param oldVersion
	 * @param newVersion
	 */
	public void upgradeDbByConfig(int oldVersion, int newVersion){
		int start = oldVersion+1;
		for(int i=start; i<=newVersion; i++){//版本跳跃时逐级升级数据库
			dropTable(i);
			addColumn(i);
			exeUpgradeSqls(i);
		}
	}
	
	/**
	 * 根据版本找到配置的增加表字段资源，并执行相关添加字段操作
	 * @param versionCode
	 */
	private void addColumn(int versionCode){
		String prefix = mDbName.replace(".db", "") + SYMBOL_ADD_COLUMN + versionCode + "_";
		List<Integer> ids = ResourceUtils.findResIdsByNamePrefix(mContext, prefix, ResoureType.ARRAY);
		String packName = mContext.getPackageName();
		for(Integer id:ids){
			String[] columns = mContext.getResources().getStringArray(id);
			String name = mContext.getResources().getResourceName(id);
			String tableName = name.replace(packName + ":array/", "").replace(prefix, "");
			if(!DbHelper.isTableExist(name)){
				continue;
			}
			for(String value:columns){
				String colName = value.toUpperCase(Locale.getDefault());
				String type = "TEXT";
				if(value.contains("#")){
					colName = value.split("#")[0];
					type = value.split("#")[1];
				}
				
				String sql = "ALTER TABLE "+ tableName +" ADD "+ colName + " " + type;
				try{
					mDbManager.execNonQuery(sql);
				}catch (Exception e) {
					Log.e(TAG, "Execute add column sql '" + sql + "' error" );
					e.printStackTrace();
				}
			}
		}
 	}
	
	private void exeUpgradeSqls(int versionCode){
		String prefix = mDbName.replace(".db", "") + SYMBOL_SQL + versionCode;
		int id = ResourceUtils.findResId(mContext, prefix, ResoureType.ARRAY);
		if(id != 0){
			String[] sqls = mContext.getResources().getStringArray(id);
            mDbManager.getDaoConfig().setAllowTransaction(true);
            for(String sql:sqls){
				try{
                    mDbManager.execNonQuery(sql);
				}catch (Exception e) {
					Log.e(TAG, "Execute upgrade sql '" + sql + "' error" );
					e.printStackTrace();
				}
			}
		}
	}
	
	/**
	 * 删除表
	 * @param versionCode
	 */
	private void dropTable(int versionCode){
		int configId = findDropTable(versionCode);
		if(configId != 0){
			String[] tables = mContext.getResources().getStringArray(configId);
			for(String name:tables){
				if(!DbHelper.isTableExist(name)){
					continue;
				}
				try{
					mDbManager.execNonQuery("DROP TABLE " + name);
                    Field tablesField = DbManagerImpl.class.getField("tableMap");
                    tablesField.setAccessible(true);
                    HashMap<Class<?>, TableEntity<?>> tablesCache = (HashMap)tablesField.get(mDbManager);
                    if(tablesCache != null) {
                        Set<Class<?>> tes = tablesCache.keySet();
                        for(Class<?> cls : tes){
                            TableEntity te = tablesCache.get(cls);
                            if(name.equalsIgnoreCase(te.getName())){
                                tablesCache.remove(cls);
                            }
                        }
                    }
				}catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	/**
	 * 根据版本号找到相应版本配置的需要删除的表的数组资源ID
	 * @param versionCode
	 * @return
	 */
	private int findDropTable(int versionCode){
		String name = mDbName.replace(".db", "") + SYMBOL_DROP_TABLE + versionCode;
		int dropTableArrayId = ResourceUtils.findResId(mContext, name, ResoureType.ARRAY);
		return dropTableArrayId;
	}

	
}
