package edsdk.utils;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;

import edsdk.bindings.EdSdkLibrary;


/**
 * A Lot of constants.
 * 
 * Copyright © 2014 Hansi Raber <super@superduper.org>, Ananta Palani
 * <anantapalani@gmail.com>
 * This work is free. You can redistribute it and/or modify it under the
 * terms of the Do What The Fuck You Want To Public License, Version 2,
 * as published by Sam Hocevar. See the COPYING file for more details.
 * 
 * @author hansi
 * @author Ananta Palani
 * 
 */
public class CanonConstants {

  private static final HashMap<String, DescriptiveEnum<?>> nameLUT = new HashMap<String, DescriptiveEnum<?>>();
  private static final HashMap<String, DescriptiveEnum<?>> descriptionLUT = new HashMap<String, DescriptiveEnum<?>>();
  private static final HashMap<String, HashSet<DescriptiveEnum<?>>> duplicateNameLUT = new HashMap<String, HashSet<DescriptiveEnum<?>>>();
  private static final HashMap<String, HashSet<DescriptiveEnum<?>>> duplicateDescriptionLUT = new HashMap<String, HashSet<DescriptiveEnum<?>>>();
  private static final HashMap<String, LinkedHashMap<String, DescriptiveEnum<?>>> tieredNameLUT = new HashMap<String, LinkedHashMap<String, DescriptiveEnum<?>>>();
  private static final HashMap<String, LinkedHashMap<Object, DescriptiveEnum<?>>> tieredValueLUT = new HashMap<String, LinkedHashMap<Object, DescriptiveEnum<?>>>();
  private static final HashMap<String, LinkedHashMap<String, DescriptiveEnum<?>>> tieredDescriptionLUT = new HashMap<String, LinkedHashMap<String, DescriptiveEnum<?>>>();
  private static final HashMap<String, HashMap<Object, HashSet<DescriptiveEnum<?>>>> tieredDuplicateValueLUT = new HashMap<String, HashMap<Object, HashSet<DescriptiveEnum<?>>>>();
  private static final HashMap<String, HashMap<String, HashSet<DescriptiveEnum<?>>>> tieredDuplicateDescriptionLUT = new HashMap<String, HashMap<String, HashSet<DescriptiveEnum<?>>>>();

  private static final HashSet<String> constantIgnoreList = new HashSet<String>(
      Arrays.asList(new String[] { "INSTANCE", "JNA_LIBRARY_NAME",
          "JNA_NATIVE_LIB", "NULL", "FALSE", "TRUE", "oldif" }));

  static {
    // The following pre-caches all enums to ensure that they will successfully
    // load values from EdSdkLibrary generated by JNAerator
    // System.out.println("Pre-caching enums");
    final Class<?>[] classes = CanonConstants.class.getClasses();

    for (final Class<?> klass : classes) {
      if (Enum.class.isAssignableFrom(klass)
          && DescriptiveEnum.class.isAssignableFrom(klass)) {

        final DescriptiveEnum<?>[] enums = (DescriptiveEnum<?>[]) klass
            .getEnumConstants();

        if (enums == null) {
          throw new IllegalStateException(
              "Class "
                  + klass.getCanonicalName()
                  + " was not initialized properly before before static initializer accessed, probably due to a dependency from a constructor accessing a static member or method of the outer class therefore requiring the static class to fully initialize first.");
        }

        for (final DescriptiveEnum<?> e : enums) {
          if (e == null) {
            throw new IllegalStateException(
                "Class "
                    + klass.getCanonicalName()
                    + " was not initialized properly before before static initializer accessed, probably due to a dependency from a constructor accessing a static member or method of the outer class therefore requiring the static class to fully initialize first.");
          }

          if (!CanonConstants.nameLUT.containsKey(e.name())) {
            // System.out.println( e.name() );
            CanonConstants.nameLUT.put(e.name(), e);
          } else {
            if (!CanonConstants.duplicateNameLUT.containsKey(e.name())) {
              CanonConstants.duplicateNameLUT.put(e.name(),
                  new HashSet<DescriptiveEnum<?>>());
            }
            CanonConstants.duplicateNameLUT.get(e.name()).add(e);
            // System.out.println(
            // "Warning: there are multiple enums named "+e.name()+". Looking up this enum by name may not work as expected."
            // );
          }

          if (!CanonConstants.descriptionLUT.containsKey(e.description())) {
            CanonConstants.descriptionLUT.put(e.description(), e);
          } else {
            if (!CanonConstants.duplicateDescriptionLUT.containsKey(e
                .description())) {
              CanonConstants.duplicateDescriptionLUT.put(e.description(),
                  new HashSet<DescriptiveEnum<?>>());
            }
            CanonConstants.duplicateDescriptionLUT.get(e.description()).add(e);
            // System.out.println(
            // "Warning: there are multiple enums with description "+e.description()+". Looking up this enum by description may not work as expected."
            // );
          }

          if (!CanonConstants.tieredNameLUT.containsKey(klass.getSimpleName())) {
            CanonConstants.tieredNameLUT.put(klass.getSimpleName(),
                new LinkedHashMap<String, DescriptiveEnum<?>>());
            CanonConstants.tieredValueLUT.put(klass.getSimpleName(),
                new LinkedHashMap<Object, DescriptiveEnum<?>>());
            CanonConstants.tieredDescriptionLUT.put(klass.getSimpleName(),
                new LinkedHashMap<String, DescriptiveEnum<?>>());
          }

          CanonConstants.tieredNameLUT.get(klass.getSimpleName()).put(e.name(),
              e);

          if (!CanonConstants.tieredValueLUT.get(klass.getSimpleName())
              .containsKey(e.value())) {
            CanonConstants.tieredValueLUT.get(klass.getSimpleName()).put(
                e.value(), e);
          } else {
            if (!CanonConstants.tieredDuplicateValueLUT.containsKey(klass
                .getSimpleName())) {
              CanonConstants.tieredDuplicateValueLUT.put(klass.getSimpleName(),
                  new HashMap<Object, HashSet<DescriptiveEnum<?>>>());
            }
            if (!CanonConstants.tieredDuplicateValueLUT.get(
                klass.getSimpleName()).containsKey(e.value())) {
              CanonConstants.tieredDuplicateValueLUT.get(klass.getSimpleName())
                  .put(e.value(), new HashSet<DescriptiveEnum<?>>());
            }
            CanonConstants.tieredDuplicateValueLUT.get(klass.getSimpleName())
                .get(e.value()).add(e);
            // System.out.println(
            // "Warning: there are multiple enums in "+klass.getCanonicalName()+" with value "+e.value()+". Looking up this enum by value may not work as expected."
            // );
          }

          if (!CanonConstants.tieredDescriptionLUT.get(klass.getSimpleName())
              .containsKey(e.description())) {
            CanonConstants.tieredDescriptionLUT.get(klass.getSimpleName()).put(
                e.description(), e);
          } else {
            if (!CanonConstants.tieredDuplicateDescriptionLUT.containsKey(klass
                .getSimpleName())) {
              CanonConstants.tieredDuplicateDescriptionLUT.put(
                  klass.getSimpleName(),
                  new HashMap<String, HashSet<DescriptiveEnum<?>>>());
            }
            if (!CanonConstants.tieredDuplicateDescriptionLUT.get(
                klass.getSimpleName()).containsKey(e.description())) {
              CanonConstants.tieredDuplicateDescriptionLUT.get(
                  klass.getSimpleName()).put(e.description(),
                  new HashSet<DescriptiveEnum<?>>());
            }
            CanonConstants.tieredDuplicateDescriptionLUT
                .get(klass.getSimpleName()).get(e.description()).add(e);
            // System.out.println(
            // "Warning: there are multiple enums in "+klass.getCanonicalName()+" with description "+e.description()+". Looking up this enum by description may not work as expected."
            // );
          }
        }
      }
    }
  }

  /**
   * Ensure that {@link edsdk.utils.CanonConstants CanonConstants} contains
   * all
   * constant values from {@link edsdk.bindings.EdSdkLibrary}
   * 
   * @return true if {@link edsdk.utils.CanonConstants CanonConstants} contains
   *         all values, bar those in
   *         {@link edsdk.utils.CanonConstants#constantIgnoreList
   *         constantIgnoreList}
   */
  public static final boolean verifyAllConstants() {
    boolean nothingMissing = true;

    System.out.println("Begin checking whether "
        + CanonConstants.class.getSimpleName()
        + " has all constants from EdSdkLibrary");

    Field[] fields = EdSdkLibrary.class.getDeclaredFields();
    for (final Field f : fields) {
      if (!CanonConstants.nameLUT.containsKey(f.getName())
          && !CanonConstants.constantIgnoreList.contains(f.getName())) {
        System.err.println(CanonConstants.class.getSimpleName()
            + " is Missing: " + f.getName() + " in "
            + EdSdkLibrary.class.getCanonicalName());
        nothingMissing = false;
      }
    }

    final Class<?>[] edsdkClasses = EdSdkLibrary.class.getClasses();
    for (final Class<?> klass : edsdkClasses) {
      fields = klass.getDeclaredFields();
      for (final Field f : fields) {
        if (!CanonConstants.nameLUT.containsKey(f.getName())
            && !CanonConstants.constantIgnoreList.contains(f.getName())) {
          System.err.println(CanonConstants.class.getSimpleName()
              + " is Missing: " + f.getName() + " in "
              + klass.getCanonicalName());
          nothingMissing = false;
        }
      }
    }

    System.out.println("Completed checking");

    return nothingMissing;
  }

  public static final DescriptiveEnum<?> enumOfName(final String name) {
    return CanonConstants.nameLUT.get(name);
  }

  public static final DescriptiveEnum<?> enumOfName(
      final Class<? extends DescriptiveEnum<?>> klass, final String name) {
    return CanonConstants.tieredNameLUT.get(klass.getSimpleName()).get(name);
  }

  public static final DescriptiveEnum<?> enumOfDescription(
      final Class<? extends DescriptiveEnum<?>> klass, final String description) {
    return CanonConstants.tieredDescriptionLUT.get(klass.getSimpleName()).get(
        description);
  }

  public static final DescriptiveEnum<?> enumOfValue(
      final Class<? extends DescriptiveEnum<?>> klass, final int value) {
    return CanonConstants.tieredValueLUT.get(klass.getSimpleName()).get(value);
  }

  public interface DescriptiveEnum<V> {

    public abstract String name();

    public abstract V value();

    public abstract String description();

  }

  /**
   * The following enum values are not defined by the EDSDK header
   * files, so values are taken from the EDSDK API documentation PDF and
   * compiled here for convenience. Interestingly, most of these are the
   * values (with
   * the exception of {@link edsdk.EdSdkLibrary#kEdsPropID_AEMode
   * kEdsPropID_AEMode} whose values are provided by the EDSDK) that can be
   * queried by {@link edsdk.EdSdkLibrary#EdsGetPropertyDesc
   * EdsGetPropertyDesc}
   */

  /**
   * Battery Quality
   * See: API Reference - 5.2.11 kEdsPropID_BatteryQuality
   */
  public enum EdsBatteryQuality implements DescriptiveEnum<Integer> {
    kEdsBatteryQuality_Low(0, "Very degraded"), kEdsBatteryQuality_Half(1,
        "Degraded"), kEdsBatteryQuality_HI(2, "Slight degradation"), kEdsBatteryQuality_Full(
        3, "No degradation");

    private final int value;
    private final String description;

    EdsBatteryQuality(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsBatteryQuality enumOfName(final String name) {
      return (EdsBatteryQuality) CanonConstants.enumOfName(
          EdsBatteryQuality.class, name);
    }

    public static final EdsBatteryQuality enumOfValue(final int value) {
      return (EdsBatteryQuality) CanonConstants.enumOfValue(
          EdsBatteryQuality.class, value);
    }

    public static final EdsBatteryQuality enumOfDescription(
        final String description) {
      return (EdsBatteryQuality) CanonConstants.enumOfDescription(
          EdsBatteryQuality.class, description);
    }
  }

  /**
   * Drive Mode Values
   * See: API Reference - 5.2.18 kEdsPropID_DriveMode
   */
  public enum EdsDriveMode implements DescriptiveEnum<Integer> {
    kEdsDriveMode_SingleFrame(0x00000000, "Single-Frame Shooting"), kEdsDriveMode_Continuous(
        0x00000001, "Continuous Shooting"), kEdsDriveMode_Video(0x00000002,
        "Video"), kEdsDriveMode_NotUsed(0x00000003, "Not used"), kEdsDriveMode_HighSpeedContinuous(
        0x00000004, "High-Speed Continuous Shooting"), kEdsDriveMode_LowSpeedContinuous(
        0x00000005, "Low-Speed Continuous Shooting"), kEdsDriveMode_SilentSingleFrame(
        0x00000006, "Silent single shooting"), kEdsDriveMode_10SecSelfTimerWithContinuous(
        0x00000007, "10-Sec Self-Timer + Continuous Shooting"), kEdsDriveMode_10SecSelfTimer(
        0x00000010, "10-Sec Self-Timer"), kEdsDriveMode_2SecSelfTimer(
        0x00000011, "2-Sec Self-Timer");

    private final int value;
    private final String description;

    EdsDriveMode(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsDriveMode enumOfName(final String name) {
      return (EdsDriveMode) CanonConstants.enumOfName(EdsDriveMode.class, name);
    }

    public static final EdsDriveMode enumOfValue(final int value) {
      return (EdsDriveMode) CanonConstants.enumOfValue(EdsDriveMode.class,
          value);
    }

    public static final EdsDriveMode enumOfDescription(final String description) {
      return (EdsDriveMode) CanonConstants.enumOfDescription(
          EdsDriveMode.class, description);
    }
  }

  /**
   * ISO Values
   * See: API Reference - 5.2.19 kEdsPropID_ISOSpeed
   * These values are taken from multiple EDSDK API Docs. Some values are
   * added are removed as support for cameras are added or removed
   */
  public enum EdsISOSpeed implements DescriptiveEnum<Integer> {
    kEdsISOSpeed_Auto(0x0, "Auto"), kEdsISOSpeed_6(0x28, "6"), kEdsISOSpeed_12(
        0x30, "12"), kEdsISOSpeed_25(0x38, "25"), kEdsISOSpeed_50(0x40, "50"), kEdsISOSpeed_100(
        0x48, "100"), kEdsISOSpeed_125(0x4B, "125"), kEdsISOSpeed_160(0x4D,
        "160"), kEdsISOSpeed_200(0x50, "200"), kEdsISOSpeed_250(0x53, "250"), kEdsISOSpeed_320(
        0x55, "320"), kEdsISOSpeed_400(0x58, "400"), kEdsISOSpeed_500(0x5B,
        "500"), kEdsISOSpeed_640(0x5D, "640"), kEdsISOSpeed_800(0x60, "800"), kEdsISOSpeed_1000(
        0x63, "1000"), kEdsISOSpeed_1250(0x65, "1250"), kEdsISOSpeed_1600(0x68,
        "1600"), kEdsISOSpeed_2000(0x6b, "2000"), kEdsISOSpeed_2500(0x6d,
        "2500"), kEdsISOSpeed_3200(0x70, "3200"), kEdsISOSpeed_4000(0x73,
        "4000"), kEdsISOSpeed_5000(0x75, "5000"), kEdsISOSpeed_6400(0x78,
        "6400"), kEdsISOSpeed_8000(0x7b, "8000"), kEdsISOSpeed_10000(0x7d,
        "10000"), kEdsISOSpeed_12800(0x80, "12800"), kEdsISOSpeed_25600(0x88,
        "25600"), kEdsISOSpeed_51200(0x90, "51200"), kEdsISOSpeed_102400(0x98,
        "102400"), kEdsISOSpeed_Unknown(0xFFFFFFFF,
        "Not valid/no settings changes");

    private final int value;
    private final String description;

    EdsISOSpeed(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsISOSpeed enumOfName(final String name) {
      return (EdsISOSpeed) CanonConstants.enumOfName(EdsISOSpeed.class, name);
    }

    public static final EdsISOSpeed enumOfValue(final int value) {
      return (EdsISOSpeed) CanonConstants.enumOfValue(EdsISOSpeed.class, value);
    }

    public static final EdsISOSpeed enumOfDescription(final String description) {
      return (EdsISOSpeed) CanonConstants.enumOfDescription(EdsISOSpeed.class,
          description);
    }
  }

  /**
   * Metering Mode Values
   * See: API Reference - 5.2.20 kEdsPropID_MeteringMode
   */
  public enum EdsMeteringMode implements DescriptiveEnum<Integer> {
    kEdsMeteringMode_Spot(1, "Spot metering"), kEdsMeteringMode_Evaluative(3,
        "Evaluative metering"), kEdsMeteringMode_Partial(4, "Partial metering"), kEdsMeteringMode_CenterWeightedAvg(
        5, "Center-weighted averaging metering"), kEdsMeteringMode_Unknown(
        0xFFFFFFFF, "Not valid/no settings changes");

    private final int value;
    private final String description;

    EdsMeteringMode(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsMeteringMode enumOfName(final String name) {
      return (EdsMeteringMode) CanonConstants.enumOfName(EdsMeteringMode.class,
          name);
    }

    public static final EdsMeteringMode enumOfValue(final int value) {
      return (EdsMeteringMode) CanonConstants.enumOfValue(
          EdsMeteringMode.class, value);
    }

    public static final EdsMeteringMode enumOfDescription(
        final String description) {
      return (EdsMeteringMode) CanonConstants.enumOfDescription(
          EdsMeteringMode.class, description);
    }
  }

  /**
   * Auto-Focus Mode Values
   * See: API Reference - 5.2.21 kEdsPropID_AFMode
   * Note: Read-only
   */
  public enum EdsAFMode implements DescriptiveEnum<Integer> {
    kEdsAFMode_OneShot(0, "One-Shot AF"), kEdsAFMode_AIServo(1, "AI Servo AF"), kEdsAFMode_AIFocus(
        2, "AI Focus AF"), kEdsAFMode_Manual(3, "Manual Focus"), kEdsAFMode_Unknown(
        0xFFFFFFFF, "Not valid/no settings changes");

    private final int value;
    private final String description;

    EdsAFMode(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsAFMode enumOfName(final String name) {
      return (EdsAFMode) CanonConstants.enumOfName(EdsAFMode.class, name);
    }

    public static final EdsAFMode enumOfValue(final int value) {
      return (EdsAFMode) CanonConstants.enumOfValue(EdsAFMode.class, value);
    }

    public static final EdsAFMode enumOfDescription(final String description) {
      return (EdsAFMode) CanonConstants.enumOfDescription(EdsAFMode.class,
          description);
    }
  }

  /**
   * Aperture Values
   * See: API Reference - 5.2.22 kEdsPropID_Av
   * Note: EdsAv names ending with 'b' represent aperture values when the
   * exposure step set in the Custom Function is 1/3 instead of 1/2.
   */
  public enum EdsAv implements DescriptiveEnum<Integer> {
    kEdsAv_1(0x08, "1"), kEdsAv_1_1(0x0B, "1.1"), kEdsAv_1_2(0x0C, "1.2"), kEdsAv_1_2b(
        0x0D, "1.2 (1/3)"), kEdsAv_1_4(0x10, "1.4"), kEdsAv_1_6(0x13, "1.6"), kEdsAv_1_8(
        0x14, "1.8"), kEdsAv_1_8b(0x15, "1.8 (1/3)"), kEdsAv_2(0x18, "2"), kEdsAv_2_2(
        0x1B, "2.2"), kEdsAv_2_5(0x1C, "2.5"), kEdsAv_2_5b(0x1D, "2.5 (1/3)"), kEdsAv_2_8(
        0x20, "2.8"), kEdsAv_3_2(0x23, "3.2"), kEdsAv_3_5(0x24, "3.5"), kEdsAv_3_5b(
        0x25, "3.5 (1/3)"), kEdsAv_4(0x28, "4"),
    // both 0x2B and 0x2C labeled '4.5' in EDSDK API docs, 1/3 value determined
    // by camera reading
    kEdsAv_4_5b(0x2B, "4.5 (1/3)"), kEdsAv_4_5(0x2C, "4.5"), kEdsAv_5_0(0x2D,
        "5.0"), kEdsAv_5_6(0x30, "5.6"), kEdsAv_6_3(0x33, "6.3"), kEdsAv_6_7(
        0x34, "6.7"), kEdsAv_7_1(0x35, "7.1"), kEdsAv_8(0x38, "8"), kEdsAv_9(
        0x3B, "9"), kEdsAv_9_5(0x3C, "9.5"), kEdsAv_10(0x3D, "10"), kEdsAv_11(
        0x40, "11"), kEdsAv_13b(0x43, "13 (1/3)"), kEdsAv_13(0x44, "13"), kEdsAv_14(
        0x45, "14"), kEdsAv_16(0x48, "16"), kEdsAv_18(0x4B, "18"), kEdsAv_19(
        0x4C, "19"), kEdsAv_20(0x4D, "20"), kEdsAv_22(0x50, "22"), kEdsAv_25(
        0x53, "25"), kEdsAv_27(0x54, "27"), kEdsAv_29(0x55, "29"), kEdsAv_32(
        0x58, "32"), kEdsAv_36(0x5B, "36"), kEdsAv_38(0x5C, "38"), kEdsAv_40(
        0x5D, "40"), kEdsAv_45(0x60, "45"), kEdsAv_51(0x63, "51"), kEdsAv_54(
        0x64, "54"), kEdsAv_57(0x65, "57"), kEdsAv_64(0x68, "64"), kEdsAv_72(
        0x6B, "72"), kEdsAv_76(0x6C, "76"), kEdsAv_80(0x6D, "80"), kEdsAv_91(
        0x70, "91"), kEdsAv_Unknown(0xFFFFFFFF, "Not valid/no settings changes");

    private final int value;
    private final String description;

    EdsAv(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsAv enumOfName(final String name) {
      return (EdsAv) CanonConstants.enumOfName(EdsAv.class, name);
    }

    public static final EdsAv enumOfValue(final int value) {
      return (EdsAv) CanonConstants.enumOfValue(EdsAv.class, value);
    }

    public static final EdsAv enumOfDescription(final String description) {
      return (EdsAv) CanonConstants.enumOfDescription(EdsAv.class, description);
    }
  }

  /**
   * Shutter Speed Values
   * See: API Reference - 5.2.23 kEdsPropID_Tv
   * Note: EdsTv names ending with 'b' represent shutter speeds when the
   * exposure step set in the Custom Function is 1/3 instead of 1/2.
   */
  public enum EdsTv implements DescriptiveEnum<Integer> {
    kEdsTv_BULB(0x0C, "BULB"), kEdsTv_30(0x10, "30\""), kEdsTv_25(0x13, "25\""), kEdsTv_20(
        0x14, "20\""), kEdsTv_20b(0x15, "20\" (1/3)"), kEdsTv_15(0x18, "15\""), kEdsTv_13(
        0x1B, "13\""), kEdsTv_10(0x1C, "10\""), kEdsTv_10b(0x1D, "10\" (1/3)"), kEdsTv_8(
        0x20, "8\""), kEdsTv_6b(0x23, "6\" (1/3)"), kEdsTv_6(0x24, "6\""), kEdsTv_5(
        0x25, "5\""), kEdsTv_4(0x28, "4\""), kEdsTv_3_2(0x2B, "3\"2"), kEdsTv_3(
        0x2C, "3\""), kEdsTv_2_5(0x2D, "2\"5"), kEdsTv_2(0x30, "2\""), kEdsTv_1_6(
        0x33, "1\"6"), kEdsTv_1_5(0x34, "1\"5"), kEdsTv_1_3(0x35, "1\"3"), kEdsTv_1(
        0x38, "1\""), kEdsTv_0_8(0x3B, "0\"8"), kEdsTv_0_7(0x3C, "0\"7"), kEdsTv_0_6(
        0x3D, "0\"6"), kEdsTv_0_5(0x40, "0\"5"), kEdsTv_0_4(0x43, "0\"4"), kEdsTv_0_3(
        0x44, "0\"3"), kEdsTv_0_3b(0x45, "0\"3 (1/3)"), kEdsTv_1by4(0x48, "1/4"), kEdsTv_1by5(
        0x4B, "1/5"), kEdsTv_1by6(0x4C, "1/6"), kEdsTv_1by6b(0x4D, "1/6 (1/3)"), kEdsTv_1by8(
        0x50, "1/8"), kEdsTv_1by10b(0x53, "1/10 (1/3)"), kEdsTv_1by10(0x54,
        "1/10"), kEdsTv_1by13(0x55, "1/13"), kEdsTv_1by15(0x58, "1/15"), kEdsTv_1by20b(
        0x5B, "1/20 (1/3)"), kEdsTv_1by20(0x5C, "1/20"), kEdsTv_1by25(0x5D,
        "1/25"), kEdsTv_1by30(0x60, "1/30"), kEdsTv_1by40(0x63, "1/40"), kEdsTv_1by45(
        0x64, "1/45"), kEdsTv_1by50(0x65, "1/50"), kEdsTv_1by60(0x68, "1/60"), kEdsTv_1by80(
        0x6B, "1/80"), kEdsTv_1by90(0x6C, "1/90"), kEdsTv_1by100(0x6D, "1/100"), kEdsTv_1by125(
        0x70, "1/125"), kEdsTv_1by160(0x73, "1/160"), kEdsTv_1by180(0x74,
        "1/180"), kEdsTv_1by200(0x75, "1/200"), kEdsTv_1by250(0x78, "1/250"), kEdsTv_1by320(
        0x7B, "1/320"), kEdsTv_1by350(0x7C, "1/350"), kEdsTv_1by400(0x7D,
        "1/400"), kEdsTv_1by500(0x80, "1/500"), kEdsTv_1by640(0x83, "1/640"), kEdsTv_1by750(
        0x84, "1/750"), kEdsTv_1by800(0x85, "1/800"), kEdsTv_1by1000(0x88,
        "1/1000"), kEdsTv_1by1250(0x8B, "1/1250"), kEdsTv_1by1500(0x8C,
        "1/1500"), kEdsTv_1by1600(0x8D, "1/1600"), kEdsTv_1by2000(0x90,
        "1/2000"), kEdsTv_1by2500(0x93, "1/2500"), kEdsTv_1by3000(0x94,
        "1/3000"), kEdsTv_1by3200(0x95, "1/3200"), kEdsTv_1by4000(0x98,
        "1/4000"), kEdsTv_1by5000(0x9B, "1/5000"), kEdsTv_1by6000(0x9C,
        "1/6000"), kEdsTv_1by6400(0x9D, "1/6400"), kEdsTv_1by8000(0xA0,
        "1/8000"), kEdsTv_Unknown(0xFFFFFFFF, "Not valid/no settings changes");

    private final int value;
    private final String description;

    EdsTv(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsTv enumOfName(final String name) {
      return (EdsTv) CanonConstants.enumOfName(EdsTv.class, name);
    }

    public static final EdsTv enumOfValue(final int value) {
      return (EdsTv) CanonConstants.enumOfValue(EdsTv.class, value);
    }

    public static final EdsTv enumOfDescription(final String description) {
      return (EdsTv) CanonConstants.enumOfDescription(EdsTv.class, description);
    }
  }

  /**
   * Exposure Compensation Values (Same as Flash Compensation Values)
   * See: API Reference - 5.2.24 kEdsPropID_ExposureCompensation
   * 5.2.26 kEdsPropID_FlashCompensation
   */
  public enum EdsExposureCompensation implements DescriptiveEnum<Integer> {
    /*
     * These extra values were available on a 550D camera and for some
     * reason are not described in the EDSDK documentation
     */
    kEdsExposureCompensation_5(0x28, "+5"), kEdsExposureCompensation_4_2by3(
        0x25, "+4 2/3"), kEdsExposureCompensation_4_1by2(0x24, "+4 1/2"), kEdsExposureCompensation_4_1by3(
        0x23, "+4 1/3"), kEdsExposureCompensation_4(0x20, "+4"), kEdsExposureCompensation_3_2by3(
        0x1D, "+3 2/3"), kEdsExposureCompensation_3_1by2(0x1C, "+3 1/2"), kEdsExposureCompensation_3_1by3(
        0x1B, "+3 1/3"),

    /* This group was defined in the EDSDK documentation */
    kEdsExposureCompensation_3(0x18, "+3"), kEdsExposureCompensation_2_2by3(
        0x15, "+2 2/3"), kEdsExposureCompensation_2_1by2(0x14, "+2 1/2"), kEdsExposureCompensation_2_1by3(
        0x13, "+2 1/3"), kEdsExposureCompensation_2(0x10, "+2"), kEdsExposureCompensation_1_2by3(
        0x0D, "+1 2/3"), kEdsExposureCompensation_1_1by2(0x0C, "+1 1/2"), kEdsExposureCompensation_1_1by3(
        0x0B, "+1 1/3"), kEdsExposureCompensation_1(0x08, "+1"), kEdsExposureCompensation_2by3(
        0x05, "+2/3"), kEdsExposureCompensation_1by2(0x04, "+1/2"), kEdsExposureCompensation_1by3(
        0x03, "+1/3"), kEdsExposureCompensation_0(0x00, "0"), kEdsExposureCompensation_n1by3(
        0xFD, "-1/3"), kEdsExposureCompensation_n1by2(0xFC, "-1/2"), kEdsExposureCompensation_n2by3(
        0xFB, "-2/3"), kEdsExposureCompensation_n1(0xF8, "-1"), kEdsExposureCompensation_n1_1by3(
        0xF5, "-1 1/3"), kEdsExposureCompensation_n1_1by2(0xF4, "-1 1/2"), kEdsExposureCompensation_n1_2by3(
        0xF3, "-1 2/3"), kEdsExposureCompensation_n2(0xF0, "-2"), kEdsExposureCompensation_n2_1by3(
        0xED, "-2 1/3"), kEdsExposureCompensation_n2_1by2(0xEC, "-2 1/2"), kEdsExposureCompensation_n2_2by3(
        0xEB, "-2 2/3"), kEdsExposureCompensation_n3(0xE8, "-3"),

    /*
     * These extra values were available on a 550D camera and for some
     * reason are not described in the EDSDK documentation
     */
    kEdsExposureCompensation_n3_1by3(0xE5, "-3 1/3"), kEdsExposureCompensation_n3_1by2(
        0xE4, "-3 1/2"), kEdsExposureCompensation_n3_2by3(0xE3, "-3 2/3"), kEdsExposureCompensation_n4(
        0xE0, "-4"), kEdsExposureCompensation_n4_1by3(0xDD, "-4 1/3"), kEdsExposureCompensation_n4_1by2(
        0xDC, "-4 1/2"), kEdsExposureCompensation_n4_2by3(0xDB, "-4 2/3"), kEdsExposureCompensation_n5(
        0xD8, "-5"),

    /* This was defined in the EDSDK documentation */
    kEdsExposureCompensation_Unknown(0xFFFFFFFF,
        "Not valid/no settings changes");

    private final int value;
    private final String description;

    EdsExposureCompensation(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsExposureCompensation enumOfName(final String name) {
      return (EdsExposureCompensation) CanonConstants.enumOfName(
          EdsExposureCompensation.class, name);
    }

    public static final EdsExposureCompensation enumOfValue(final int value) {
      return (EdsExposureCompensation) CanonConstants.enumOfValue(
          EdsExposureCompensation.class, value);
    }

    public static final EdsExposureCompensation enumOfDescription(
        final String description) {
      return (EdsExposureCompensation) CanonConstants.enumOfDescription(
          EdsExposureCompensation.class, description);
    }
  }

  /**
   * Live View Histogram Status
   * See: API Reference - 5.2.77 kEdsPropID_Evf_HistogramStatus
   */
  public enum EdsEvfHistogramStatus implements DescriptiveEnum<Integer> {
    kEdsEvfHistogramStatus_Hide(0, "Hide"), kEdsEvfHistogramStatus_Normal(0,
        "Display"), kEdsEvfHistogramStatus_Grayout(0, "Gray Out");

    private final int value;
    private final String description;

    EdsEvfHistogramStatus(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfHistogramStatus enumOfName(final String name) {
      return (EdsEvfHistogramStatus) CanonConstants.enumOfName(
          EdsEvfHistogramStatus.class, name);
    }

    public static final EdsEvfHistogramStatus enumOfValue(final int value) {
      return (EdsEvfHistogramStatus) CanonConstants.enumOfValue(
          EdsEvfHistogramStatus.class, value);
    }

    public static final EdsEvfHistogramStatus enumOfDescription(
        final String description) {
      return (EdsEvfHistogramStatus) CanonConstants.enumOfDescription(
          EdsEvfHistogramStatus.class, description);
    }
  }

  /**
   * Custom Function Values
   * Value passed as a 'param' to getPropertyData or setPropertyData
   * 
   * Data values given in comments are for Canon EOS 550D
   * 
   * 0x1xx = C.Fn I :Exposure
   * 0x2xx = C.Fn II :Image
   * 0x5xx = C.Fn III :Autofocus/Drive
   * 0x6xx = C.Fn III :Autofocus/Drive
   * 0x7xx = C.Fn IV :Operation/Others
   * 0x8xx = C.Fn IV :Operation/Others
   * 
   */
  public enum EdsCustomFunction implements DescriptiveEnum<Integer> {
    kEdsCustomFunction_ExposureIncrements(0x101, "Exposure level increments"), // 0:1/3-stop,
                                                                               // 1:1/2-stop
    kEdsCustomFunction_ISOExpansion(0x103, "ISO Expansion"), // 0:Off, 1:On
    kEdsCustomFunction_FlashSyncSpeed(0x10F, "Flash sync. speed in Av mode"), // 0:Auto,
                                                                              // 1:1/200-1/60sec.
                                                                              // auto,
                                                                              // 2:1/200sec.
                                                                              // (fixed)
    kEdsCustomFunction_LongExpNoiseReduction(0x201, "Long exp. noise reduction"), // 0:Off,
                                                                                  // 1:Auto,
                                                                                  // 2:On
    kEdsCustomFunction_HighISONoiseReduction(0x202,
        "High ISO speed noise reduct'n"), // 0:Standard, 1:Low, 2:Strong,
                                          // 3:Disable
    kEdsCustomFunction_HighlighTonePriority(0x203, "Highlight tone priority"), // 0:Disable,
                                                                               // 1:Enable
    kEdsCustomFunction_AFAssistBeam(0x50E, "AF-assist beam firing"), // 0:Enable,
                                                                     // 1:Disable,
                                                                     // 2:Enable
                                                                     // external
                                                                     // flash
                                                                     // only,
                                                                     // 3:IR AF
                                                                     // assist
                                                                     // beam
                                                                     // only
    kEdsCustomFunction_MirrorLockup(0x60F, "Mirror lockup"), // 0:Disable,
                                                             // 1:Enable
    kEdsCustomFunction_ShutterAELockButton(0x701, "Shutter/AE lock button"), // 0:AF/AE
                                                                             // lock,
                                                                             // 1:AE
                                                                             // lock/AF,
                                                                             // 2:AF/AF
                                                                             // lock,
                                                                             // no
                                                                             // AE
                                                                             // lock,
                                                                             // 3:AE/AF,
                                                                             // no
                                                                             // AE
                                                                             // lock
    kEdsCustomFunction_AssignSETButton(0x704, "Assign SET button"), // 0:Normal
                                                                    // (disabled),
                                                                    // 1:Image
                                                                    // quality,
                                                                    // 2:Flash
                                                                    // exposure
                                                                    // comp.,
                                                                    // 3:LCD
                                                                    // monitor
                                                                    // On/Off,
                                                                    // 4:Menu
                                                                    // display,
                                                                    // 5:ISO
                                                                    // speed
    kEdsCustomFunction_LCDDisplayWhenOn(0x80F, "LCD display when power ON"), // 0:Display
                                                                             // on,
                                                                             // 1:Previous
                                                                             // display
                                                                             // status
    kEdsCustomFunction_AddVerificationData(0x811, "Add image verification data"); // 0:Disable,
                                                                                  // 1:Enable

    private final int value;
    private final String description;

    EdsCustomFunction(final int value, final String description) {
      this.value = value;
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsCustomFunction enumOfName(final String name) {
      return (EdsCustomFunction) CanonConstants.enumOfName(
          EdsCustomFunction.class, name);
    }

    public static final EdsCustomFunction enumOfValue(final int value) {
      return (EdsCustomFunction) CanonConstants.enumOfValue(
          EdsCustomFunction.class, value);
    }

    public static final EdsCustomFunction enumOfDescription(
        final String description) {
      return (EdsCustomFunction) CanonConstants.enumOfDescription(
          EdsCustomFunction.class, description);
    }
  }

  /******************************************************************************
   * Definition of Constants
   ******************************************************************************/

  public enum EdsConstant implements DescriptiveEnum<Integer> {
    EDS_MAX_NAME("Maximum File Name Length"), EDS_TRANSFER_BLOCK_SIZE(
        "Transfer Block Size");

    private final int value;
    private final String description;

    EdsConstant(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsConstant enumOfName(final String name) {
      return (EdsConstant) CanonConstants.enumOfName(EdsConstant.class, name);
    }

    public static final EdsConstant enumOfValue(final int value) {
      return (EdsConstant) CanonConstants.enumOfValue(EdsConstant.class, value);
    }

    public static final EdsConstant enumOfDescription(final String description) {
      return (EdsConstant) CanonConstants.enumOfDescription(EdsConstant.class,
          description);
    }

  }

  /******************************************************************************
   * Definition of Error Codes
   ******************************************************************************/

  public enum EdsError implements DescriptiveEnum<Integer> {
    /*-----------------------------------------------------------------------
    ED-SDK Error Code Masks
     ------------------------------------------------------------------------*/
    EDS_ISSPECIFIC_MASK("IS Specific Mask"), EDS_COMPONENTID_MASK(
        "Component ID Mask"), EDS_RESERVED_MASK("Reserved Mask"), EDS_ERRORID_MASK(
        "Error ID Mask"),

    /*-----------------------------------------------------------------------
       ED-SDK Base Component IDs
    ------------------------------------------------------------------------*/
    EDS_CMP_ID_CLIENT_COMPONENTID("Component ID Client"), EDS_CMP_ID_LLSDK_COMPONENTID(
        "Component ID LLSDK"), EDS_CMP_ID_HLSDK_COMPONENTID(
        "Component ID HLSDK"),

    /*-----------------------------------------------------------------------
       ED-SDK Function Success Code
    ------------------------------------------------------------------------*/
    EDS_ERR_OK("Success, No Errors"),

    /*-----------------------------------------------------------------------
       ED-SDK Generic Error IDs
    ------------------------------------------------------------------------*/
    /* Miscellaneous errors */
    EDS_ERR_UNIMPLEMENTED("Unimplemented"), EDS_ERR_INTERNAL_ERROR(
        "Internal Error"), EDS_ERR_MEM_ALLOC_FAILED("Memory Allocation Failed"), EDS_ERR_MEM_FREE_FAILED(
        "Memory Release Failed"), EDS_ERR_OPERATION_CANCELLED(
        "Operation Cancelled"), EDS_ERR_INCOMPATIBLE_VERSION(
        "Incompatible Version"), EDS_ERR_NOT_SUPPORTED("Not Supported"), EDS_ERR_UNEXPECTED_EXCEPTION(
        "Unexpected Exception"), EDS_ERR_PROTECTION_VIOLATION(
        "Protection Violation"), EDS_ERR_MISSING_SUBCOMPONENT(
        "Missing Subcomponent"), EDS_ERR_SELECTION_UNAVAILABLE(
        "Selection Unavailable"),

    /* File errors */
    EDS_ERR_FILE_IO_ERROR("File I/O Error"), EDS_ERR_FILE_TOO_MANY_OPEN(
        "Too Many Files Open"), EDS_ERR_FILE_NOT_FOUND("File Not Found"), EDS_ERR_FILE_OPEN_ERROR(
        "File Open Error"), EDS_ERR_FILE_CLOSE_ERROR("File Close Error"), EDS_ERR_FILE_SEEK_ERROR(
        "File Seek Error"), EDS_ERR_FILE_TELL_ERROR("File Tell Error"), EDS_ERR_FILE_READ_ERROR(
        "File Read Error"), EDS_ERR_FILE_WRITE_ERROR("File Write Error"), EDS_ERR_FILE_PERMISSION_ERROR(
        "File Permission Error"), EDS_ERR_FILE_DISK_FULL_ERROR(
        "Disk Full Error"), EDS_ERR_FILE_ALREADY_EXISTS("File Already Exists"), EDS_ERR_FILE_FORMAT_UNRECOGNIZED(
        "File Format Not Recognized"), EDS_ERR_FILE_DATA_CORRUPT(
        "File Data Corrupt"), EDS_ERR_FILE_NAMING_NA("File Naming Error"),

    /* Directory errors */
    EDS_ERR_DIR_NOT_FOUND("Directory Does Not Exist"), EDS_ERR_DIR_IO_ERROR(
        "Directory I/O Error"), EDS_ERR_DIR_ENTRY_NOT_FOUND(
        "No Files In Directory"), EDS_ERR_DIR_ENTRY_EXISTS(
        "Directory Contains Files"), EDS_ERR_DIR_NOT_EMPTY("Directory Full"),

    /* Property errors */
    EDS_ERR_PROPERTIES_UNAVAILABLE("Property Unavailable"), EDS_ERR_PROPERTIES_MISMATCH(
        "Property Mismatch"), EDS_ERR_PROPERTIES_NOT_LOADED(
        "Property Not Loaded"),

    /* Function Parameter errors */
    EDS_ERR_INVALID_PARAMETER("Invalid Function Parameter"), EDS_ERR_INVALID_HANDLE(
        "Function Handle Error"), EDS_ERR_INVALID_POINTER(
        "Function Pointer Error"), EDS_ERR_INVALID_INDEX("Function Index Error"), EDS_ERR_INVALID_LENGTH(
        "Function Length Error"), EDS_ERR_INVALID_FN_POINTER(
        "Function FN Pointer Error"), EDS_ERR_INVALID_SORT_FN(
        "Function Sort FN Error"),

    /* Device errors */
    EDS_ERR_DEVICE_NOT_FOUND("Device Not Found"), EDS_ERR_DEVICE_BUSY(
        "Device Busy"), EDS_ERR_DEVICE_INVALID("Device Error"), EDS_ERR_DEVICE_EMERGENCY(
        "Device Emergency"), EDS_ERR_DEVICE_MEMORY_FULL("Device Memory Full"), EDS_ERR_DEVICE_INTERNAL_ERROR(
        "Internal Device Error"), EDS_ERR_DEVICE_INVALID_PARAMETER(
        "Invalid Device Parameter"), EDS_ERR_DEVICE_NO_DISK("No Device Disk"), EDS_ERR_DEVICE_DISK_ERROR(
        "Device Disk Error"), EDS_ERR_DEVICE_CF_GATE_CHANGED(
        "Device CF Gate Changed"), EDS_ERR_DEVICE_DIAL_CHANGED(
        "Device Dial Changed"), EDS_ERR_DEVICE_NOT_INSTALLED(
        "Device Not Installed"), EDS_ERR_DEVICE_STAY_AWAKE(
        "Device Connect In Awake Mode"), EDS_ERR_DEVICE_NOT_RELEASED(
        "Device Not Released"),

    /* Stream errors */
    EDS_ERR_STREAM_IO_ERROR("Stream I/O Error"), EDS_ERR_STREAM_NOT_OPEN(
        "Stream Not Open"), EDS_ERR_STREAM_ALREADY_OPEN("Stream Already Open"), EDS_ERR_STREAM_OPEN_ERROR(
        "Stream Open Error"), EDS_ERR_STREAM_CLOSE_ERROR("Stream Close Error"), EDS_ERR_STREAM_SEEK_ERROR(
        "Stream Seek Error"), EDS_ERR_STREAM_TELL_ERROR("Stream Tell Error"), EDS_ERR_STREAM_READ_ERROR(
        "Stream Read Error"), EDS_ERR_STREAM_WRITE_ERROR("Stream Write Error"), EDS_ERR_STREAM_PERMISSION_ERROR(
        "Stream Permission Error"), EDS_ERR_STREAM_COULDNT_BEGIN_THREAD(
        "Stream Could Not Start Reading Thumbnail"), EDS_ERR_STREAM_BAD_OPTIONS(
        "Invalid Stream Options"), EDS_ERR_STREAM_END_OF_STREAM(
        "Invalid Stream Termination"),

    /* Communications errors */
    EDS_ERR_COMM_PORT_IS_IN_USE("Communication Port In Use"), EDS_ERR_COMM_DISCONNECTED(
        "Communication Port Disconnected"), EDS_ERR_COMM_DEVICE_INCOMPATIBLE(
        "Communication Device Incompatible"), EDS_ERR_COMM_BUFFER_FULL(
        "Communication Buffer Full"), EDS_ERR_COMM_USB_BUS_ERR("USB Bus Error"),

    /* Lock/Unlock */
    EDS_ERR_USB_DEVICE_LOCK_ERROR("Failed To Lock The UI"), EDS_ERR_USB_DEVICE_UNLOCK_ERROR(
        "Failed To Unlock The UI"),

    /* STI/WIA */
    EDS_ERR_STI_UNKNOWN_ERROR("Unknown STI"), EDS_ERR_STI_INTERNAL_ERROR(
        "Internal STI Error"), EDS_ERR_STI_DEVICE_CREATE_ERROR(
        "STI Device Creation Error"), EDS_ERR_STI_DEVICE_RELEASE_ERROR(
        "STI Device Release Error"), EDS_ERR_DEVICE_NOT_LAUNCHED(
        "Device Start-up Failed"),

    EDS_ERR_ENUM_NA("Enumeration Terminated"), EDS_ERR_INVALID_FN_CALL(
        "Function Call Made In Incompatible Mode"), EDS_ERR_HANDLE_NOT_FOUND(
        "Handle Not Found"), EDS_ERR_INVALID_ID("Invalid ID"), EDS_ERR_WAIT_TIMEOUT_ERROR(
        "Timeout"),

    /* PTP */
    EDS_ERR_SESSION_NOT_OPEN("Session Open Error"), EDS_ERR_INVALID_TRANSACTIONID(
        "Invalid Transaction ID"), EDS_ERR_INCOMPLETE_TRANSFER(
        "Transfer Incomplete"), EDS_ERR_INVALID_STRAGEID("Invalid Storage ID"), EDS_ERR_DEVICEPROP_NOT_SUPPORTED(
        "Unsupported Device Property"), EDS_ERR_INVALID_OBJECTFORMATCODE(
        "Invalid Object Format Code"), EDS_ERR_SELF_TEST_FAILED(
        "Failed Self-Diagnosis"), EDS_ERR_PARTIAL_DELETION(
        "Partial Deletion Failed"), EDS_ERR_SPECIFICATION_BY_FORMAT_UNSUPPORTED(
        "Unsupported Format Specification"), EDS_ERR_NO_VALID_OBJECTINFO(
        "Invalid Object Information"), EDS_ERR_INVALID_CODE_FORMAT(
        "Invalid Code Format"), EDS_ERR_UNKNOWN_VENDOR_CODE(
        "Unknown Vendor Code"), EDS_ERR_CAPTURE_ALREADY_TERMINATED(
        "Capture Already Terminated"), EDS_ERR_INVALID_PARENTOBJECT(
        "Invalid Parent Object"), EDS_ERR_INVALID_DEVICEPROP_FORMAT(
        "Invalid Device Property Format"), EDS_ERR_INVALID_DEVICEPROP_VALUE(
        "Invalid Device Property Value"), EDS_ERR_SESSION_ALREADY_OPEN(
        "Session Already Open"), EDS_ERR_TRANSACTION_CANCELLED(
        "Transaction Cancelled"), EDS_ERR_SPECIFICATION_OF_DESTINATION_UNSUPPORTED(
        "Unsupported Destination Specification"),

    /* PTP Vendor */
    EDS_ERR_UNKNOWN_COMMAND("Unknown Command"), EDS_ERR_OPERATION_REFUSED(
        "Operation Refused"), EDS_ERR_LENS_COVER_CLOSE("Lens Cover Closed"), EDS_ERR_LOW_BATTERY(
        "Low Battery"), EDS_ERR_OBJECT_NOTREADY(
        "Live View Image Data Set Not Ready"), EDS_ERR_CANNOT_MAKE_OBJECT(
        "Cannot Make Object"),

    /* Take Picture errors */
    EDS_ERR_TAKE_PICTURE_AF_NG("Focus Failed"), EDS_ERR_TAKE_PICTURE_RESERVED(
        "Reserved"), EDS_ERR_TAKE_PICTURE_MIRROR_UP_NG(
        "Currently Configuring Mirror Up"), EDS_ERR_TAKE_PICTURE_SENSOR_CLEANING_NG(
        "Currently Cleaning Sensor"), EDS_ERR_TAKE_PICTURE_SILENCE_NG(
        "Currently Performing Silent Operations"), EDS_ERR_TAKE_PICTURE_NO_CARD_NG(
        "Card Not Installed"), EDS_ERR_TAKE_PICTURE_CARD_NG(
        "Error Writing To Card"), EDS_ERR_TAKE_PICTURE_CARD_PROTECT_NG(
        "Card Write Protected"), EDS_ERR_TAKE_PICTURE_MOVIE_CROP_NG(
        "Cropping Movie"), EDS_ERR_TAKE_PICTURE_STROBO_CHARGE_NG(
        "Strobe Charging"),

    EDS_ERR_LAST_GENERIC_ERROR_PLUS_ONE("Not Used");

    private final int value;
    private final String description;

    EdsError(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsError enumOfName(final String name) {
      return (EdsError) CanonConstants.enumOfName(EdsError.class, name);
    }

    public static final EdsError enumOfValue(final int value) {
      return (EdsError) CanonConstants.enumOfValue(EdsError.class, value);
    }

    public static final EdsError enumOfDescription(final String description) {
      return (EdsError) CanonConstants.enumOfDescription(EdsError.class,
          description);
    }

  }

  /******************************************************************************
   * Definition of Data Types
   ******************************************************************************/
  /*-----------------------------------------------------------------------------
   Data Types
   -----------------------------------------------------------------------------*/
  public enum EdsDataType implements DescriptiveEnum<Integer> {
    kEdsDataType_Unknown("Unknown"), kEdsDataType_Bool("EdsBool"), kEdsDataType_String(
        "EdsChar[]"), kEdsDataType_Int8("EdsInt8"), kEdsDataType_UInt8(
        "EdsUInt8"), kEdsDataType_Int16("EdsInt16"), kEdsDataType_UInt16(
        "EdsUInt16"), kEdsDataType_Int32("EdsInt32"), kEdsDataType_UInt32(
        "EdsUInt32"), kEdsDataType_Int64("EdsInt64"), kEdsDataType_UInt64(
        "EdsUInt64"), kEdsDataType_Float("EdsFloat"), kEdsDataType_Double(
        "EdsDouble"), kEdsDataType_ByteBlock("Byte Block"), // According to API,
                                                            // is either
                                                            // EdsInt8[] or
                                                            // EdsUInt32[], but
                                                            // perhaps former is
                                                            // a typo or an old
                                                            // value
    kEdsDataType_Rational("EdsRational"), kEdsDataType_Point("EdsPoint"), kEdsDataType_Rect(
        "EdsRect"), kEdsDataType_Time("EdsTime"),

    kEdsDataType_Bool_Array("EdsBool[]"), kEdsDataType_Int8_Array("EdsInt8[]"), kEdsDataType_Int16_Array(
        "EdsInt16[]"), kEdsDataType_Int32_Array("EdsInt32[]"), kEdsDataType_UInt8_Array(
        "EdsUInt8[]"), kEdsDataType_UInt16_Array("EdsUInt16[]"), kEdsDataType_UInt32_Array(
        "EdsUInt32[]"), kEdsDataType_Rational_Array("EdsRational[]"),

    kEdsDataType_FocusInfo("EdsFocusInfo"), kEdsDataType_PictureStyleDesc(
        "EdsPictureStyleDesc");

    private final int value;
    private final String description;

    EdsDataType(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsDataType.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsDataType enumOfName(final String name) {
      return (EdsDataType) CanonConstants.enumOfName(EdsDataType.class, name);
    }

    public static final EdsDataType enumOfValue(final int value) {
      return (EdsDataType) CanonConstants.enumOfValue(EdsDataType.class, value);
    }

    public static final EdsDataType enumOfDescription(final String description) {
      return (EdsDataType) CanonConstants.enumOfDescription(EdsDataType.class,
          description);
    }

  }

  /*-----------------------------------------------------------------------------
   Property IDs
   -----------------------------------------------------------------------------*/
  public enum EdsPropertyID implements DescriptiveEnum<Integer> {
    /*----------------------------------
    Camera Setting Properties
    ----------------------------------*/
    kEdsPropID_Unknown("Unknown", EdsDataType.kEdsDataType_Unknown),

    kEdsPropID_ProductName("Product Name", EdsDataType.kEdsDataType_String), kEdsPropID_OwnerName(
        "Owner Name", EdsDataType.kEdsDataType_String), kEdsPropID_MakerName(
        "Maker Name", EdsDataType.kEdsDataType_String), kEdsPropID_DateTime(
        "Date/Time", EdsDataType.kEdsDataType_Time), kEdsPropID_FirmwareVersion(
        "Firmware Version", EdsDataType.kEdsDataType_String), kEdsPropID_BatteryLevel(
        "Battery Level", EdsDataType.kEdsDataType_UInt32), kEdsPropID_CFn(
        "Custom Function #", EdsDataType.kEdsDataType_UInt32), // Not stated in
                                                               // the API, but
                                                               // all values
                                                               // seen on
                                                               // various
                                                               // cameras seem
                                                               // to be
                                                               // EdsUInt32
    kEdsPropID_SaveTo("Save To", EdsDataType.kEdsDataType_UInt32), kEdsPropID_CurrentStorage(
        "Current Storage", EdsDataType.kEdsDataType_String), kEdsPropID_CurrentFolder(
        "Current Folder", EdsDataType.kEdsDataType_String), kEdsPropID_MyMenu(
        "My Menu", EdsDataType.kEdsDataType_UInt32_Array),

    kEdsPropID_BatteryQuality("Battery Quality",
        EdsDataType.kEdsDataType_UInt32),

    kEdsPropID_BodyIDEx("Body ID Ex", EdsDataType.kEdsDataType_String), kEdsPropID_HDDirectoryStructure(
        "Hard Drive Directory Structure", EdsDataType.kEdsDataType_String),

    /*----------------------------------
    Image Properties
    ----------------------------------*/
    kEdsPropID_ImageQuality("Image Quality", EdsDataType.kEdsDataType_UInt32), kEdsPropID_JpegQuality(
        "JPEG Quality", EdsDataType.kEdsDataType_UInt32), kEdsPropID_Orientation(
        "Orientation", EdsDataType.kEdsDataType_UInt32), kEdsPropID_ICCProfile(
        "ICC Profile", EdsDataType.kEdsDataType_ByteBlock), // API lists
                                                            // EdsInt8[]
    kEdsPropID_FocusInfo("Focus Info", EdsDataType.kEdsDataType_FocusInfo), kEdsPropID_DigitalExposure(
        "Digital Exposure", EdsDataType.kEdsDataType_Rational), kEdsPropID_WhiteBalance(
        "White Balance", EdsDataType.kEdsDataType_Int32), kEdsPropID_ColorTemperature(
        "Color Temperature", EdsDataType.kEdsDataType_UInt32), kEdsPropID_WhiteBalanceShift(
        "White Balance Shift", EdsDataType.kEdsDataType_Int32_Array), kEdsPropID_Contrast(
        "Contrast", EdsDataType.kEdsDataType_Int32), kEdsPropID_ColorSaturation(
        "Color Saturation", EdsDataType.kEdsDataType_Int32), kEdsPropID_ColorTone(
        "Color Tone", EdsDataType.kEdsDataType_Int32), kEdsPropID_Sharpness(
        "Sharpness", EdsDataType.kEdsDataType_Int32), // kEdsDataType_Int32_Array
                                                      // for 1D/1Ds
    kEdsPropID_ColorSpace("Color Space", EdsDataType.kEdsDataType_UInt32), kEdsPropID_ToneCurve(
        "Tone Curve", EdsDataType.kEdsDataType_UInt32), kEdsPropID_PhotoEffect(
        "Photo Effect", EdsDataType.kEdsDataType_UInt32), kEdsPropID_FilterEffect(
        "Filter Effect", EdsDataType.kEdsDataType_UInt32), kEdsPropID_ToningEffect(
        "Toning Effect", EdsDataType.kEdsDataType_UInt32), kEdsPropID_ParameterSet(
        "Parameter Set", EdsDataType.kEdsDataType_UInt32), kEdsPropID_ColorMatrix(
        "Color Matrix", EdsDataType.kEdsDataType_UInt32), kEdsPropID_PictureStyle(
        "Picture Style", EdsDataType.kEdsDataType_UInt32), kEdsPropID_PictureStyleDesc(
        "Picture Style Description", EdsDataType.kEdsDataType_PictureStyleDesc), kEdsPropID_PictureStyleCaption(
        "Picture Style Caption", EdsDataType.kEdsDataType_String),

    /*----------------------------------
    Image Processing Properties
    ----------------------------------*/
    kEdsPropID_Linear("Linear Processing Status", EdsDataType.kEdsDataType_Bool), kEdsPropID_ClickWBPoint(
        "Coordinates for White Balance", EdsDataType.kEdsDataType_Point), kEdsPropID_WBCoeffs(
        "White Balance Values", EdsDataType.kEdsDataType_ByteBlock),

    /*----------------------------------
    Image GPS Properties
    ----------------------------------*/
    kEdsPropID_GPSVersionID("GPS Version ID", EdsDataType.kEdsDataType_UInt8), kEdsPropID_GPSLatitudeRef(
        "GPS N or S Latitude", EdsDataType.kEdsDataType_String), kEdsPropID_GPSLatitude(
        "GPS Latitude", EdsDataType.kEdsDataType_Rational_Array), kEdsPropID_GPSLongitudeRef(
        "GPS E or W Longitude", EdsDataType.kEdsDataType_String), kEdsPropID_GPSLongitude(
        "GPS Longitude", EdsDataType.kEdsDataType_Rational_Array), kEdsPropID_GPSAltitudeRef(
        "GPS Reference Altitude", EdsDataType.kEdsDataType_UInt8), kEdsPropID_GPSAltitude(
        "GPS Altitude", EdsDataType.kEdsDataType_Rational), kEdsPropID_GPSTimeStamp(
        "GPS Time Stamp", EdsDataType.kEdsDataType_Rational_Array), kEdsPropID_GPSSatellites(
        "GPS Satellites", EdsDataType.kEdsDataType_String), kEdsPropID_GPSStatus(
        "GPS Status", EdsDataType.kEdsDataType_String), kEdsPropID_GPSMapDatum(
        "GPS Geodetic Data", EdsDataType.kEdsDataType_String), kEdsPropID_GPSDateStamp(
        "GPS Date Stamp", EdsDataType.kEdsDataType_String),

    /*----------------------------------
    Property Mask
    ----------------------------------*/
    kEdsPropID_AtCapture_Flag("Get Properties at Time of Shooting",
        EdsDataType.kEdsDataType_UInt32),

    /*----------------------------------
    Capture Properties
    ----------------------------------*/
    /** Shooting Mode */
    kEdsPropID_AEMode("Shooting Mode", EdsDataType.kEdsDataType_UInt32),
    /** Drive Mode */
    kEdsPropID_DriveMode("Drive Mode", EdsDataType.kEdsDataType_UInt32),
    /** ISO Speed */
    kEdsPropID_ISOSpeed("ISO Speed", EdsDataType.kEdsDataType_UInt32),
    /** Metering Mode */
    kEdsPropID_MeteringMode("Metering Mode", EdsDataType.kEdsDataType_UInt32),
    /** Auto-Focus Mode */
    kEdsPropID_AFMode("Auto-Focus Mode", EdsDataType.kEdsDataType_UInt32),
    /** Aperture Value */
    kEdsPropID_Av("Aperture Value", EdsDataType.kEdsDataType_UInt32),
    /** Shutter Speed */
    kEdsPropID_Tv("Shutter Speed", EdsDataType.kEdsDataType_UInt32), // EdsImageRef
                                                                     // uses
                                                                     // EdsDataType.kEdsDataType_Rational
    /** Exposure Compensation */
    kEdsPropID_ExposureCompensation("Exposure Compensation",
        EdsDataType.kEdsDataType_UInt32), // EdsImageRef uses
                                          // EdsDataType.kEdsDataType_Rational
    /** Flash Compensation */
    kEdsPropID_FlashCompensation("Flash Compensation",
        EdsDataType.kEdsDataType_UInt32),
    /** Focal Length */
    kEdsPropID_FocalLength("Focal Length",
        EdsDataType.kEdsDataType_Rational_Array),
    /** Available Shots */
    kEdsPropID_AvailableShots("Available Shots",
        EdsDataType.kEdsDataType_UInt32),
    /** Bracket */
    kEdsPropID_Bracket("Bracket", EdsDataType.kEdsDataType_UInt32),
    /** White Balance Bracket */
    kEdsPropID_WhiteBalanceBracket("White Balance Bracket",
        EdsDataType.kEdsDataType_Int32_Array),
    /** Lens Name */
    kEdsPropID_LensName("Lens Name", EdsDataType.kEdsDataType_String),
    /** AE Bracket */
    kEdsPropID_AEBracket("AE Bracket", EdsDataType.kEdsDataType_Rational),
    /** FE Bracket */
    kEdsPropID_FEBracket("FE Bracket", EdsDataType.kEdsDataType_Rational),
    /** ISO Bracket */
    kEdsPropID_ISOBracket("ISO Bracket", EdsDataType.kEdsDataType_Rational),
    /** Noise Reduction */
    kEdsPropID_NoiseReduction("Noise Reduction",
        EdsDataType.kEdsDataType_UInt32),
    /** Flash Status */
    kEdsPropID_FlashOn("Flash Status", EdsDataType.kEdsDataType_UInt32),
    /** Red Eye Status */
    kEdsPropID_RedEye("Red Eye Status", EdsDataType.kEdsDataType_UInt32),
    /** Flash Mode */
    kEdsPropID_FlashMode("Flash Mode", EdsDataType.kEdsDataType_UInt32_Array),
    /** Lens Status */
    kEdsPropID_LensStatus("Lens Status", EdsDataType.kEdsDataType_UInt32),
    /** Artist */
    kEdsPropID_Artist("Artist", EdsDataType.kEdsDataType_String),
    /** Copyright */
    kEdsPropID_Copyright("Copyright", EdsDataType.kEdsDataType_String),
    /** Depth of Field */
    kEdsPropID_DepthOfField("Depth of Field", EdsDataType.kEdsDataType_UInt32),
    /** EF Compensation */
    kEdsPropID_EFCompensation("EF Compensation",
        EdsDataType.kEdsDataType_Unknown),
    /** Shooting Mode Select */
    kEdsPropID_AEModeSelect(0x00000436, "Shooting Mode Select",
        EdsDataType.kEdsDataType_UInt32), // in EDSDK 2.11.3 (or earlier),
                                          // removed EDSDK <= 2.13.2
    /** Movie Shooting Status */
    kEdsPropID_Record(0x00000510, "Movie Shooting Status",
        EdsDataType.kEdsDataType_UInt32), // in EDSDK 2.11.3 (or earlier),
                                          // removed EDSDK <= 2.13.2

    /*----------------------------------
    EVF Properties
    ----------------------------------*/
    kEdsPropID_Evf_OutputDevice("Live View Output Device",
        EdsDataType.kEdsDataType_UInt32), kEdsPropID_Evf_Mode("Live View Mode",
        EdsDataType.kEdsDataType_UInt32), kEdsPropID_Evf_WhiteBalance(
        "Live View White Balance", EdsDataType.kEdsDataType_UInt32), kEdsPropID_Evf_ColorTemperature(
        "Live View Color Temperature", EdsDataType.kEdsDataType_UInt32), kEdsPropID_Evf_DepthOfFieldPreview(
        "Live View Depth of Field in Preview", EdsDataType.kEdsDataType_UInt32),

    // EVF IMAGE DATA Properties
    kEdsPropID_Evf_Zoom("Live View Zoom Ratio", EdsDataType.kEdsDataType_UInt32), kEdsPropID_Evf_ZoomPosition(
        "Live View Zoom Position", EdsDataType.kEdsDataType_Point), kEdsPropID_Evf_FocusAid(
        "Live View Focus Aid", EdsDataType.kEdsDataType_Unknown), kEdsPropID_Evf_Histogram(
        "Live View Histogram", EdsDataType.kEdsDataType_ByteBlock), // API lists
                                                                    // EdsUInt32[]
    kEdsPropID_Evf_ImagePosition("Live View Crop Position",
        EdsDataType.kEdsDataType_Point), kEdsPropID_Evf_HistogramStatus(
        "Live View Histogram Status", EdsDataType.kEdsDataType_ByteBlock), // API
                                                                           // lists
                                                                           // type
                                                                           // as
                                                                           // kEdsDataType_UInt32,
                                                                           // but
                                                                           // camera
                                                                           // reports
                                                                           // EdsDataType.kEdsDataType_ByteBlock
    kEdsPropID_Evf_AFMode("Live View Auto-Focus Mode",
        EdsDataType.kEdsDataType_UInt32),

    kEdsPropID_Evf_HistogramY(0x00000515, "Live View Histogram Y",
        EdsDataType.kEdsDataType_ByteBlock), // in EDSDK 2.11.3 (or earlier),
                                             // removed EDSDK <= 2.13.2 // API
                                             // lists EdsUInt32[]
    kEdsPropID_Evf_HistogramR(0x00000516, "Live View Histogram R",
        EdsDataType.kEdsDataType_ByteBlock), // in EDSDK 2.11.3 (or earlier),
                                             // removed EDSDK <= 2.13.2 // API
                                             // lists EdsUInt32[]
    kEdsPropID_Evf_HistogramG(0x00000517, "Live View Histogram G",
        EdsDataType.kEdsDataType_ByteBlock), // in EDSDK 2.11.3 (or earlier),
                                             // removed EDSDK <= 2.13.2 // API
                                             // lists EdsUInt32[]
    kEdsPropID_Evf_HistogramB(0x00000518, "Live View Histogram B",
        EdsDataType.kEdsDataType_ByteBlock), // in EDSDK 2.11.3 (or earlier),
                                             // removed EDSDK <= 2.13.2 // API
                                             // lists EdsUInt32[]

    kEdsPropID_Evf_CoordinateSystem("Live View Coordinate System",
        EdsDataType.kEdsDataType_ByteBlock), // API lists conflicting info, says
                                             // 'Data type number =
                                             // kEdsDataType_Point', but 'Data
                                             // type = EdsSize'... but the
                                             // camera reports
                                             // kEdsDataType_ByteBlock
    kEdsPropID_Evf_ZoomRect("Live View Zoom Rectangle",
        EdsDataType.kEdsDataType_ByteBlock), // API lists conflicting info, says
                                             // 'Data type number =
                                             // kEdsDataType_Point', but 'Data
                                             // type = EdsRect'... but the
                                             // camera reports
                                             // kEdsDataType_ByteBlock
    kEdsPropID_Evf_ImageClipRect(0x00000545, "Live View Crop Rectangle",
        EdsDataType.kEdsDataType_ByteBlock); // in EDSDK 2.11.3 (or earlier),
                                             // removed EDSDK <= 2.13.2

    private final int value;
    private final String description;
    private final EdsDataType type;

    EdsPropertyID(final int value, final String description,
        final EdsDataType type) {
      this.value = value;
      this.description = description;
      this.type = type;
    }

    EdsPropertyID(final String description, final EdsDataType type) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
      this.type = type;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public final EdsDataType type() {
      return type;
    }

    public static final EdsPropertyID enumOfName(final String name) {
      return (EdsPropertyID) CanonConstants.enumOfName(EdsPropertyID.class,
          name);
    }

    public static final EdsPropertyID enumOfValue(final int value) {
      return (EdsPropertyID) CanonConstants.enumOfValue(EdsPropertyID.class,
          value);
    }

    public static final EdsPropertyID enumOfDescription(final String description) {
      return (EdsPropertyID) CanonConstants.enumOfDescription(
          EdsPropertyID.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Camera Commands
   -----------------------------------------------------------------------------*/

  /*----------------------------------
  Send Commands
  ----------------------------------*/

  // some EdsCameraCommand values are contained in EdsEvfAf and EdsShutterButton
  public enum EdsCameraCommand implements DescriptiveEnum<Integer> {
    kEdsCameraCommand_TakePicture("Take Picture"), kEdsCameraCommand_ExtendShutDownTimer(
        "Extend Auto-off Timer"), kEdsCameraCommand_BulbStart(
        "Start Bulb Shooting"), kEdsCameraCommand_BulbEnd("Stop Bulb Shooting"), kEdsCameraCommand_DoEvfAf(
        "Change Live View AF"), kEdsCameraCommand_DriveLensEvf(
        "Change Live View Focus"), kEdsCameraCommand_DoClickWBEvf(
        "Change Live View WB at Location"),

    kEdsCameraCommand_PressShutterButton("Change Shutter Button");

    private final int value;
    private final String description;

    EdsCameraCommand(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsCameraCommand enumOfName(final String name) {
      return (EdsCameraCommand) CanonConstants.enumOfName(
          EdsCameraCommand.class, name);
    }

    public static final EdsCameraCommand enumOfValue(final int value) {
      return (EdsCameraCommand) CanonConstants.enumOfValue(
          EdsCameraCommand.class, value);
    }

    public static final EdsCameraCommand enumOfDescription(
        final String description) {
      return (EdsCameraCommand) CanonConstants.enumOfDescription(
          EdsCameraCommand.class, description);
    }
  }

  public enum EdsEvfAf implements DescriptiveEnum<Integer> {
    kEdsCameraCommand_EvfAf_OFF("AF Off"), kEdsCameraCommand_EvfAf_ON("AF On");

    private final int value;
    private final String description;

    EdsEvfAf(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsEvfAf.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfAf enumOfName(final String name) {
      return (EdsEvfAf) CanonConstants.enumOfName(EdsEvfAf.class, name);
    }

    public static final EdsEvfAf enumOfValue(final int value) {
      return (EdsEvfAf) CanonConstants.enumOfValue(EdsEvfAf.class, value);
    }

    public static final EdsEvfAf enumOfDescription(final String description) {
      return (EdsEvfAf) CanonConstants.enumOfDescription(EdsEvfAf.class,
          description);
    }
  }

  public enum EdsShutterButton implements DescriptiveEnum<Integer> {
    kEdsCameraCommand_ShutterButton_OFF("Not Depressed"), kEdsCameraCommand_ShutterButton_Halfway(
        "Halfway Depressed"), kEdsCameraCommand_ShutterButton_Completely(
        "Fully Depressed"), kEdsCameraCommand_ShutterButton_Halfway_NonAF(
        "Halfway Depressed (Non-AF)"), kEdsCameraCommand_ShutterButton_Completely_NonAF(
        "Fully Depressed (Non-AF)");

    private final int value;
    private final String description;

    EdsShutterButton(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsShutterButton.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsShutterButton enumOfName(final String name) {
      return (EdsShutterButton) CanonConstants.enumOfName(
          EdsShutterButton.class, name);
    }

    public static final EdsShutterButton enumOfValue(final int value) {
      return (EdsShutterButton) CanonConstants.enumOfValue(
          EdsShutterButton.class, value);
    }

    public static final EdsShutterButton enumOfDescription(
        final String description) {
      return (EdsShutterButton) CanonConstants.enumOfDescription(
          EdsShutterButton.class, description);
    }
  }

  /*----------------------------------
   Camera Status Commands
   ----------------------------------*/
  public enum EdsCameraStatusCommand implements DescriptiveEnum<Integer> {
    kEdsCameraStatusCommand_UILock("UI Lock"), kEdsCameraStatusCommand_UIUnLock(
        "UI Unlock"), kEdsCameraStatusCommand_EnterDirectTransfer(
        "Enter Direct Transfer"), kEdsCameraStatusCommand_ExitDirectTransfer(
        "Exit Direct Transfer");

    private final int value;
    private final String description;

    EdsCameraStatusCommand(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsCameraStatusCommand enumOfName(final String name) {
      return (EdsCameraStatusCommand) CanonConstants.enumOfName(
          EdsCameraStatusCommand.class, name);
    }

    public static final EdsCameraStatusCommand enumOfValue(final int value) {
      return (EdsCameraStatusCommand) CanonConstants.enumOfValue(
          EdsCameraStatusCommand.class, value);
    }

    public static final EdsCameraStatusCommand enumOfDescription(
        final String description) {
      return (EdsCameraStatusCommand) CanonConstants.enumOfDescription(
          EdsCameraStatusCommand.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Camera Events
   -----------------------------------------------------------------------------*/

  /*----------------------------------
  Property Event
  ----------------------------------*/

  public enum EdsPropertyEvent implements DescriptiveEnum<Integer> {
    /* Notifies all property events. */
    kEdsPropertyEvent_All("Notify All"),

    /*
     * Notifies that a camera property value has been changed.
     * The changed property can be retrieved from event data.
     * The changed value can be retrieved by means of EdsGetPropertyData.
     * In the case of type 1 protocol standard cameras,
     * notification of changed properties can only be issued for custom
     * functions (CFn).
     * If the property type is "", the changed property cannot be
     * identified.
     * Thus, retrieve all required properties repeatedly.
     */
    kEdsPropertyEvent_PropertyChanged("Camera Property Changed"),

    /*
     * Notifies of changes in the list of camera properties with
     * configurable values.
     * The list of configurable values for property IDs indicated in event
     * data
     * can be retrieved by means of EdsGetPropertyDesc.
     * For type 1 protocol standard cameras, the property ID is identified
     * as "Unknown"
     * during notification.
     * Thus, you must retrieve a list of configurable values for all
     * properties and
     * retrieve the property values repeatedly.
     * (For details on properties for which you can retrieve a list of
     * configurable
     * properties, see the description of EdsGetPropertyDesc).
     */
    kEdsPropertyEvent_PropertyDescChanged("Details of Property Changed");

    private final int value;
    private final String description;

    EdsPropertyEvent(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsPropertyEvent enumOfName(final String name) {
      return (EdsPropertyEvent) CanonConstants.enumOfName(
          EdsPropertyEvent.class, name);
    }

    public static final EdsPropertyEvent enumOfValue(final int value) {
      return (EdsPropertyEvent) CanonConstants.enumOfValue(
          EdsPropertyEvent.class, value);
    }

    public static final EdsPropertyEvent enumOfDescription(
        final String description) {
      return (EdsPropertyEvent) CanonConstants.enumOfDescription(
          EdsPropertyEvent.class, description);
    }
  }

  /*----------------------------------
   Object Event
  ----------------------------------*/

  public enum EdsObjectEvent implements DescriptiveEnum<Integer> {
    /* Notifies all object events. */
    kEdsObjectEvent_All("Notify All"),

    /*
     * Notifies that the volume object (memory card) state (VolumeInfo)
     * has been changed.
     * Changed objects are indicated by event data.
     * The changed value can be retrieved by means of EdsGetVolumeInfo.
     * Notification of this event is not issued for type 1 protocol standard
     * cameras.
     */
    kEdsObjectEvent_VolumeInfoChanged("Memory Card Changed"),

    /*
     * Notifies if the designated volume on a camera has been formatted.
     * If notification of this event is received, get sub-items of the
     * designated
     * volume again as needed.
     * Changed volume objects can be retrieved from event data.
     * Objects cannot be identified on cameras earlier than the D30
     * if files are added or deleted.
     * Thus, these events are subject to notification.
     */
    kEdsObjectEvent_VolumeUpdateItems("Memory Card Formatted"),

    /*
     * Notifies if many images are deleted in a designated folder on a
     * camera.
     * If notification of this event is received, get sub-items of the
     * designated
     * folder again as needed.
     * Changed folders (specifically, directory item objects) can be
     * retrieved
     * from event data.
     */
    kEdsObjectEvent_FolderUpdateItems("Images Deleted"),

    /*
     * Notifies of the creation of objects such as new folders or files
     * on a camera compact flash card or the like.
     * This event is generated if the camera has been set to store captured
     * images simultaneously on the camera and a computer,
     * for example, but not if the camera is set to store images
     * on the computer alone.
     * Newly created objects are indicated by event data.
     * Because objects are not indicated for type 1 protocol standard
     * cameras,
     * (that is, objects are indicated as NULL),
     * you must again retrieve child objects under the camera object to
     * identify the new objects.
     */
    kEdsObjectEvent_DirItemCreated("Folders/Files Created"),

    /*
     * Notifies of the deletion of objects such as folders or files on a
     * camera
     * compact flash card or the like.
     * Deleted objects are indicated in event data.
     * Because objects are not indicated for type 1 protocol standard
     * cameras,
     * you must again retrieve child objects under the camera object to
     * identify deleted objects.
     */
    kEdsObjectEvent_DirItemRemoved("Folders/Files Deleted"),

    /*
     * Notifies that information of DirItem objects has been changed.
     * Changed objects are indicated by event data.
     * The changed value can be retrieved by means of
     * EdsGetDirectoryItemInfo.
     * Notification of this event is not issued for type 1 protocol standard
     * cameras.
     */
    kEdsObjectEvent_DirItemInfoChanged("Folders/Files Changed"),

    /*
     * Notifies that header information has been updated, as for rotation
     * information
     * of image files on the camera.
     * If this event is received, get the file header information again, as
     * needed.
     * This function is for type 2 protocol standard cameras only.
     */
    kEdsObjectEvent_DirItemContentChanged("Images Updated"),

    /*
     * Notifies that there are objects on a camera to be transferred to a
     * computer.
     * This event is generated after remote release from a computer or local
     * release
     * from a camera.
     * If this event is received, objects indicated in the event data must
     * be downloaded.
     * Furthermore, if the application does not require the objects, instead
     * of downloading them,
     * execute EdsDownloadCancel and release resources held by the camera.
     * The order of downloading from type 1 protocol standard cameras must
     * be the order
     * in which the events are received.
     */
    kEdsObjectEvent_DirItemRequestTransfer("Folders/Files Ready for Transfer"),

    /*
     * Notifies if the camera's direct transfer button is pressed.
     * If this event is received, objects indicated in the event data must
     * be downloaded.
     * Furthermore, if the application does not require the objects, instead
     * of
     * downloading them,
     * execute EdsDownloadCancel and release resources held by the camera.
     * Notification of this event is not issued for type 1 protocol standard
     * cameras.
     */
    kEdsObjectEvent_DirItemRequestTransferDT("Direct Transfer Pressed"),

    /*
     * Notifies of requests from a camera to cancel object transfer
     * if the button to cancel direct transfer is pressed on the camera.
     * If the parameter is 0, it means that cancellation of transfer is
     * requested for
     * objects still not downloaded,
     * with these objects indicated by
     * kEdsObjectEvent_DirItemRequestTransferDT.
     * Notification of this event is not issued for type 1 protocol standard
     * cameras.
     */
    kEdsObjectEvent_DirItemCancelTransferDT("Direct Tranfer Cancelled"),

    kEdsObjectEvent_VolumeAdded("Memory Card Added"), kEdsObjectEvent_VolumeRemoved(
        "Memory Card Removed");

    private final int value;
    private final String description;

    EdsObjectEvent(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsObjectEvent enumOfName(final String name) {
      return (EdsObjectEvent) CanonConstants.enumOfName(EdsObjectEvent.class,
          name);
    }

    public static final EdsObjectEvent enumOfValue(final int value) {
      return (EdsObjectEvent) CanonConstants.enumOfValue(EdsObjectEvent.class,
          value);
    }

    public static final EdsObjectEvent enumOfDescription(
        final String description) {
      return (EdsObjectEvent) CanonConstants.enumOfDescription(
          EdsObjectEvent.class, description);
    }
  }

  /*----------------------------------
   State Event
  ----------------------------------*/
  public enum EdsStateEvent implements DescriptiveEnum<Integer> {
    /* Notifies all state events. */
    kEdsStateEvent_All("Notify All"),

    /*
     * Indicates that a camera is no longer connected to a computer,
     * whether it was disconnected by unplugging a cord, opening
     * the compact flash compartment,
     * turning the camera off, auto shut-off, or by other means.
     */
    kEdsStateEvent_Shutdown("Camera Unavailable"),

    /*
     * Notifies of whether or not there are objects waiting to
     * be transferred to a host computer.
     * This is useful when ensuring all shot images have been transferred
     * when the application is closed.
     * Notification of this event is not issued for type 1 protocol
     * standard cameras.
     */
    kEdsStateEvent_JobStatusChanged("Job State Changed"),

    /*
     * Notifies that the camera will shut down after a specific period.
     * Generated only if auto shut-off is set.
     * Exactly when notification is issued (that is, the number of
     * seconds until shutdown) varies depending on the camera model.
     * To continue operation without having the camera shut down,
     * use EdsSendCommand to extend the auto shut-off timer.
     * The time in seconds until the camera shuts down is returned
     * as the initial value.
     */
    kEdsStateEvent_WillSoonShutDown("Camera Auto-off Active"),

    /*
     * As the counterpart event to kEdsStateEvent_WillSoonShutDown,
     * this event notifies of updates to the number of seconds until
     * a camera shuts down.
     * After the update, the period until shutdown is model-dependent.
     */
    kEdsStateEvent_ShutDownTimerUpdate("Seconds Until Auto-Off"),

    /*
     * Notifies that a requested release has failed, due to focus
     * failure or similar factors.
     */
    kEdsStateEvent_CaptureError("Remote Release Error"),

    /*
     * Notifies of internal SDK errors.
     * If this error event is received, the issuing device will probably
     * not be able to continue working properly,
     * so cancel the remote connection.
     */
    kEdsStateEvent_InternalError("SDK Software Error"),

    kEdsStateEvent_AfResult("AF Result"), kEdsStateEvent_BulbExposureTime(
        "Bulb Exposure Time");

    private final int value;
    private final String description;

    EdsStateEvent(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsStateEvent enumOfName(final String name) {
      return (EdsStateEvent) CanonConstants.enumOfName(EdsStateEvent.class,
          name);
    }

    public static final EdsStateEvent enumOfValue(final int value) {
      return (EdsStateEvent) CanonConstants.enumOfValue(EdsStateEvent.class,
          value);
    }

    public static final EdsStateEvent enumOfDescription(final String description) {
      return (EdsStateEvent) CanonConstants.enumOfDescription(
          EdsStateEvent.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Drive Lens
   -----------------------------------------------------------------------------*/
  public enum EdsEvfDriveLens implements DescriptiveEnum<Integer> {
    kEdsEvfDriveLens_Near1("Near 1"), kEdsEvfDriveLens_Near2("Near 2"), kEdsEvfDriveLens_Near3(
        "Near 3"), kEdsEvfDriveLens_Far1("Far 1"), kEdsEvfDriveLens_Far2(
        "Far 2"), kEdsEvfDriveLens_Far3("Far 3");

    private final int value;
    private final String description;

    EdsEvfDriveLens(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsEvfDriveLens.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfDriveLens enumOfName(final String name) {
      return (EdsEvfDriveLens) CanonConstants.enumOfName(EdsEvfDriveLens.class,
          name);
    }

    public static final EdsEvfDriveLens enumOfValue(final int value) {
      return (EdsEvfDriveLens) CanonConstants.enumOfValue(
          EdsEvfDriveLens.class, value);
    }

    public static final EdsEvfDriveLens enumOfDescription(
        final String description) {
      return (EdsEvfDriveLens) CanonConstants.enumOfDescription(
          EdsEvfDriveLens.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Depth of Field Preview
   -----------------------------------------------------------------------------*/
  public enum EdsEvfDepthOfFieldPreview implements DescriptiveEnum<Integer> {
    kEdsEvfDepthOfFieldPreview_OFF("Off"), kEdsEvfDepthOfFieldPreview_ON("On");

    private final int value;
    private final String description;

    EdsEvfDepthOfFieldPreview(final String description) {
      value = CanonUtils.classIntField(
          EdSdkLibrary.EdsEvfDepthOfFieldPreview.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfDepthOfFieldPreview enumOfName(final String name) {
      return (EdsEvfDepthOfFieldPreview) CanonConstants.enumOfName(
          EdsEvfDepthOfFieldPreview.class, name);
    }

    public static final EdsEvfDepthOfFieldPreview enumOfValue(final int value) {
      return (EdsEvfDepthOfFieldPreview) CanonConstants.enumOfValue(
          EdsEvfDepthOfFieldPreview.class, value);
    }

    public static final EdsEvfDepthOfFieldPreview enumOfDescription(
        final String description) {
      return (EdsEvfDepthOfFieldPreview) CanonConstants.enumOfDescription(
          EdsEvfDepthOfFieldPreview.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Stream Seek Origins
   -----------------------------------------------------------------------------*/
  public enum EdsSeekOrigin implements DescriptiveEnum<Integer> {
    kEdsSeek_Cur("Current"), kEdsSeek_Begin("Beginning"), kEdsSeek_End("Ending");

    private final int value;
    private final String description;

    EdsSeekOrigin(final String description) {
      value = CanonUtils
          .classIntField(EdSdkLibrary.EdsSeekOrigin.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsSeekOrigin enumOfName(final String name) {
      return (EdsSeekOrigin) CanonConstants.enumOfName(EdsSeekOrigin.class,
          name);
    }

    public static final EdsSeekOrigin enumOfValue(final int value) {
      return (EdsSeekOrigin) CanonConstants.enumOfValue(EdsSeekOrigin.class,
          value);
    }

    public static final EdsSeekOrigin enumOfDescription(final String description) {
      return (EdsSeekOrigin) CanonConstants.enumOfDescription(
          EdsSeekOrigin.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   File and Properties Access
   -----------------------------------------------------------------------------*/
  public enum EdsAccess implements DescriptiveEnum<Integer> {
    kEdsAccess_Read("Read-only"), kEdsAccess_Write("Write-only"), kEdsAccess_ReadWrite(
        "Read + Write"), kEdsAccess_Error("Error");

    private final int value;
    private final String description;

    EdsAccess(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsAccess.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsAccess enumOfName(final String name) {
      return (EdsAccess) CanonConstants.enumOfName(EdsAccess.class, name);
    }

    public static final EdsAccess enumOfValue(final int value) {
      return (EdsAccess) CanonConstants.enumOfValue(EdsAccess.class, value);
    }

    public static final EdsAccess enumOfDescription(final String description) {
      return (EdsAccess) CanonConstants.enumOfDescription(EdsAccess.class,
          description);
    }
  }

  /*-----------------------------------------------------------------------------
   File Create Disposition
   -----------------------------------------------------------------------------*/
  public enum EdsFileCreateDisposition implements DescriptiveEnum<Integer> {
    kEdsFileCreateDisposition_CreateNew("Create New"), kEdsFileCreateDisposition_CreateAlways(
        "Create New or Overwrite Existing"), kEdsFileCreateDisposition_OpenExisting(
        "Open Existing"), kEdsFileCreateDisposition_OpenAlways(
        "Open Existing or Create New"), kEdsFileCreateDisposition_TruncateExsisting(
        "Open and Erase Existing");

    private final int value;
    private final String description;

    EdsFileCreateDisposition(final String description) {
      value = CanonUtils.classIntField(
          EdSdkLibrary.EdsFileCreateDisposition.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsFileCreateDisposition enumOfName(final String name) {
      return (EdsFileCreateDisposition) CanonConstants.enumOfName(
          EdsFileCreateDisposition.class, name);
    }

    public static final EdsFileCreateDisposition enumOfValue(final int value) {
      return (EdsFileCreateDisposition) CanonConstants.enumOfValue(
          EdsFileCreateDisposition.class, value);
    }

    public static final EdsFileCreateDisposition enumOfDescription(
        final String description) {
      return (EdsFileCreateDisposition) CanonConstants.enumOfDescription(
          EdsFileCreateDisposition.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Image Types
   -----------------------------------------------------------------------------*/
  public enum EdsImageType implements DescriptiveEnum<Integer> {
    kEdsImageType_Unknown("Folder or Unknown"), kEdsImageType_Jpeg("JPEG"), kEdsImageType_CRW(
        "CRW"), kEdsImageType_RAW("RAW"), kEdsImageType_CR2("CR2");

    private final int value;
    private final String description;

    EdsImageType(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsImageType.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsImageType enumOfName(final String name) {
      return (EdsImageType) CanonConstants.enumOfName(EdsImageType.class, name);
    }

    public static final EdsImageType enumOfValue(final int value) {
      return (EdsImageType) CanonConstants.enumOfValue(EdsImageType.class,
          value);
    }

    public static final EdsImageType enumOfDescription(final String description) {
      return (EdsImageType) CanonConstants.enumOfDescription(
          EdsImageType.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Image Size
   -----------------------------------------------------------------------------*/
  public enum EdsImageSize implements DescriptiveEnum<Integer> {
    kEdsImageSize_Large("Large"), kEdsImageSize_Middle("Medium"), kEdsImageSize_Small(
        "Small"), kEdsImageSize_Middle1("Medium 1"), kEdsImageSize_Middle2(
        "Medium 2"), kEdsImageSize_Small1("Small 1"), kEdsImageSize_Small2(
        "Small 2"), kEdsImageSize_Small3("Small 3"), kEdsImageSize_Unknown(
        "Unknown");

    private final int value;
    private final String description;

    EdsImageSize(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsImageSize.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsImageSize enumOfName(final String name) {
      return (EdsImageSize) CanonConstants.enumOfName(EdsImageSize.class, name);
    }

    public static final EdsImageSize enumOfValue(final int value) {
      return (EdsImageSize) CanonConstants.enumOfValue(EdsImageSize.class,
          value);
    }

    public static final EdsImageSize enumOfDescription(final String description) {
      return (EdsImageSize) CanonConstants.enumOfDescription(
          EdsImageSize.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Image Compress Quality
   -----------------------------------------------------------------------------*/
  public enum EdsCompressQuality implements DescriptiveEnum<Integer> {
    kEdsCompressQuality_Normal("Normal"), kEdsCompressQuality_Fine("Fine"), kEdsCompressQuality_Lossless(
        "Lossless"), kEdsCompressQuality_SuperFine("Superfine"), kEdsCompressQuality_Unknown(
        "Unknown");

    private final int value;
    private final String description;

    EdsCompressQuality(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsCompressQuality.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsCompressQuality enumOfName(final String name) {
      return (EdsCompressQuality) CanonConstants.enumOfName(
          EdsCompressQuality.class, name);
    }

    public static final EdsCompressQuality enumOfValue(final int value) {
      return (EdsCompressQuality) CanonConstants.enumOfValue(
          EdsCompressQuality.class, value);
    }

    public static final EdsCompressQuality enumOfDescription(
        final String description) {
      return (EdsCompressQuality) CanonConstants.enumOfDescription(
          EdsCompressQuality.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Image Quality
   -----------------------------------------------------------------------------*/
  public enum EdsImageQuality implements DescriptiveEnum<Integer> {
    EdsImageQuality_LJ("JPEG Large"), EdsImageQuality_M1J("JPEG Medium 1"), EdsImageQuality_M2J(
        "JPEG Medium 2"), EdsImageQuality_SJ("JPEG Small"), EdsImageQuality_LJF(
        "JPEG Large Fine"), EdsImageQuality_LJN("JPEG Large Normal"), EdsImageQuality_MJF(
        "JPEG Medium Fine"), EdsImageQuality_MJN("JPEG Medium Normal"), EdsImageQuality_SJF(
        "JPEG Small Fine"), EdsImageQuality_SJN("JPEG Small Normal"), EdsImageQuality_S1JF(
        "JPEG Small 1 Fine"), EdsImageQuality_S1JN("JPEG Small 1 Normal"), EdsImageQuality_S2JF(
        "JPEG Small 2"), EdsImageQuality_S3JF("JPEG Small 3"),

    EdsImageQuality_LR("RAW"), EdsImageQuality_LRLJF("RAW + JPEG Large Fine"), EdsImageQuality_LRLJN(
        "RAW + JPEG Large Normal"), EdsImageQuality_LRMJF(
        "RAW + JPEG Middle Fine"), EdsImageQuality_LRMJN(
        "RAW + JPEG Middle Normal"), EdsImageQuality_LRSJF(
        "RAW + JPEG Small Fine"), EdsImageQuality_LRSJN(
        "RAW + JPEG Small Normal"), EdsImageQuality_LRS1JF(
        "RAW + JPEG Small 1 Fine"), EdsImageQuality_LRS1JN(
        "RAW + JPEG Small 1 Normal"), EdsImageQuality_LRS2JF(
        "RAW + JPEG Small 2"), EdsImageQuality_LRS3JF("RAW + JPEG Small 3"),

    EdsImageQuality_LRLJ("RAW + JPEG Large"), EdsImageQuality_LRM1J(
        "RAW + JPEG Middle 1"), EdsImageQuality_LRM2J("RAW + JPEG Middle 2"), EdsImageQuality_LRSJ(
        "RAW + JPEG Small"),

    EdsImageQuality_MR("MRAW (SRAW1)"), EdsImageQuality_MRLJF(
        "MRAW (SRAW1) + JPEG Large Fine"), EdsImageQuality_MRLJN(
        "MRAW (SRAW1) + JPEG Large Normal"), EdsImageQuality_MRMJF(
        "MRAW (SRAW1) + JPEG Medium Fine"), EdsImageQuality_MRMJN(
        "MRAW (SRAW1) + JPEG Medium Normal"), EdsImageQuality_MRSJF(
        "MRAW (SRAW1) + JPEG Small Fine"), EdsImageQuality_MRSJN(
        "MRAW (SRAW1) + JPEG Small Normal"), EdsImageQuality_MRS1JF(
        "MRAW (SRAW1) + JPEG Small 1 Fine"), EdsImageQuality_MRS1JN(
        "MRAW (SRAW1) + JPEG Small 1 Normal"), EdsImageQuality_MRS2JF(
        "MRAW (SRAW1) + JPEG Small 2"), EdsImageQuality_MRS3JF(
        "MRAW (SRAW1) + JPEG Small 3"),

    EdsImageQuality_MRLJ("MRAW (SRAW1) + JPEG Large"), EdsImageQuality_MRM1J(
        "MRAW (SRAW1) + JPEG Medium 1"), EdsImageQuality_MRM2J(
        "MRAW (SRAW1) + JPEG Medium 2"), EdsImageQuality_MRSJ(
        "MRAW (SRAW1) + JPEG Small"),

    EdsImageQuality_SR("SRAW (SRAW2)"), EdsImageQuality_SRLJF(
        "SRAW (SRAW2) + JPEG Large Fine"), EdsImageQuality_SRLJN(
        "SRAW (SRAW2) + JPEG Large Normal"), EdsImageQuality_SRMJF(
        "SRAW (SRAW2) + JPEG Middle Fine"), EdsImageQuality_SRMJN(
        "SRAW (SRAW2) + JPEG Middle Normal"), EdsImageQuality_SRSJF(
        "SRAW (SRAW2) + JPEG Small Fine"), EdsImageQuality_SRSJN(
        "SRAW (SRAW2) + JPEG Small Normal"), EdsImageQuality_SRS1JF(
        "SRAW (SRAW2) + JPEG Small1 Fine"), EdsImageQuality_SRS1JN(
        "SRAW (SRAW2) + JPEG Small1 Normal"), EdsImageQuality_SRS2JF(
        "SRAW (SRAW2) + JPEG Small2"), EdsImageQuality_SRS3JF(
        "SRAW (SRAW2) + JPEG Small3"),

    EdsImageQuality_SRLJ("SRAW (SRAW2) + JPEG Large"), EdsImageQuality_SRM1J(
        "SRAW (SRAW2) + JPEG Medium 1"), EdsImageQuality_SRM2J(
        "SRAW (SRAW2) + JPEG Medium 2"), EdsImageQuality_SRSJ(
        "SRAW (SRAW2) + JPEG Small"),

    EdsImageQuality_Unknown("Unknown");

    private final int value;
    private final String description;

    EdsImageQuality(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsImageQuality.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsImageQuality enumOfName(final String name) {
      return (EdsImageQuality) CanonConstants.enumOfName(EdsImageQuality.class,
          name);
    }

    public static final EdsImageQuality enumOfValue(final int value) {
      return (EdsImageQuality) CanonConstants.enumOfValue(
          EdsImageQuality.class, value);
    }

    public static final EdsImageQuality enumOfDescription(
        final String description) {
      return (EdsImageQuality) CanonConstants.enumOfDescription(
          EdsImageQuality.class, description);
    }
  }

  public enum EdsImageQualityForLegacy implements DescriptiveEnum<Integer> {
    kEdsImageQualityForLegacy_LJ("JPEG large"), kEdsImageQualityForLegacy_M1J(
        "JPEG medium 1"), kEdsImageQualityForLegacy_M2J("JPEG medium 2"), kEdsImageQualityForLegacy_SJ(
        "JPEG small"),

    kEdsImageQualityForLegacy_LJF("JPEG large fine"), kEdsImageQualityForLegacy_LJN(
        "JPEG large normal"), kEdsImageQualityForLegacy_MJF("JPEG medium fine"), kEdsImageQualityForLegacy_MJN(
        "JPEG medium normal"), kEdsImageQualityForLegacy_SJF("JPEG small fine"), kEdsImageQualityForLegacy_SJN(
        "JPEG small normal"),

    kEdsImageQualityForLegacy_LR("RAW"), kEdsImageQualityForLegacy_LRLJF(
        "RAW + JPEG large fine"), kEdsImageQualityForLegacy_LRLJN(
        "RAW + JPEG large normal"), kEdsImageQualityForLegacy_LRMJF(
        "RAW + JPEG medium fine"), kEdsImageQualityForLegacy_LRMJN(
        "RAW + JPEG medium normal"), kEdsImageQualityForLegacy_LRSJF(
        "RAW + JPEG small fine"), kEdsImageQualityForLegacy_LRSJN(
        "RAW + JPEG small normal"),

    kEdsImageQualityForLegacy_LR2("RAW "), kEdsImageQualityForLegacy_LR2LJ(
        "RAW + JPEG large"), kEdsImageQualityForLegacy_LR2M1J(
        "RAW + JPEG medium 1"), kEdsImageQualityForLegacy_LR2M2J(
        "RAW + JPEG medium 2"), kEdsImageQualityForLegacy_LR2SJ(
        "RAW + JPEG small"),

    kEdsImageQualityForLegacy_Unknown("Unknown");

    private final int value;
    private final String description;

    EdsImageQualityForLegacy(final String description) {
      value = CanonUtils.classIntField(
          EdSdkLibrary.EdsImageQualityForLegacy.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsImageQualityForLegacy enumOfName(final String name) {
      return (EdsImageQualityForLegacy) CanonConstants.enumOfName(
          EdsImageQualityForLegacy.class, name);
    }

    public static final EdsImageQualityForLegacy enumOfValue(final int value) {
      return (EdsImageQualityForLegacy) CanonConstants.enumOfValue(
          EdsImageQualityForLegacy.class, value);
    }

    public static final EdsImageQualityForLegacy enumOfDescription(
        final String description) {
      return (EdsImageQualityForLegacy) CanonConstants.enumOfDescription(
          EdsImageQualityForLegacy.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Image Source
   -----------------------------------------------------------------------------*/
  public enum EdsImageSource implements DescriptiveEnum<Integer> {
    kEdsImageSrc_FullView("Full-size"), kEdsImageSrc_Thumbnail("Thumbnail"), kEdsImageSrc_Preview(
        "Preview"), kEdsImageSrc_RAWThumbnail("RAW thumbnail"), kEdsImageSrc_RAWFullView(
        "RAW full-size");

    private final int value;
    private final String description;

    EdsImageSource(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsImageSource.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsImageSource enumOfName(final String name) {
      return (EdsImageSource) CanonConstants.enumOfName(EdsImageSource.class,
          name);
    }

    public static final EdsImageSource enumOfValue(final int value) {
      return (EdsImageSource) CanonConstants.enumOfValue(EdsImageSource.class,
          value);
    }

    public static final EdsImageSource enumOfDescription(
        final String description) {
      return (EdsImageSource) CanonConstants.enumOfDescription(
          EdsImageSource.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Target Image Types
   -----------------------------------------------------------------------------*/
  public enum EdsTargetImageType implements DescriptiveEnum<Integer> {
    kEdsTargetImageType_Unknown("Folder or unknown"), kEdsTargetImageType_Jpeg(
        "JPEG"), kEdsTargetImageType_TIFF("8-bit TIFF"), kEdsTargetImageType_TIFF16(
        "16-bit TIFF"), kEdsTargetImageType_RGB("8-bit RGB"), kEdsTargetImageType_RGB16(
        "16-bit RGB"), kEdsTargetImageType_DIB("DIB (BMP)");

    private final int value;
    private final String description;

    EdsTargetImageType(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsTargetImageType.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsTargetImageType enumOfName(final String name) {
      return (EdsTargetImageType) CanonConstants.enumOfName(
          EdsTargetImageType.class, name);
    }

    public static final EdsTargetImageType enumOfValue(final int value) {
      return (EdsTargetImageType) CanonConstants.enumOfValue(
          EdsTargetImageType.class, value);
    }

    public static final EdsTargetImageType enumOfDescription(
        final String description) {
      return (EdsTargetImageType) CanonConstants.enumOfDescription(
          EdsTargetImageType.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Progress Option
   -----------------------------------------------------------------------------*/
  public enum EdsProgressOption implements DescriptiveEnum<Integer> {
    kEdsProgressOption_NoReport("No callback"), kEdsProgressOption_Done(
        "Callback when done"), kEdsProgressOption_Periodically(
        "Periodic Callback");

    private final int value;
    private final String description;

    EdsProgressOption(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsProgressOption.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsProgressOption enumOfName(final String name) {
      return (EdsProgressOption) CanonConstants.enumOfName(
          EdsProgressOption.class, name);
    }

    public static final EdsProgressOption enumOfValue(final int value) {
      return (EdsProgressOption) CanonConstants.enumOfValue(
          EdsProgressOption.class, value);
    }

    public static final EdsProgressOption enumOfDescription(
        final String description) {
      return (EdsProgressOption) CanonConstants.enumOfDescription(
          EdsProgressOption.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   File attribute
   -----------------------------------------------------------------------------*/
  public enum EdsFileAttributes implements DescriptiveEnum<Integer> {
    kEdsFileAttribute_Normal("Normal"), kEdsFileAttribute_ReadOnly("Read-Only"), kEdsFileAttribute_Hidden(
        "Hidden"), kEdsFileAttribute_System("System"), kEdsFileAttribute_Archive(
        "Archive");

    private final int value;
    private final String description;

    EdsFileAttributes(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsFileAttributes.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsFileAttributes enumOfName(final String name) {
      return (EdsFileAttributes) CanonConstants.enumOfName(
          EdsFileAttributes.class, name);
    }

    public static final EdsFileAttributes enumOfValue(final int value) {
      return (EdsFileAttributes) CanonConstants.enumOfValue(
          EdsFileAttributes.class, value);
    }

    public static final EdsFileAttributes enumOfDescription(
        final String description) {
      return (EdsFileAttributes) CanonConstants.enumOfDescription(
          EdsFileAttributes.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Battery level
   -----------------------------------------------------------------------------*/
  // TODO: C++ enum values can have the same value, as in
  // EdSdkLibrary.EdsBatteryLevel2... how to handle reverse lookup in JAVA? Have
  // moved kEdsBatteryLevel2_Error as last identical value so that at least the
  // user doesn't think something is OK when it might not be
  public enum EdsBatteryLevel2 implements DescriptiveEnum<Integer> {
    kEdsBatteryLevel2_Empty("Empty"), kEdsBatteryLevel2_Low("Low"), kEdsBatteryLevel2_Half(
        "Half"), kEdsBatteryLevel2_Normal("Normal"), kEdsBatteryLevel2_Hi("Hi"), kEdsBatteryLevel2_Quarter(
        "Quarter"), kEdsBatteryLevel2_BCLevel("BC Level"), kEdsBatteryLevel2_Error(
        "Error"), kEdsBatteryLevel2_AC("AC power");

    private final int value;
    private final String description;

    EdsBatteryLevel2(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsBatteryLevel2.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsBatteryLevel2 enumOfName(final String name) {
      return (EdsBatteryLevel2) CanonConstants.enumOfName(
          EdsBatteryLevel2.class, name);
    }

    public static final EdsBatteryLevel2 enumOfValue(final int value) {
      return (EdsBatteryLevel2) CanonConstants.enumOfValue(
          EdsBatteryLevel2.class, value);
    }

    public static final EdsBatteryLevel2 enumOfDescription(
        final String description) {
      return (EdsBatteryLevel2) CanonConstants.enumOfDescription(
          EdsBatteryLevel2.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Save To
   -----------------------------------------------------------------------------*/
  public enum EdsSaveTo implements DescriptiveEnum<Integer> {
    kEdsSaveTo_Camera("Camera"), kEdsSaveTo_Host("Host Computer"), kEdsSaveTo_Both(
        "Both");

    private final int value;
    private final String description;

    EdsSaveTo(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsSaveTo.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsSaveTo enumOfName(final String name) {
      return (EdsSaveTo) CanonConstants.enumOfName(EdsSaveTo.class, name);
    }

    public static final EdsSaveTo enumOfValue(final int value) {
      return (EdsSaveTo) CanonConstants.enumOfValue(EdsSaveTo.class, value);
    }

    public static final EdsSaveTo enumOfDescription(final String description) {
      return (EdsSaveTo) CanonConstants.enumOfDescription(EdsSaveTo.class,
          description);
    }
  }

  /*-----------------------------------------------------------------------------
   StorageType
   -----------------------------------------------------------------------------*/
  public enum EdsStorageType implements DescriptiveEnum<Integer> {
    kEdsStorageType_Non("No memory card"), kEdsStorageType_CF("Compact Flash"), kEdsStorageType_SD(
        "SD Flash"), kEdsStorageType_HD("Hard Drive");

    private final int value;
    private final String description;

    EdsStorageType(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsStorageType.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsStorageType enumOfName(final String name) {
      return (EdsStorageType) CanonConstants.enumOfName(EdsStorageType.class,
          name);
    }

    public static final EdsStorageType enumOfValue(final int value) {
      return (EdsStorageType) CanonConstants.enumOfValue(EdsStorageType.class,
          value);
    }

    public static final EdsStorageType enumOfDescription(
        final String description) {
      return (EdsStorageType) CanonConstants.enumOfDescription(
          EdsStorageType.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   White Balance
   -----------------------------------------------------------------------------*/
  public enum EdsWhiteBalance implements DescriptiveEnum<Integer> {
    kEdsWhiteBalance_Auto("Auto"), kEdsWhiteBalance_Daylight("Daylight"), kEdsWhiteBalance_Cloudy(
        "Cloudy"), kEdsWhiteBalance_Tangsten("Tungsten"), kEdsWhiteBalance_Fluorescent(
        "Fluorescent"), kEdsWhiteBalance_Strobe("Flash"), kEdsWhiteBalance_WhitePaper(
        "Manual"), kEdsWhiteBalance_Shade("Shade"), kEdsWhiteBalance_ColorTemp(
        "Color temperature"), kEdsWhiteBalance_PCSet1("Custom: PC-1"), kEdsWhiteBalance_PCSet2(
        "Custom: PC-2"), kEdsWhiteBalance_PCSet3("Custom: PC-3"), kEdsWhiteBalance_WhitePaper2(
        "Manual 2"), kEdsWhiteBalance_WhitePaper3("Manual 3"), kEdsWhiteBalance_WhitePaper4(
        "Manual 4"), kEdsWhiteBalance_WhitePaper5("Manual 5"), kEdsWhiteBalance_PCSet4(
        "Custom: PC-4"), kEdsWhiteBalance_PCSet5("Custom: PC-5"), kEdsWhiteBalance_Click(
        "Click to set"), kEdsWhiteBalance_Pasted("Copied from image");

    private final int value;
    private final String description;

    EdsWhiteBalance(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsWhiteBalance.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsWhiteBalance enumOfName(final String name) {
      return (EdsWhiteBalance) CanonConstants.enumOfName(EdsWhiteBalance.class,
          name);
    }

    public static final EdsWhiteBalance enumOfValue(final int value) {
      return (EdsWhiteBalance) CanonConstants.enumOfValue(
          EdsWhiteBalance.class, value);
    }

    public static final EdsWhiteBalance enumOfDescription(
        final String description) {
      return (EdsWhiteBalance) CanonConstants.enumOfDescription(
          EdsWhiteBalance.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Photo Effects
   -----------------------------------------------------------------------------*/
  public enum EdsPhotoEffect implements DescriptiveEnum<Integer> {
    kEdsPhotoEffect_Off("Off"), kEdsPhotoEffect_Monochrome("Monochrome");

    private final int value;
    private final String description;

    EdsPhotoEffect(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsPhotoEffect.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsPhotoEffect enumOfName(final String name) {
      return (EdsPhotoEffect) CanonConstants.enumOfName(EdsPhotoEffect.class,
          name);
    }

    public static final EdsPhotoEffect enumOfValue(final int value) {
      return (EdsPhotoEffect) CanonConstants.enumOfValue(EdsPhotoEffect.class,
          value);
    }

    public static final EdsPhotoEffect enumOfDescription(
        final String description) {
      return (EdsPhotoEffect) CanonConstants.enumOfDescription(
          EdsPhotoEffect.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Color Matrix
   -----------------------------------------------------------------------------*/
  public enum EdsColorMatrix implements DescriptiveEnum<Integer> {
    kEdsColorMatrix_Custom("Custom"), kEdsColorMatrix_1("ColorMatrix1"), kEdsColorMatrix_2(
        "ColorMatrix2"), kEdsColorMatrix_3("ColorMatrix3"), kEdsColorMatrix_4(
        "ColorMatrix4"), kEdsColorMatrix_5("ColorMatrix5"), kEdsColorMatrix_6(
        "ColorMatrix6"), kEdsColorMatrix_7("ColorMatrix7");

    private final int value;
    private final String description;

    EdsColorMatrix(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsColorMatrix.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsColorMatrix enumOfName(final String name) {
      return (EdsColorMatrix) CanonConstants.enumOfName(EdsColorMatrix.class,
          name);
    }

    public static final EdsColorMatrix enumOfValue(final int value) {
      return (EdsColorMatrix) CanonConstants.enumOfValue(EdsColorMatrix.class,
          value);
    }

    public static final EdsColorMatrix enumOfDescription(
        final String description) {
      return (EdsColorMatrix) CanonConstants.enumOfDescription(
          EdsColorMatrix.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Filter Effects
   -----------------------------------------------------------------------------*/
  public enum EdsFilterEffect implements DescriptiveEnum<Integer> {
    kEdsFilterEffect_None("None"), kEdsFilterEffect_Yellow("Yellow"), kEdsFilterEffect_Orange(
        "Orange"), kEdsFilterEffect_Red("Red"), kEdsFilterEffect_Green("Green");

    private final int value;
    private final String description;

    EdsFilterEffect(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsFilterEffect.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsFilterEffect enumOfName(final String name) {
      return (EdsFilterEffect) CanonConstants.enumOfName(EdsFilterEffect.class,
          name);
    }

    public static final EdsFilterEffect enumOfValue(final int value) {
      return (EdsFilterEffect) CanonConstants.enumOfValue(
          EdsFilterEffect.class, value);
    }

    public static final EdsFilterEffect enumOfDescription(
        final String description) {
      return (EdsFilterEffect) CanonConstants.enumOfDescription(
          EdsFilterEffect.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Toning Effects
   -----------------------------------------------------------------------------*/
  public enum EdsTonigEffect implements DescriptiveEnum<Integer> {
    kEdsTonigEffect_None("None"), kEdsTonigEffect_Sepia("Sepia"), kEdsTonigEffect_Blue(
        "Blue"), kEdsTonigEffect_Purple("Purple"), kEdsTonigEffect_Green(
        "Green");

    private final int value;
    private final String description;

    EdsTonigEffect(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsTonigEffect.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsTonigEffect enumOfName(final String name) {
      return (EdsTonigEffect) CanonConstants.enumOfName(EdsTonigEffect.class,
          name);
    }

    public static final EdsTonigEffect enumOfValue(final int value) {
      return (EdsTonigEffect) CanonConstants.enumOfValue(EdsTonigEffect.class,
          value);
    }

    public static final EdsTonigEffect enumOfDescription(
        final String description) {
      return (EdsTonigEffect) CanonConstants.enumOfDescription(
          EdsTonigEffect.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Color Space
   -----------------------------------------------------------------------------*/
  public enum EdsColorSpace implements DescriptiveEnum<Integer> {
    kEdsColorSpace_sRGB("sRGB"), kEdsColorSpace_AdobeRGB("AdobeRGB"), kEdsColorSpace_Unknown(
        "Unknown");

    private final int value;
    private final String description;

    EdsColorSpace(final String description) {
      value = CanonUtils
          .classIntField(EdSdkLibrary.EdsColorSpace.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsColorSpace enumOfName(final String name) {
      return (EdsColorSpace) CanonConstants.enumOfName(EdsColorSpace.class,
          name);
    }

    public static final EdsColorSpace enumOfValue(final int value) {
      return (EdsColorSpace) CanonConstants.enumOfValue(EdsColorSpace.class,
          value);
    }

    public static final EdsColorSpace enumOfDescription(final String description) {
      return (EdsColorSpace) CanonConstants.enumOfDescription(
          EdsColorSpace.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   PictureStyle
   -----------------------------------------------------------------------------*/
  public enum EdsPictureStyle implements DescriptiveEnum<Integer> {
    kEdsPictureStyle_Standard("Standard"), kEdsPictureStyle_Portrait("Portrait"), kEdsPictureStyle_Landscape(
        "Landscape"), kEdsPictureStyle_Neutral("Neutral"), kEdsPictureStyle_Faithful(
        "Faithful"), kEdsPictureStyle_Monochrome("Monochrome"), kEdsPictureStyle_Auto(
        "Auto"), kEdsPictureStyle_User1("User 1"), kEdsPictureStyle_User2(
        "User 2"), kEdsPictureStyle_User3("User 3"), kEdsPictureStyle_PC1(
        "Computer 1"), kEdsPictureStyle_PC2("Computer 2"), kEdsPictureStyle_PC3(
        "Computer 3");

    private final int value;
    private final String description;

    EdsPictureStyle(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsPictureStyle.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsPictureStyle enumOfName(final String name) {
      return (EdsPictureStyle) CanonConstants.enumOfName(EdsPictureStyle.class,
          name);
    }

    public static final EdsPictureStyle enumOfValue(final int value) {
      return (EdsPictureStyle) CanonConstants.enumOfValue(
          EdsPictureStyle.class, value);
    }

    public static final EdsPictureStyle enumOfDescription(
        final String description) {
      return (EdsPictureStyle) CanonConstants.enumOfDescription(
          EdsPictureStyle.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Transfer Option
   -----------------------------------------------------------------------------*/
  public enum EdsTransferOption implements DescriptiveEnum<Integer> {
    kEdsTransferOption_ByDirectTransfer("By Direct Transfer"), kEdsTransferOption_ByRelease(
        "By Release"), kEdsTransferOption_ToDesktop("To Desktop");

    private final int value;
    private final String description;

    EdsTransferOption(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsTransferOption.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsTransferOption enumOfName(final String name) {
      return (EdsTransferOption) CanonConstants.enumOfName(
          EdsTransferOption.class, name);
    }

    public static final EdsTransferOption enumOfValue(final int value) {
      return (EdsTransferOption) CanonConstants.enumOfValue(
          EdsTransferOption.class, value);
    }

    public static final EdsTransferOption enumOfDescription(
        final String description) {
      return (EdsTransferOption) CanonConstants.enumOfDescription(
          EdsTransferOption.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Shooting Mode
   -----------------------------------------------------------------------------*/
  public enum EdsAEMode implements DescriptiveEnum<Integer> {
    kEdsAEMode_Program("Program AE"), kEdsAEMode_Tv("Shutter-Speed Priority AE"), kEdsAEMode_Av(
        "Aperture Priority AE"), kEdsAEMode_Manual("Manual Exposure"), kEdsAEMode_Bulb(
        "Bulb"), kEdsAEMode_A_DEP("Auto Depth-of-Field AE"), kEdsAEMode_DEP(
        "Depth-of-Field AE"), kEdsAEMode_Custom("Camera settings registered"), kEdsAEMode_Lock(
        "Lock"), kEdsAEMode_Green("Auto"), kEdsAEMode_NightPortrait(
        "Night Scene Portrait"), kEdsAEMode_Sports("Sports"), kEdsAEMode_Portrait(
        "Portrait"), kEdsAEMode_Landscape("Landscape"), kEdsAEMode_Closeup(
        "Close-Up"), kEdsAEMode_FlashOff("Flash Off"), kEdsAEMode_CreativeAuto(
        "Creative Auto"), kEdsAEMode_Movie("Movie"), kEdsAEMode_PhotoInMovie(
        "Photo In Movie"),
    // kEdsAEMode_SceneIntelligentAuto( "Scene Intelligent Auto" ), // in EDSDK
    // 2.11.3 (or earlier), removed EDSDK <= 2.13.2
    kEdsAEMode_Unknown("Unknown");

    private final int value;
    private final String description;

    EdsAEMode(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsAEMode.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsAEMode enumOfName(final String name) {
      return (EdsAEMode) CanonConstants.enumOfName(EdsAEMode.class, name);
    }

    public static final EdsAEMode enumOfValue(final int value) {
      return (EdsAEMode) CanonConstants.enumOfValue(EdsAEMode.class, value);
    }

    public static final EdsAEMode enumOfDescription(final String description) {
      return (EdsAEMode) CanonConstants.enumOfDescription(EdsAEMode.class,
          description);
    }
  }

  /*-----------------------------------------------------------------------------
   Bracket
   -----------------------------------------------------------------------------*/
  public enum EdsBracket implements DescriptiveEnum<Integer> {
    kEdsBracket_AEB("AE"), kEdsBracket_ISOB("ISO"), kEdsBracket_WBB("WB"), kEdsBracket_FEB(
        "FE"), kEdsBracket_Unknown("Off");

    private final int value;
    private final String description;

    EdsBracket(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsBracket.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsBracket enumOfName(final String name) {
      return (EdsBracket) CanonConstants.enumOfName(EdsBracket.class, name);
    }

    public static final EdsBracket enumOfValue(final int value) {
      return (EdsBracket) CanonConstants.enumOfValue(EdsBracket.class, value);
    }

    public static final EdsBracket enumOfDescription(final String description) {
      return (EdsBracket) CanonConstants.enumOfDescription(EdsBracket.class,
          description);
    }
  }

  /*-----------------------------------------------------------------------------
   EVF Output Device [Flag]
   -----------------------------------------------------------------------------*/
  public enum EdsEvfOutputDevice implements DescriptiveEnum<Integer> {
    kEdsEvfOutputDevice_TFT("Camera"), kEdsEvfOutputDevice_PC("Host Computer");

    private final int value;
    private final String description;

    EdsEvfOutputDevice(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsEvfOutputDevice.class,
          name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfOutputDevice enumOfName(final String name) {
      return (EdsEvfOutputDevice) CanonConstants.enumOfName(
          EdsEvfOutputDevice.class, name);
    }

    public static final EdsEvfOutputDevice enumOfValue(final int value) {
      return (EdsEvfOutputDevice) CanonConstants.enumOfValue(
          EdsEvfOutputDevice.class, value);
    }

    public static final EdsEvfOutputDevice enumOfDescription(
        final String description) {
      return (EdsEvfOutputDevice) CanonConstants.enumOfDescription(
          EdsEvfOutputDevice.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   EVF Zoom
   -----------------------------------------------------------------------------*/
  public enum EdsEvfZoom implements DescriptiveEnum<Integer> {
    kEdsEvfZoom_Fit("Fit Screen"), kEdsEvfZoom_x5("5 times"), kEdsEvfZoom_x10(
        "10 times");

    private final int value;
    private final String description;

    EdsEvfZoom(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsEvfZoom.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfZoom enumOfName(final String name) {
      return (EdsEvfZoom) CanonConstants.enumOfName(EdsEvfZoom.class, name);
    }

    public static final EdsEvfZoom enumOfValue(final int value) {
      return (EdsEvfZoom) CanonConstants.enumOfValue(EdsEvfZoom.class, value);
    }

    public static final EdsEvfZoom enumOfDescription(final String description) {
      return (EdsEvfZoom) CanonConstants.enumOfDescription(EdsEvfZoom.class,
          description);
    }
  }

  /*-----------------------------------------------------------------------------
   EVF AF Mode
   -----------------------------------------------------------------------------*/
  public enum EdsEvfAFMode implements DescriptiveEnum<Integer> {
    Evf_AFMode_Quick("Quick"), Evf_AFMode_Live("Live"), Evf_AFMode_LiveFace(
        "Live Face");

    private final int value;
    private final String description;

    EdsEvfAFMode(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsEvfAFMode.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsEvfAFMode enumOfName(final String name) {
      return (EdsEvfAFMode) CanonConstants.enumOfName(EdsEvfAFMode.class, name);
    }

    public static final EdsEvfAFMode enumOfValue(final int value) {
      return (EdsEvfAFMode) CanonConstants.enumOfValue(EdsEvfAFMode.class,
          value);
    }

    public static final EdsEvfAFMode enumOfDescription(final String description) {
      return (EdsEvfAFMode) CanonConstants.enumOfDescription(
          EdsEvfAFMode.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   Strobo Mode
   -----------------------------------------------------------------------------*/
  public enum EdsStroboMode implements DescriptiveEnum<Integer> {
    kEdsStroboModeInternal("Internal"), kEdsStroboModeExternalETTL(
        "External ETTL"), kEdsStroboModeExternalATTL("External ATTL"), kEdsStroboModeExternalTTL(
        "External TTL"), kEdsStroboModeExternalAuto("External Auto"), kEdsStroboModeExternalManual(
        "External Manual"), kEdsStroboModeManual("Manual");

    private final int value;
    private final String description;

    EdsStroboMode(final String description) {
      value = CanonUtils
          .classIntField(EdSdkLibrary.EdsStroboMode.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsStroboMode enumOfName(final String name) {
      return (EdsStroboMode) CanonConstants.enumOfName(EdsStroboMode.class,
          name);
    }

    public static final EdsStroboMode enumOfValue(final int value) {
      return (EdsStroboMode) CanonConstants.enumOfValue(EdsStroboMode.class,
          value);
    }

    public static final EdsStroboMode enumOfDescription(final String description) {
      return (EdsStroboMode) CanonConstants.enumOfDescription(
          EdsStroboMode.class, description);
    }
  }

  /*-----------------------------------------------------------------------------
   ETTL-II Mode
   -----------------------------------------------------------------------------*/
  public enum EdsETTL2Mode implements DescriptiveEnum<Integer> {
    kEdsETTL2ModeEvaluative("Evaluative"), kEdsETTL2ModeAverage("Average");

    private final int value;
    private final String description;

    EdsETTL2Mode(final String description) {
      value = CanonUtils.classIntField(EdSdkLibrary.EdsETTL2Mode.class, name());
      this.description = description;
    }

    @Override
    public final Integer value() {
      return value;
    }

    @Override
    public final String description() {
      return description;
    }

    public static final EdsETTL2Mode enumOfName(final String name) {
      return (EdsETTL2Mode) CanonConstants.enumOfName(EdsETTL2Mode.class, name);
    }

    public static final EdsETTL2Mode enumOfValue(final int value) {
      return (EdsETTL2Mode) CanonConstants.enumOfValue(EdsETTL2Mode.class,
          value);
    }

    public static final EdsETTL2Mode enumOfDescription(final String description) {
      return (EdsETTL2Mode) CanonConstants.enumOfDescription(
          EdsETTL2Mode.class, description);
    }
  }

}
