package com.beaconsinspace.android.beacon.detector;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.location.Location;
import android.os.BatteryManager;

import org.altbeacon.beacon.Beacon;

import java.util.Date;

/**
 * Created by kyleshank on 7/24/17.
 */

public class BISCollectionEvent {
    public enum BISCollectionEventType {
        SLC,
        Visit,
        BeaconInteraction,
        GeoFence,
        GPS
    }

    public enum BISCollectionEventDirection {
        None,
        Enter,
        Exit,
        Dwell
    }

    public Date date;

    public Location location;

    public Beacon beacon;
    public BISDetectorRSSICollector beaconRssiCollector;

    public BISCollectionEventType eventType;
    public BISCollectionEventDirection eventDirection;

    public BISDetector.ApplicationState applicationState = BISDetector.ApplicationState.BACKGROUND;
    public float batteryLevel;
    public String batteryState;

    public BISCollectionEvent(Context context, BISCollectionEventType eventType, BISDetector.ApplicationState applicationState) {
        this.date = new Date();
        this.eventType = eventType;
        this.applicationState = applicationState;

        powerCheck(context);
    }

    public BISCollectionEvent(Context context, BISCollectionEventType eventType, BISDetector.ApplicationState applicationState, Location location) {
        this.date = new Date();
        this.eventType = eventType;
        this.location = location;
        this.applicationState = applicationState;

        powerCheck(context);
    }

    public BISCollectionEvent(Context context, BISCollectionEventType eventType, BISDetector.ApplicationState applicationState, BISCollectionEventDirection eventDirection, Location location) {
        this.date = new Date();
        this.eventType = eventType;
        this.eventDirection = eventDirection;
        this.location = location;
        this.applicationState = applicationState;

        powerCheck(context);
    }

    public BISCollectionEvent(Context context, BISCollectionEventType eventType, BISDetector.ApplicationState applicationState, BISCollectionEventDirection eventDirection, Beacon beacon) {
        this.date = new Date();
        this.eventType = eventType;
        this.eventDirection = eventDirection;
        this.beacon = beacon;
        this.applicationState = applicationState;

        powerCheck(context);
    }

    public BISCollectionEvent(Context context, BISCollectionEventType eventType, BISDetector.ApplicationState applicationState, BISCollectionEventDirection eventDirection, Beacon beacon, BISDetectorRSSICollector beaconRssiCollector) {
        this.date = new Date();
        this.eventType = eventType;
        this.eventDirection = eventDirection;
        this.beacon = beacon;
        this.beaconRssiCollector = beaconRssiCollector;
        this.applicationState = applicationState;

        powerCheck(context);
    }

    private void powerCheck(Context context) {
        try {
            IntentFilter ifilter = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
            Intent batteryStatus = context.registerReceiver(null, ifilter);

            int status = batteryStatus.getIntExtra(BatteryManager.EXTRA_STATUS, -1);
            switch (status) {
                case BatteryManager.BATTERY_STATUS_CHARGING:
                    batteryState = "charging";
                    break;
                case BatteryManager.BATTERY_STATUS_NOT_CHARGING:
                    batteryState = "notcharging";
                    break;
                case BatteryManager.BATTERY_STATUS_DISCHARGING:
                    batteryState = "unplugged";
                    break;
                case BatteryManager.BATTERY_STATUS_FULL:
                    batteryState = "full";
                    break;
                default:
                    batteryState = "unknown";
            }

            int level = batteryStatus.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            int scale = batteryStatus.getIntExtra(BatteryManager.EXTRA_SCALE, -1);

            batteryLevel = (float) level / (float) scale;
        }catch(Exception e){
            batteryState = "unknown";
            batteryLevel = 0.0f;
        }
    }

    @Override
    public boolean equals(Object o) {
        boolean same = true;

        if (o.getClass().equals(BISCollectionEvent.class)) {
            BISCollectionEvent other = (BISCollectionEvent) o;
            if (other.date != this.date) {
                return false;
            }
            if (other.eventDirection != this.eventDirection) {
                return false;
            }
            if (other.eventType != this.eventType) {
                return false;
            }
            if (other.location != this.location) {
                return false;
            }
        }

        return same;
    }

    public String getDetect() {
        if (eventDirection != null) {
            switch (eventDirection) {
                case Enter:
                    return "enter";
                case Exit:
                    return "exit";
                case Dwell:
                    return "dwell";
            }
        }
        return "gps";
    }

    public String getEventType() {
        switch (eventType) {
            case SLC:
                return "slc";
            case Visit:
                return "visit";
            case BeaconInteraction:
                return "beaconinteraction";
            case GeoFence:
                return "geofence";
        }
        return "gps";
    }

    public String getUniqueId() {
        String uniqueId = getDetect() + "_" + getEventType();
        if (this.location != null) {
            uniqueId = uniqueId + "_" + this.location.getLatitude() + "," + this.location.getLongitude();
        }
        if (this.beacon != null) {
            uniqueId = uniqueId + "_";
            Integer numIdentifiers = this.beacon.getIdentifiers().size();
            if (numIdentifiers >= 1) {
                uniqueId = uniqueId + this.beacon.getId1() + "_";
            }
            if (numIdentifiers >= 2) {
                uniqueId = uniqueId + this.beacon.getId2() + "_";
            }
            if (numIdentifiers >= 3) {
                uniqueId = uniqueId + this.beacon.getId3();
            }
        }
        return uniqueId;
    }
}
