package com.beaconsinspace.android.beacon.detector;

import android.bluetooth.BluetoothAdapter;
import android.location.Location;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Base64;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

/**
 * Created by kyleshank on 9/28/17.
 */

class BISNetworking {
    private static final String TAG = "BIS_NETWORKING";

    static final String BIS_BASE_URL = "https://api.beaconsinspace.com";
    //    static final String BIS_BASE_URL = "http://api.staging.beaconsinspace.com";
    static final String BIS_BASE_ENDPOINT = "/v1/secure/initialize?uam=1";
    static final String BIS_URL_INITIALIZE = BIS_BASE_URL + BIS_BASE_ENDPOINT;

    @NonNull
    public static void getConfiguration(@NonNull BISDetector detector, String apiUrl, @NonNull BISNetworkingThreadListener<String> listener) {
        BISNetworkingGetThread urlCaller = new BISNetworkingGetThread(detector, apiUrl);
        urlCaller.setListener(listener);
        urlCaller.start();
    }

    @NonNull
    public static void getGeofences(@NonNull BISDetector detector, @NonNull String apiUrl, @NonNull Location location, @NonNull BISNetworkingThreadListener<String> listener) {
        Map<String, String> params = new HashMap<String, String>();
        params.put("latitude", Double.toString(location.getLatitude()));
        params.put("longitude", Double.toString(location.getLongitude()));
        params.put("gpsHorizontalAccuracy", Float.toString(location.getAccuracy()));
        params.put("gpsSpeed", Double.toString(location.getSpeed()));

        String apiUrlWithLocation = apiUrl + "?" +
                encodeParams(params);

        BISNetworkingGetThread urlCaller = new BISNetworkingGetThread(detector, apiUrlWithLocation);
        urlCaller.setListener(listener);
        urlCaller.start();
    }

    public static void reportSingleEvent(@NonNull BISDetector detector, @NonNull String apiUrl,
                                         @NonNull BISCollectionEvent event,
                                         @Nullable String bisData,
                                         @Nullable BISNetworkingThreadListener<BISCollectionEvent> listener) {
        BISNetworkingEventThread task = new BISNetworkingEventThread(detector, apiUrl, bisData, event);
        task.setListener(listener);
        task.start();
    }

    public static void reportBatchEvent(@NonNull BISDetector detector, @NonNull String apiUrl,
                                        @NonNull List<BISCollectionEvent> events,
                                        @Nullable String bisData,
                                        @Nullable BISNetworkingThreadListener<List<BISCollectionEvent>> listener) {
        BISLog.d(TAG, "reportBatchEvent(" + events.size() + ")");

        BISNetworkingEventsThread task = new BISNetworkingEventsThread(detector, apiUrl, bisData, events);
        task.setListener(listener);
        task.start();
    }

    public static String getAuthorizationHeader(BISDetector detector) {
        String USER_KEY = detector.getAPIKey();
        String PACKAGE_NAME = detector.getContext() != null ? detector.getContext().getPackageName() : "";
        String authString = USER_KEY + ":" + PACKAGE_NAME;
        byte[] authBytes = authString.getBytes();
        String base64AuthString = Base64.encodeToString(authBytes, Base64.DEFAULT, authString.length(), Base64.NO_WRAP);
        String authorizationHeader = "Basic " + base64AuthString;
        return authorizationHeader;
    }

    public static Map<String, String> getCommonRequestArguments(BISDetector detector) {
        Map<String, String> common = new HashMap<>();

        common.put("userId", ((detector.getUUID() != null && detector.getUUID().toString() != null) ? detector.getUUID().toString() : ""));
        common.put("userIdType", "AUUID");
        common.put("userId2", (detector.getADID() != null ? detector.getADID() : ""));
        common.put("userIdType2", "ADID");
        common.put("os", "ANDROID" + " " + Build.VERSION.RELEASE);
        common.put("device", Build.DEVICE);
        common.put("brand", Build.BRAND);
        common.put("model", Build.MODEL);
        common.put("manufacturer", Build.MANUFACTURER);
        common.put("userAgent", System.getProperty("http.agent"));
        common.put("tz", TimeZone.getDefault().getID());
        common.put("language", Locale.getDefault().toString());
        common.put("country", Locale.getDefault().toString());
        common.put("sdkVersion", BISDetector.SDK_VERSION);
        common.put("creationTimeInEpochMillis", String.valueOf(System.currentTimeMillis()));
        common.put("isBluetoothEnabled", isBluetoothEnabled() ? "1" : "0");

        if (detector.getPersistentStorage().isDeviceMetaDataCollected() && !detector.getPersistentStorage().isdeviceMetaDataSent()) {
            String deviceMetaDataString = detector.getPersistentStorage().getDeviceMetaDataString();
            if (deviceMetaDataString != null) {
                detector.getPersistentStorage().setDeviceMetaDataSent(true);
                common.put("deviceMetaData", deviceMetaDataString);
            }
        }

        return common;
    }

    public static Map<String, String> getLocationRequestArguments(Location location) {
        Map<String, String> request = new HashMap<>();

        if (location != null) {
            request.put("gpsLatitude", Double.toString(location.getLatitude()));
            request.put("gpsLongitude", Double.toString(location.getLongitude()));
            request.put("gpsHorizontalAccuracy", Float.toString(location.getAccuracy()));
            request.put("gpsAltitude", Double.toString(location.getAltitude()));
            request.put("gpsSpeed", Float.toString(location.getSpeed()));
            request.put("gpsBearing", Float.toString(location.getBearing()));
        }

        return request;
    }

    public static String encodeParams(Map<String, String> parameters) {
        List<String> encoded = new ArrayList<>();

        for (Map.Entry<String, String> entry : parameters.entrySet()) {
            try {
                if (entry.getValue() != null) {
                    encoded.add(entry.getKey() + "=" + URLEncoder.encode(entry.getValue(), "UTF-8"));
                }
            } catch (UnsupportedEncodingException e) {
                BISLog.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
            }
        }

        return TextUtils.join("&", encoded.toArray());
    }

    private static Boolean isBluetoothEnabled() {
        try {
            BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

            if (adapter == null) {
                return false;
            }
            return adapter.isEnabled();
        } catch (Throwable e) {
            return false;
        }
    }
}
