package com.beaconsinspace.android.beacon.detector;

import android.bluetooth.BluetoothAdapter;
import android.location.Location;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Base64;

import com.google.android.gms.location.Geofence;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;

/**
 * Created by kyleshank on 9/28/17.
 */

class BISNetworking {
    private static final String TAG = "BIS_NETWORKING";

    static final String BIS_BASE_URL = "https://api.beaconsinspace.com";
    static final String BIS_BASE_ENDPOINT = "/v1/secure/initialize?uam=1";
    static final String BIS_URL_INITIALIZE = BIS_BASE_URL + BIS_BASE_ENDPOINT;

    @NonNull
    public static JSONObject getConfiguration(@NonNull  BISDetector detector, String apiUrl)
            throws InterruptedException, ExecutionException, JSONException {
        BISNetworkingGetTask urlCaller = new BISNetworkingGetTask(detector);

        urlCaller.execute(apiUrl);

        String responseString = urlCaller.get();

        if (responseString == null) {
            throw new JSONException("response is null");
        }

        return new JSONObject(responseString);
    }

    @NonNull
    public static List<Geofence> getGeofences(@NonNull BISDetector detector,@NonNull String apiUrl, @NonNull Location location)
            throws ExecutionException, InterruptedException, JSONException {
        BISNetworkingGetTask urlCaller = new BISNetworkingGetTask(detector);

        Map<String, String> params = new HashMap<String, String>();
        params.put("latitude", Double.toString(location.getLatitude()));
        params.put("longitude", Double.toString(location.getLongitude()));
        params.put("gpsHorizontalAccuracy", Float.toString(location.getAccuracy()));
        params.put("gpsSpeed", Double.toString(location.getSpeed()));

        String apiUrlWithLocation = apiUrl + "?" +
                encodeParams(params);

        urlCaller.execute(apiUrlWithLocation);

        String responseString = urlCaller.get();

        if (responseString == null) {
            throw new JSONException("response is null");
        }

        JSONObject response = new JSONObject(responseString);
        JSONObject data = response.getJSONObject("data");
        JSONArray array = data.getJSONArray("geofence");

        List<Geofence> geofences = new ArrayList<Geofence>();

        for (int i = 0; i < array.length(); i++) {
            JSONObject geofenceJson = array.getJSONObject(i);
            double latitude = geofenceJson.getDouble("latitude");
            double longitude = geofenceJson.getDouble("longitude");
            float radius = (float) geofenceJson.getDouble("radius");

            Geofence geofence = new Geofence.Builder()
                    .setRequestId(latitude + "+" + longitude + "+" + radius)
                    .setCircularRegion(
                            latitude,
                            longitude,
                            radius
                    )
                    .setExpirationDuration(Geofence.NEVER_EXPIRE)
                    .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER |
                            Geofence.GEOFENCE_TRANSITION_DWELL |
                            Geofence.GEOFENCE_TRANSITION_EXIT)
                    .setLoiteringDelay(30000)
                    .build();

            geofences.add(geofence);
        }

        return geofences;
    }

    public static Boolean reportSingleEvent(@NonNull BISDetector detector, @NonNull String apiUrl,
                                     @NonNull BISCollectionEvent event,
                                     @Nullable String bisData) throws Exception {
        BISNetworkingEventTask task = new BISNetworkingEventTask(detector, apiUrl, bisData);

        task.execute(event);

        String result = task.get();

        return (result != null);
    }

    public static Boolean reportBatchEvent(@NonNull BISDetector detector, @NonNull String apiUrl,
                                    @NonNull List<BISCollectionEvent> events,
                                    @Nullable String bisData) throws Exception {
        if (events.isEmpty()) {
            return false;
        }

        BISLog.d(TAG, "reportBatchEvent(" + events.size() + ")");

        BISNetworkingEventsTask task = new BISNetworkingEventsTask(detector, apiUrl, bisData);

        task.execute(events);

        String result = task.get();

        return (result != null);
    }

    public static String getAuthorizationHeader(BISDetector detector) {
        String USER_KEY = detector.getAPIKey();
        String PACKAGE_NAME = detector.getContext() != null ? detector.getContext().getPackageName() : "";
        String authString = USER_KEY + ":" + PACKAGE_NAME;
        byte[] authBytes = authString.getBytes();
        String base64AuthString = Base64.encodeToString(authBytes, Base64.DEFAULT, authString.length(), Base64.NO_WRAP);
        String authorizationHeader = "Basic " + base64AuthString;
        return authorizationHeader;
    }

    public static Map<String, String> getCommonRequestArguments(BISDetector detector) {
        Map<String, String> common = new HashMap<>();

        common.put("userId", ((detector.getUUID() != null && detector.getUUID().toString() != null) ? detector.getUUID().toString() : ""));
        common.put("userIdType", "AUUID");
        common.put("userId2", (detector.getADID() != null ? detector.getADID() : ""));
        common.put("userIdType2", "ADID");
        common.put("os", "ANDROID" + " " + Build.VERSION.RELEASE);
        common.put("device", Build.DEVICE);
        common.put("brand", Build.BRAND);
        common.put("model", Build.MODEL);
        common.put("manufacturer", Build.MANUFACTURER);
        common.put("userAgent", System.getProperty("http.agent"));
        common.put("tz", TimeZone.getDefault().getID());
        common.put("language", Locale.getDefault().toString());
        common.put("country", Locale.getDefault().toString());
        common.put("sdkVersion", BISDetector.SDK_VERSION);
        common.put("creationTimeInEpochMillis", String.valueOf(System.currentTimeMillis()));
        common.put("isBluetoothEnabled", isBluetoothEnabled() ? "1" : "0");

        if (detector.getPersistentStorage().isDeviceMetaDataCollected() && !detector.getPersistentStorage().isdeviceMetaDataSent()) {
            String deviceMetaDataString = detector.getPersistentStorage().getDeviceMetaDataString();
            if (deviceMetaDataString != null) {
                detector.getPersistentStorage().setDeviceMetaDataSent(true);
                common.put("deviceMetaData", deviceMetaDataString);
            }
        }

        return common;
    }

    public static Map<String, String> getLocationRequestArguments(Location location) {
        Map<String, String> request = new HashMap<>();

        if (location != null) {
            request.put("gpsLatitude", Double.toString(location.getLatitude()));
            request.put("gpsLongitude", Double.toString(location.getLongitude()));
            request.put("gpsHorizontalAccuracy", Float.toString(location.getAccuracy()));
            request.put("gpsAltitude", Double.toString(location.getAltitude()));
            request.put("gpsSpeed", Float.toString(location.getSpeed()));
            request.put("gpsBearing", Float.toString(location.getBearing()));
        }

        return request;
    }

    public static String encodeParams(Map<String, String> parameters) {
        List<String> encoded = new ArrayList<>();

        for (Map.Entry<String, String> entry : parameters.entrySet()) {
            try {
                if (entry.getValue() != null) {
                    encoded.add(entry.getKey() + "=" + URLEncoder.encode(entry.getValue(), "UTF-8"));
                }
            } catch (UnsupportedEncodingException e) {
                BISLog.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
            }
        }

        return TextUtils.join("&", encoded.toArray());
    }

    private static Boolean isBluetoothEnabled() {
        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

        if (adapter == null) {
            return false;
        }
        return adapter.isEnabled();
    }
}
