package com.beaconsinspace.android.beacon.detector;

import android.app.IntentService;
import android.content.Intent;
import android.util.Log;

import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingEvent;

/**
 * Listener for geofence transition changes.
 * <p>
 * Receives geofence transition events from Location Services in the form of an Intent containing
 * the transition type and geofence id(s) that triggered the transition. Creates a notification
 * as the output.
 */
public class BISGeofenceTransitionsIntentService extends IntentService {
    public static final String INTENT = "BISGeofenceTransitionsIntentService";

    private static final String TAG = "GeofenceTransitionsIS";

    /**
     * This constructor is required, and calls the super IntentService(String)
     * constructor with the name for a worker thread.
     */
    public BISGeofenceTransitionsIntentService() {
        // Use the TAG to name the worker thread.
        super(TAG);
    }

    /**
     * Handles incoming intents.
     *
     * @param intent sent by Location Services. This Intent is provided to Location
     *               Services (inside a PendingIntent) when addGeofences() is called.
     */
    @Override
    protected void onHandleIntent(Intent intent) {
        GeofencingEvent geofencingEvent = GeofencingEvent.fromIntent(intent);
        if (geofencingEvent.hasError()) {
            String errorMessage = "Geofence error: " +
                    geofencingEvent.getErrorCode();
            Log.e(TAG, errorMessage);
            return;
        }

        BISDetector detector = null;
        try {
            detector = new BISDetector(getApplicationContext());
            detector.bootstrap();

            BISCollectionManager collectionManager = detector.getCollectionManager();

            if (collectionManager != null) {
                BISCollectionEvent.BISCollectionEventDirection direction = BISCollectionEvent.BISCollectionEventDirection.None;

                switch (geofencingEvent.getGeofenceTransition()) {
                    case Geofence.GEOFENCE_TRANSITION_ENTER:
                        BISLog.d(TAG, "GEOFENCE_TRANSITION_ENTER");
                        direction = BISCollectionEvent.BISCollectionEventDirection.Enter;
                        break;
                    case Geofence.GEOFENCE_TRANSITION_EXIT:
                        BISLog.d(TAG, "GEOFENCE_TRANSITION_EXIT");
                        direction = BISCollectionEvent.BISCollectionEventDirection.Exit;
                        break;
                    case Geofence.GEOFENCE_TRANSITION_DWELL:
                        BISLog.d(TAG, "GEOFENCE_TRANSITION_DWELL");
                        direction = BISCollectionEvent.BISCollectionEventDirection.Dwell;
                        break;
                    default:
                        break;
                }

                BISCollectionEvent event =
                        new BISCollectionEvent(this,
                                BISCollectionEvent.BISCollectionEventType.GeoFence,
                                detector.applicationState,
                                direction,
                                geofencingEvent.getTriggeringLocation()
                        );


                collectionManager.triggerEvent(event);
            }
        } catch (Throwable t) {
            if (detector != null) {
                detector.reportCrash(t);
            }
        }
    }
}