package com.beaconsinspace.android.beacon.detector;

import android.app.Activity;
import android.app.Application;
import android.app.Service;
import android.app.job.JobInfo;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.util.Log;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.util.UUID;

public class BISDetector extends Service implements
        Application.ActivityLifecycleCallbacks {

    static public final String SDK_VERSION = "2.0.4";
    static private final String TAG = "BIS_API";

    private String apiKey = "";
    private String adid = "";
    private UUID uuid;

    static BISLogDelegate logDelegate;
    private BISCollectionManager collectionManager;

    private BISConfiguration configuration;

    private Thread primaryProcessThread;

    private Context context;
    private BISPersistentStorage persistentStorage;

    public BISDetector() {

    }

    public BISDetector(@NonNull Context context) {
        this.context = context;

        if (context != null) {
            this.persistentStorage = new BISPersistentStorage(context);
            this.apiKey = persistentStorage.getString(BISPersistentStorage.KEY_APIKEY);
            setupCallbacks();
        }
    }

    public BISDetector(@NonNull Context context, @NonNull String apiKey) {
        this.context = context;

        if (context != null) {
            this.persistentStorage = new BISPersistentStorage(context);
            this.apiKey = apiKey;

            // Store in persistent storage
            persistentStorage.storeString(BISPersistentStorage.KEY_APIKEY, apiKey);

            setupCallbacks();
        }
    }

    private void setupCallbacks() {
        if (context instanceof Application) {
            Application application = (Application) context;
            application.registerActivityLifecycleCallbacks(this);
        }
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle bundle) {
        applicationState = ApplicationState.INACTIVE;
    }

    @Override
    public void onActivityStarted(Activity activity) {
        applicationState = ApplicationState.ACTIVE;
    }

    @Override
    public void onActivityResumed(Activity activity) {
        applicationState = ApplicationState.ACTIVE;
    }

    @Override
    public void onActivityPaused(Activity activity) {
        applicationState = ApplicationState.BACKGROUND;
    }

    @Override
    public void onActivityStopped(Activity activity) {
        applicationState = ApplicationState.BACKGROUND;
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle bundle) {

    }

    @Override
    public void onActivityDestroyed(Activity activity) {

    }

    enum ApplicationState {
        ACTIVE,
        INACTIVE,
        BACKGROUND
    }

    public ApplicationState applicationState = ApplicationState.BACKGROUND;


    /**
     * Constructor for the BeaconsInSpace Detector
     *
     * @param key String
     * @param ctx Context
     */
    public static void configure(final String key, final Context ctx) {
        configure(key, ctx, null);
    }

    /**
     * Constructor for the BeaconsInSpace Detector
     *
     * @param key String
     * @param ctx Context
     * @param ctx BISLogDelegate
     */
    private static void configure(final String key, final Context ctx, BISLogDelegate logDelegate) {
        Log.d(TAG, "Configuring BISDetector SDK " + SDK_VERSION);

        BISDetector.logDelegate = logDelegate;

        BISDetector detector = new BISDetector(ctx, key);
        scheduleJob(detector);

        // spawn sticky service
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            detector.foregroundService();
        } else {
            ctx.sendBroadcast(new Intent("BootstrapBeaconsInSpace"));
        }
    }

    private void foregroundService() {
        if (primaryProcessThread == null) {
            primaryProcessThread = new Thread() {
                public void run() {
                    try {
                        bootstrap();
                    } catch (Throwable e) {
                        BISLog.e(TAG, e.getMessage());
                    }
                }
            };
            primaryProcessThread.start();
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        context = getApplicationContext();
        persistentStorage = new BISPersistentStorage(context);
        apiKey = persistentStorage.getString(BISPersistentStorage.KEY_APIKEY);

        if (primaryProcessThread == null) {
            primaryProcessThread = new Thread() {
                public void run() {
                    try {
                        bootstrap();
                    } catch (Throwable t) {
                        reportCrash(t);
                    }
                }
            };
            primaryProcessThread.start();
        }

        return START_STICKY;
    }

    private static void scheduleJob(BISDetector detector) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            BISLog.d(TAG, "Scheduling job");

            BISConfiguration configuration = new BISConfiguration(detector);

            int jobId = 1337;
            ComponentName mServiceComponent = new ComponentName(detector.getContext(), BISJobService.class);
            JobInfo.Builder builder = new JobInfo.Builder(jobId, mServiceComponent);
            builder.setPeriodic(configuration.locationMonitoringInterval);
            builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);

            JobScheduler tm = (JobScheduler) detector.getContext().getSystemService(Context.JOB_SCHEDULER_SERVICE);
            tm.cancel(jobId);
            tm.schedule(builder.build());
        }
    }

    public void bootstrap() {
        if (context == null) {
            Log.d(TAG, "Context must be set before bootstrap.");
            return;
        }

        if (persistentStorage == null) {
            Log.d(TAG, "Persistent storage must be set before bootstrap.");
            return;
        }

        if (apiKey == null) {
            Log.d(TAG, "API KEY must be set before bootstrap.");
            return;
        }

        // Ensure library should run
        if (!isSafeToBootstrap()) {
            Log.d(TAG, "This device is not supported. BeaconsInSpace Detector shutting down");
            return;
        }

        try {
            // Get config
            configuration = new BISConfiguration(this);

            // Get UUID
            BISDeviceUUID deviceUUID = new BISDeviceUUID(context);
            uuid = deviceUUID.getDeviceUuid();

            // Read Users Advertisement Identifier
            getUserADID();

            // Device atlas
            if (!persistentStorage.isDeviceMetaDataCollected()) {
                //data is not yet collected from deviceAtlas lib, so booting deviceAtlas..
                bootDeviceAtlas();
            }

            // Create and start Collection Manager
            collectionManager = new BISCollectionManager(this);
        } catch (Throwable t) {
            reportCrash(t);
        }
    }

    /*
     * ===================================SERVICE METHODS=======================================
     */
    @Override
    public IBinder onBind(Intent arg0) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public void onDestroy() {
        //send broadcast before destroying service
        sendBroadcast(new Intent("BootstrapBeaconsInSpace"));
        super.onDestroy();
    }
    /*
     * ===========================================================================================
     */

    /**
     * Checks for unsupported device models
     *
     * @return true if the device is supported; false otherwise
     */
    private static boolean isSafeToBootstrap() {
        // Check that BLE is available
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2) {
            Log.e(TAG, "BeaconsInSpace Detector Library does not run on Android: " + Build.VERSION.SDK_INT);
            return false;
        }
        return true;
    }

    /**
     * Checks for required services like location, internet, bluetooth
     * Gets beacons data from servers
     */
    static void performInitialSetup() {
        Log.i(TAG, "BeaconsInSpace has bootstrapped successfully");
    }

    private void getUserADID() {

        if (context == null) {
            return;
        }

        Thread thread = new Thread() {

            public void run() {
                AdvertisingIdClient.Info adInfo = null;

                try {
                    adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                } catch (IOException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesRepairableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesNotAvailableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (Exception e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } finally {
                    if (adInfo == null) {
                        onUserADIDReceiveFail();
                    } else {
                        boolean limitAdTrackingIsEnabled = adInfo.isLimitAdTrackingEnabled();
                        adid = limitAdTrackingIsEnabled ? "00000000-0000-0000-0000-000000000000" : adInfo.getId();
//                            final boolean isLAT = adInfo.isLimitAdTrackingEnabled();
                        final boolean success = adid != null && adid.length() > 0;

                        // NOTE: we need to post in to execute on UI thread, because user can have UI updates in his overload.
                        Handler handler = new Handler(Looper.getMainLooper());
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (success)
                                    onUserADIDReceiveSuccess();
                                else
                                    onUserADIDReceiveFail();
                            }
                        });
                    }
                } // finally
            } // Thread run()
        }; // new Thread()

        thread.start();
    }

    private void onUserADIDReceiveSuccess() {
        BISDetector.performInitialSetup();
    }

    private void onUserADIDReceiveFail() {
        BISDetector.performInitialSetup();
    }

    /**
     * boot deviceAtlas lib with a translucent Activity and collect data
     */
    private void bootDeviceAtlas() {
        //BISDeviceAtlas is used to provide deviceAtlas with activityContext
        Intent transparentActivityIntent = new Intent(context, BISDeviceAtlas.class);
        transparentActivityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(transparentActivityIntent);
    }

    public BISCollectionManager getCollectionManager() {
        return this.collectionManager;
    }

    public BISConfiguration getConfiguration() {
        return configuration;
    }

    public String getAPIKey() {
        return apiKey;
    }

    public Context getContext() {
        return context;
    }

    public String getADID() {
        return adid;
    }

    public UUID getUUID() {
        return uuid;
    }

    public BISPersistentStorage getPersistentStorage() {
        return this.persistentStorage;
    }

    public void reportCrash(Throwable throwable) {
        BISLog.e("BIS_CRASH", throwable.getClass().getName() + ": " + throwable.getMessage());

        try {
            final BISConfiguration configuration = getConfiguration();

            final BISNetworkingCrashTask task;
            {
                if (configuration == null) {
                    task = new BISNetworkingCrashTask(
                            "https://sentry.io/api/284645/store/",
                            "3d915666beba410c89fe6cb9a3034539",
                            "512bdf97ca284100b0eed543bca78654"
                    );
                } else {
                    task = new BISNetworkingCrashTask(
                            configuration.sentryUrl,
                            configuration.sentryKey,
                            configuration.sentrySecret
                    );
                }
            }

            task.execute(throwable);
            task.get();
        } catch (Throwable crashThrowable) {
            BISLog.e("BIS_CRASH_REPORTING", throwable.getClass().getName() + ": " + throwable.getMessage());
        }

    }
}
