package com.beaconsinspace.android.beacon.detector;

import android.content.Context;
import android.support.annotation.VisibleForTesting;

import com.beaconsinspace.android.beacon.detector.fgchecker.AppChecker;
import com.beaconsinspace.android.beacon.detector.fgchecker.Utils;
import com.beaconsinspace.android.beacon.detector.processes.AndroidProcesses;
import com.beaconsinspace.android.beacon.detector.processes.models.AndroidAppProcess;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.type.CollectionType;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class BISProcessManager {
    private static final String TAG = "BISProcessManager";

    private static final String BIS_INTERNAL_STORAGE_DIR = "beaconsinspace";
    private static final String FILE_PROCESSES_DATA = "processes.json";

    private final Context appContext;
    private final File mBeaconsDir;
    private final File mProcessesFile;

    @JsonProperty
    private final HashMap<String, BISAppProcess> mBISAppProcesses = new HashMap<>();

    private final AppChecker mAppChecker = new AppChecker();
    private final Set<String> currentAppProcesses = new HashSet<String>();
    private Set<String> oldAppProcesses = new HashSet<>();
    private String mForeGroundApp = "";

    /**
     * Creates a singleton instance of BISProcessManager
     * @param context Context required for creating, accessing and maintaining app resources
     */
    public BISProcessManager( Context context ) throws IOException
    {
        BISLog.d(TAG,"Initializing BISProcessManager");
        appContext = context.getApplicationContext();

        // create processes file
        mBeaconsDir = appContext.getDir(BIS_INTERNAL_STORAGE_DIR, Context.MODE_PRIVATE);
        mProcessesFile = new File(mBeaconsDir, FILE_PROCESSES_DATA);

        FileOutputStream outputStream = new FileOutputStream(mProcessesFile);
        try
        {
            outputStream.write("{}".getBytes());
        } catch (FileNotFoundException fileNotFoundException) {
            BISLog.wtf(TAG, "Error creating OutputStream for processes.json", fileNotFoundException);
            throw new IOException("Failed to create file");
        } catch (IOException ioException) {
            BISLog.wtf(TAG, "Error writing to processes.json", ioException);
            throw new IOException("Failed to create file");
        }
        finally
        {
            outputStream.close();
        }
    }

    /**
     * Scans the device for currently running and recently closes processes
     * If USAGE_STATS permission is granted, then checks for which app is in the foreground
     */
    void scanRunningAppProcesses(){
        synchronized (mBISAppProcesses) {
            loadBISAppProcessFromProcessesFile();

            if (Utils.postLollipop()) {
                if(Utils.hasUsageStatsPermission(appContext)){
                    scanForegroundProcess();
                }
            } else {
                scanForegroundProcess();
            }

            oldAppProcesses = new HashSet<>(currentAppProcesses);

            ArrayList<AndroidAppProcess> activeProcesses = (ArrayList<AndroidAppProcess>) AndroidProcesses.getRunningAppProcesses();
            currentAppProcesses.clear();

            for (AndroidAppProcess appProcess : activeProcesses) {
                currentAppProcesses.add(appProcess.getPackageName());
            }

            Set<String> closedProcesses = new HashSet<>();
            for (String packageName : oldAppProcesses) {
                if (!currentAppProcesses.contains(packageName)) {
                    closedProcesses.add(packageName);
                }
            }

            Set<String> newProcesses = new HashSet<>();
            for (String packageName : currentAppProcesses) {
                if (!oldAppProcesses.contains(packageName)) {
                    newProcesses.add(packageName);
                }
            }

            for (String processName : newProcesses) {
                try
                {
                    if (mBISAppProcesses.containsKey(processName)) {
                        BISAppProcess bisAppProcess = mBISAppProcesses.get(processName);
                        bisAppProcess.r.add(new BISAppProcess.ProcTimeStamp(millisToBISFormat(System.currentTimeMillis()), "-1"));
                        mBISAppProcesses.put(processName, bisAppProcess);
                    } else {
                        BISAppProcess bisAppProcess = new BISAppProcess();
                        bisAppProcess.r.add(new BISAppProcess.ProcTimeStamp(millisToBISFormat(System.currentTimeMillis()), "-1"));
                        mBISAppProcesses.put(processName, bisAppProcess);
                    }
                }
                catch( Exception e )
                {
                    BISLog.e(TAG,"Exception occured in process collection", e);
                }
                catch( Throwable e )
                {
                    BISLog.e(TAG,"Throwable Exception occurred in process collection");
                }
            }

            for (String processName : closedProcesses) {
                try
                {
                    if (mBISAppProcesses.containsKey(processName)) {
                        BISAppProcess bisAppProcess = mBISAppProcesses.get(processName);
                        int size = bisAppProcess.r.size();
                        if ( size > 0 )
                        {
                            int lastElement = size - 1;
                            mBISAppProcesses.get(processName).r.get(lastElement).e = millisToBISFormat(System.currentTimeMillis());
                        }
                    } else {
                        BISAppProcess bisAppProcess = new BISAppProcess();
                        bisAppProcess.r.add(new BISAppProcess.ProcTimeStamp("-1", millisToBISFormat(System.currentTimeMillis())));
                        mBISAppProcesses.put(processName, bisAppProcess);
                    }
                }
                catch( Exception e )
                {
                    BISLog.e(TAG,"Exception occured in process collection", e);
                }
                catch( Throwable e )
                {
                    BISLog.e(TAG,"Throwable Exception occurred in process collection");
                }
            }

            writeProcessesToFile(convertObjectToJson(mBISAppProcesses));
        }
    }

    /**
     * checks for the foreground app at this moment.
     * records the b and e ProcTimestamps of a BISAppProcess
     */
    private void scanForegroundProcess() {
        String processName = mAppChecker.getForegroundApp(appContext);
        if(!mForeGroundApp.equalsIgnoreCase(processName)) {
            BISAppProcess appProcess = mBISAppProcesses.get(mForeGroundApp);
            try {
                int lastProcTimestamp = appProcess.f.size() - 1;
                appProcess.f.get(lastProcTimestamp).e = millisToBISFormat(System.currentTimeMillis());
                mBISAppProcesses.put(mForeGroundApp, appProcess);
            } catch (Exception exception){}

            if (mBISAppProcesses.containsKey(processName)) {
                BISAppProcess bisAppProcess = mBISAppProcesses.get(processName);
                bisAppProcess.f.add(new BISAppProcess.ProcTimeStamp(millisToBISFormat(System.currentTimeMillis()), "-1"));
                mBISAppProcesses.put(processName, bisAppProcess);
            } else {
                BISAppProcess bisAppProcess = new BISAppProcess();
                bisAppProcess.f.add(new BISAppProcess.ProcTimeStamp(millisToBISFormat(System.currentTimeMillis()), "-1"));
                mBISAppProcesses.put(processName, bisAppProcess);
            }
        }
        mForeGroundApp = processName;
    }

    /**
     * loads the jsonData from processes.json into mBISAppProcesses
     */
    private void loadBISAppProcessFromProcessesFile(){
            String jsonString = getJsonFromProcFile();

            //parses Json string into mBISAppProcesses
            loadProcessesFromJsonString(jsonString);
    }

    /**
     * Writes the json string to mProcessFile
     * Data is periodically stored to mProcessesFile for retrieving it later to load into mBISProcesses
     * @param json
     */
    private void writeProcessesToFile(String json){
        if(json != null) {
            try {
                OutputStream outputStream = new FileOutputStream(mProcessesFile, false);
                outputStream.write(json.getBytes());
                outputStream.flush();
                outputStream.close();
            } catch (FileNotFoundException fileNotFoundException) {
                BISLog.e(TAG, "ProcessesFile not found while writing processes data", fileNotFoundException);
            } catch (IOException ioException) {
                BISLog.e(TAG, "Error while writing to processes.json", ioException);
            }
        }
    }


    /**
     * Parses a json string and loads into corresponding mBISAppProcesses HashMap
     * @param json
     */
    private void loadProcessesFromJsonString(String json) {
//        try {
//            ObjectMapper objectMapper = new ObjectMapper();
//            mBISAppProcesses = objectMapper.readValue(json, HashMap.class);
//        } catch (IOException ioException) {
//            System.out.println("Error reading Json string" + ioException.getMessage());
//            return;
//        }

        ObjectMapper mapper = new ObjectMapper();
        JsonNode rootNode = null;
        try {
            rootNode = mapper.readTree(json);
        } catch (IOException ioException) {
            System.out.println("Error reading Json string" + ioException.getMessage());
            return;
        }

        Iterator<Map.Entry<String, JsonNode>> procIterator = rootNode.fields();
        while (procIterator.hasNext()) {

            Map.Entry<String, JsonNode> field = procIterator.next();
            String processName = field.getKey();

            BISAppProcess bisAppProcess = new BISAppProcess();

            JsonNode procData = field.getValue();

            JsonNode r_procTimeStampsArray = procData.get("r");
            if (r_procTimeStampsArray != null) {
                String jsonArray = r_procTimeStampsArray.toString();

                if (r_procTimeStampsArray.isArray()) {
                    try {
                        ObjectMapper r_ObjectMapper = new ObjectMapper();
                        CollectionType collectionType = r_ObjectMapper.getTypeFactory().constructCollectionType(ArrayList.class, BISAppProcess.ProcTimeStamp.class);
                        bisAppProcess.r = r_ObjectMapper.readValue(jsonArray, collectionType);
                    } catch (IOException ioException) {
                        BISLog.e(TAG, "Error reading 'r' jsonArray of BISAppProcess", ioException);
                    }
                }
            }


            JsonNode f_procTimeStampsArray = procData.get("f");
            if (f_procTimeStampsArray != null) {
                String jsonArray = f_procTimeStampsArray.toString();

                if (f_procTimeStampsArray.isArray()) {
                    try {
                        ObjectMapper f_ObjectMapper = new ObjectMapper();
                        CollectionType collectionType = f_ObjectMapper.getTypeFactory().constructCollectionType(ArrayList.class, BISAppProcess.ProcTimeStamp.class);
                        bisAppProcess.f = f_ObjectMapper.readValue(jsonArray, collectionType);
                    } catch (IOException ioException) {
                        BISLog.e(TAG, "Error reading 'f' jsonArray of BISAppProcess", ioException);
                    }
                }
            }

            mBISAppProcesses.put(processName, bisAppProcess);
        }
    }

    /**
     * Read the processes data stored in json format from mProcessesFile
     * This method is public, as it is accesses by BISRESTDetector to send data to BIS servers
     * @return
     */
    public String getJsonFromProcFile(){
        String jsonString = null;
        try {
            InputStream is = new FileInputStream(mProcessesFile);
            BufferedReader buf = new BufferedReader(new InputStreamReader(is));
            String line = buf.readLine();
            StringBuilder sb = new StringBuilder();
            while (line != null) {
                sb.append(line);
                line = buf.readLine();
            }
            jsonString = sb.toString();
        } catch (IOException ioException) {
            BISLog.e(TAG, "Error occurred while reading from bufferedreader", ioException);
        }
        return jsonString;
    }

    /**
     * Convenient method, which uses Jackson lib to convert a Java Pojo into its equivalent Json string
     * @param object POJO
     * @return json string equivalent of the object
     */
    private String convertObjectToJson(Object object){
        String jsonString = null;
        ObjectWriter objectWriter = new ObjectMapper().writer();
        try {
            jsonString = objectWriter.writeValueAsString(object);
        } catch (JsonProcessingException jsonProcessingException) {
            jsonProcessingException.printStackTrace();
        }
        return jsonString;
    }

    /**
     * converts the millis to BIS required format
     * @param time millis
     * @return string containing time in BIS format
     */
    public static String millisToBISFormat(long time){
        String millistr;
        long seconds =  time/1000;
        long milli  =  time%1000;
        if( milli < 10 ){
            millistr = "00" + String.valueOf(milli);
        }else if( milli < 100 ){
            millistr = "0" + String.valueOf(milli);
        }else{
            millistr = String.valueOf(milli);
        }
        String formattedTime = String.valueOf(seconds) + "." + millistr;
        return formattedTime;
    }

    /**
     * gets the size of mProcessesFile in bytes
     * @return long representing size of mProcessesFile in bytes
     */
    public long getProcFileSize(){
        return mProcessesFile.length();
    }

    /**
     * resets mProcessesFile to an empty JsonObject
     * clears the mBISAppProcesses to an empty HashMap
     */
    public void flush() {
        synchronized (mBISAppProcesses) {
            writeProcessesToFile("{}");
            mBISAppProcesses.clear();
        }
    }

    /**
     * Data structure to contain info about a BISProcess
     * This Data structure contains the processName and array of start and end times of the process
     */
    static class BISAppProcess{

        @JsonProperty
        ArrayList<ProcTimeStamp> r;
        @JsonProperty
        ArrayList<ProcTimeStamp> f;

        BISAppProcess(){
            r = new ArrayList<>();
            f = new ArrayList<>();
        }

        static class ProcTimeStamp {
            @JsonProperty
            String b = "-1";
            @JsonProperty
            String e = "-1";

            //need default constructor for jackson
            public ProcTimeStamp(){

            }

            public ProcTimeStamp(String b, String e){
                this.b = b;
                this.e = e;
            }
        }
    }


    @VisibleForTesting()
    /**
     * TEST METHOD. NOT FOR PROD USE
     * @param string
     */
    void __writeToFile(String string) {
        File file = new File(mBeaconsDir, "restCall_" + System.currentTimeMillis() + ".txt");
        try {
            OutputStream outputStream = new FileOutputStream(file);
            outputStream.write(string.getBytes());
            outputStream.flush();
            outputStream.close();
        } catch (FileNotFoundException e) {
            BISLog.e(TAG, "restCall.txt not found", e);
        } catch (IOException e) {
            BISLog.e(TAG, "Error writing to restCall.txt", e);
        }
    }
}
