package com.beaconsinspace.android.beacon.detector;

import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.RemoteException;
import android.util.Log;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.utils.UrlBeaconUrlCompressor;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Collection;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;

class BISDetectorManager implements BeaconConsumer {

    static private final String TAG = "BIS_BEACONS_MANAGER";
    static private final Long BEACON_EXIT_THRESHOLD = 30000L;

    static BISDetectorInternalDelegate delegate;

    Context context = null;
    BeaconManager beaconManager = null;

    private static final ConcurrentHashMap<Beacon, Long> beaconTimestampMap = new ConcurrentHashMap<Beacon, Long>();
    private static final ConcurrentHashMap<Beacon, BISDetectorRSSICollector> beaconRssiMap = new ConcurrentHashMap<Beacon, BISDetectorRSSICollector>();
    private static final ConcurrentHashMap<Beacon, Integer> beaconGPSRssiMap = new ConcurrentHashMap<Beacon, Integer>();

    static public void setDelegate(BISDetectorInternalDelegate _delegate) { delegate = _delegate; }

    static void setBeaconGPSRssi(String beaconId)
    {
        for( Beacon beacon : beaconRssiMap.keySet() )
        {
            String bId = uniqueIdentifierForBeacon(beacon);
            if ( beaconId.equals(bId) )
            {
                Integer rssi = beacon.getRssi();
                beaconGPSRssiMap.put(beacon,rssi);
            }
        }
    }

    static Integer getBeaconGPSRssi(Beacon beacon)
    {
        return beaconGPSRssiMap.get( beacon );
    }

    public boolean setContextAndInit(Context ctx) {

        if(ctx == null) {
            BISLog.e(TAG, "Context is null");
            return false;
        }

        context = ctx;

        if(beaconManager == null) {
            try
            {
                beaconManager = BeaconManager.getInstanceForApplication(context);

                setScanPeriodsFromConfigurationData();

                String iBeaconLayout = "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24";
                beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(iBeaconLayout));

                String altBeaconLayout = "m:2-3=beac,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25";
                beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(altBeaconLayout));

                String eddystoneUIDBeaconLayout = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";
                beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(eddystoneUIDBeaconLayout));

                String eddystoneURLBeaconLayout = "s:0-1=feaa,m:2-2=10,p:3-3:-41,i:4-20v";
                beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(eddystoneURLBeaconLayout));

                doBind();
            }
            catch( Exception e )
            {
                Log.e( TAG, "Failed to bind beacon parsers: " + e.getMessage() );
                return false;
            }
        }

        return true;
    }

    public void setScanPeriodsFromConfigurationData()
    {
        String foregroundBluetoothScanPeriodString = BISConfiguration.get( BISConfiguration.KEY_foregroundBluetoothScanPeriod );
        long foregroundBluetoothScanPeriod = Long.parseLong( foregroundBluetoothScanPeriodString );
        String foregroundBluetoothBetweenScanPeriodString = BISConfiguration.get( BISConfiguration.KEY_foregroundBluetoothBetweenScanPeriod );
        long foregroundBluetoothBetweenScanPeriod = Long.parseLong( foregroundBluetoothBetweenScanPeriodString );
        String backgroundBluetoothScanPeriodString = BISConfiguration.get( BISConfiguration.KEY_backgroundBluetoothScanPeriod );
        long backgroundBluetoothScanPeriod = Long.parseLong( backgroundBluetoothScanPeriodString );
        String backgroundBluetoothBetweenScanPeriodString = BISConfiguration.get( BISConfiguration.KEY_backgroundBluetoothBetweenScanPeriod );
        long backgroundBluetoothBetweenScanPeriod = Long.parseLong( backgroundBluetoothBetweenScanPeriodString );

        setScanPeriods( foregroundBluetoothScanPeriod, foregroundBluetoothBetweenScanPeriod, backgroundBluetoothScanPeriod, backgroundBluetoothBetweenScanPeriod );
    }

    /**
     * Set duration of bluetooth scan and sleep periods in both the foreground and background
     *
     * @param foregroundScanPeriod
     * @param foregroundBetweenScanPeriod
     * @param backgroundScanPeriod
     * @param backgroundBetweenScanPeriod
     */
    public void setScanPeriods
    (
            Long foregroundScanPeriod,
            Long foregroundBetweenScanPeriod,
            Long backgroundScanPeriod,
            Long backgroundBetweenScanPeriod
    )
    {
        try
        {
            // Ensure beaconManager is strapped
            if ( beaconManager == null ) { Log.i(TAG,"Beacon Manager not initialized, not setting scan periods.");return; }
            // Foreground
            if ( foregroundScanPeriod != null ) { beaconManager.setForegroundScanPeriod(foregroundScanPeriod); }
            if ( foregroundBetweenScanPeriod != null ) { beaconManager.setForegroundBetweenScanPeriod(foregroundBetweenScanPeriod); }
            // Background
            if ( backgroundScanPeriod != null ) { beaconManager.setBackgroundScanPeriod(backgroundScanPeriod); }
            if ( backgroundBetweenScanPeriod != null ) { beaconManager.setBackgroundBetweenScanPeriod(backgroundBetweenScanPeriod); }
            // Update scan periods
            if( beaconManager.isBound(this) )
            {
                beaconManager.updateScanPeriods();
            }
        }
        catch ( Exception e )
        {
            Log.e(TAG,"Error updating scan periods: "+e.getMessage());
        }
    }

    public void startRanging()
    {
        try
        {
            // We need to stop ranging of old beacons
            stopRanging();
//        Log.d(TAG, "Starting Ranging of UUIDs: " + uuids.toString());
            doBind();
        }
        catch (Exception e)
        {
            Log.e(TAG, "Exception: " + e.toString());
        }
    }

    public void stopRanging()
    {
        try
        {
            Collection<Region> regions = beaconManager.getRangedRegions();
            for ( Region region : regions )
            {
                beaconManager.stopRangingBeaconsInRegion(region);
            }
            undoBind();
        }
        catch (RemoteException e)
        {
            Log.e(TAG, "RemoteException: " + e.toString());
        }
        catch (Exception e)
        {
            Log.e(TAG, "Exception: " + e.toString());
        }
    }

    public void doBind() {
        try
        {
            if(beaconManager != null && !beaconManager.isBound(this)) {
                beaconManager.bind(this);
            }
        }
        catch (Exception e)
        {
            Log.e(TAG, "Failed to bind: " + e.toString());
        }
    }

    public void undoBind() {
        try
        {
            if(beaconManager != null && beaconManager.isBound(this)) {
                beaconManager.unbind(this);
            }
        }
        catch (Exception e)
        {
            Log.e(TAG, "Failed to unbind: " + e.toString());
        }
    }

    @Override
    public void onBeaconServiceConnect() {

        try
        {
            beaconManager.setRangeNotifier(

                    new RangeNotifier() {

                        @Override
                        public void didRangeBeaconsInRegion(Collection<Beacon> collection, org.altbeacon.beacon.Region region) {

                            // get currentTimestampstamp
                            Long currentTimestamp = System.currentTimeMillis();

                    /*
                     * Loop on beacons that are in range and
                     */
                            for ( Beacon beacon : collection )
                            {
                                // determine if we have seen beacon before
                                Long beaconLastSeenTimestamp = beaconTimestampMap.get( beacon );

                                // BRAND-NEW beacon
                                if ( beaconLastSeenTimestamp == null )
                                {
                                    BISLog.d(TAG,"ENTERED BEACON "+uniqueIdentifierForBeacon( beacon ));
                                    handleBeaconEnter(beacon);
                                    beaconRssiMap.put(beacon, new BISDetectorRSSICollector());
                                }

                                // Update
                                BISDetectorRSSICollector rssiCollector = beaconRssiMap.get( beacon );
                                rssiCollector.add( beacon.getRssi() );
                                beaconRssiMap.put( beacon, rssiCollector );
                                beaconTimestampMap.put( beacon, currentTimestamp );
                            }

                    /*
                     * Loop on beacons and check current timestamps
                     */
                            for( Beacon beacon : beaconTimestampMap.keySet() )
                            {
                                Long beaconLastSeenTimestamp = beaconTimestampMap.get( beacon );
                                Long diff = currentTimestamp - beaconLastSeenTimestamp;

                                if ( diff > BEACON_EXIT_THRESHOLD )
                                {
                                    // OLD BEACON
                                    BISLog.d(TAG,"EXITED BEACON "+uniqueIdentifierForBeacon( beacon ));
                                    handleBeaconExit( beacon );
                                    beaconTimestampMap.remove(beacon);
                                }
                                else
                                {
//                                    Log.d(TAG,"BEACON "+uniqueIdentifierForBeacon(beacon)+" TIME SINCE SEEN "+diff+" ");
                                }
                            }
                        }
                    }
            );

            beaconManager.startRangingBeaconsInRegion( new Region ( "ALL_BEACONS", null, null, null ) );
        }
        catch( RemoteException e )
        {
            Log.e(TAG, "RemoteException: " + e.toString());
        }
        catch ( Exception e )
        {
            Log.e(TAG, "Exception: " + e.toString());
        }
    }

    public static String getBeaconRssiJson( Beacon beacon )
    {
        String s = null;
        try
        {
            s = beaconRssiMap.get( beacon ).toJson();
        }
        catch ( Exception e )
        {
            Log.e(TAG, "Exception: " + e.toString());
        }
        return s;
    }

    private Region getBeaconRegion( Beacon beacon )
    {
        Region r = null;
        try
        {
            r = new Region( uniqueIdentifierForBeacon(beacon), beacon.getId1(), null, null );
        }
        catch ( Exception e )
        {
            Log.e(TAG, "Exception: " + e.toString());
        }
        return r;
    }

    public static HashMap<String,String> collectBeaconInfo( Beacon beacon )
    {
        HashMap<String, String> data = new HashMap<String, String>();

        //ids
        String[] ids = idsForBeacon(beacon);
        data.put("beaconId1",ids[0]);
        data.put("beaconId2",ids[1]);
        data.put("beaconId3",ids[2]);

        data.put("beaconServiceUUID",Integer.toString(beacon.getServiceUuid()));
        // TxPower
        int TxPower = beacon.getTxPower();
        String TxPowerString = Integer.toString( TxPower );
        data.put("beaconTxPower", TxPowerString );
        // Distance
        Double distance = beacon.getDistance();
        String distanceString = Double.toString(distance);
        data.put("beaconDistance",distanceString);
        // Name
        String bName = beacon.getBluetoothName();
        data.put("beaconName",bName);
        // Manufacturer
        int manufacturer = beacon.getManufacturer();
        String manufacturerString = Integer.toString(manufacturer);
        data.put("beaconManufacturer",manufacturerString);
        // Type
        int type = beacon.getBeaconTypeCode();
        String typeString = Integer.toString( type );
        data.put("beaconTypeCode", typeString);

        // Check for Eddystone URL
        if ( beacon.getServiceUuid() == 0xfeaa && beacon.getBeaconTypeCode() == 0x10 )
        {
            // This is a Eddystone-URL frame
            String url = UrlBeaconUrlCompressor.uncompress(beacon.getId1().toByteArray());
            data.put("beaconURL",url);
        }
        // Check for Eddystone UID
        if ( beacon.getServiceUuid() == 0xfeaa && beacon.getBeaconTypeCode() == 0x00 )
        {
            // Check for Telemetry Data
            if ( beacon.getExtraDataFields().size() > 0 )
            {
                long telemetryVersion = beacon.getExtraDataFields().get(0);
                long batteryMilliVolts = beacon.getExtraDataFields().get(1);
                long pduCount = beacon.getExtraDataFields().get(3);
                long uptime = beacon.getExtraDataFields().get(4);
                data.put( "beaconTelemetryVersion", Long.toString( telemetryVersion ) );
                data.put( "beaconTelemetryBattery", Long.toString( batteryMilliVolts ) );
                data.put( "beaconTelemetryPduCount", Long.toString( pduCount ) );
                data.put( "beaconTelemetryUptime", Long.toString( uptime ) );
            }
        }

        return data;
    }

    public static String beaconIdentifierToString ( Identifier i )
    {
        try
        {
            if ( i == null ) { return null; }
            String s = i.toString();
            if ( s == null ) { return null; }

            if ( s.length() >= 2 && s.substring(0,2).equals("0x") )
            {
                s = s.substring(2); // get rid of the 0x
            }
            return s;
        }
        catch( Exception e )
        {
            Log.e( TAG,"Exception: " + e.getMessage() );
        }
        return null;
    }

    private void handleBeaconEnter(Beacon b) {

        final Beacon beacon = b;
        Thread thread = new Thread()
        {
            public void run()
            {
                //        Log.d(TAG, "BEACON EXIT ::: " + beacon.toString());
                String jsonDataString = BISDetectorREST.notifyAboutBeaconEnter(beacon);
                String beaconId = null;

                if ( jsonDataString != null )
                {
                    try
                    {
                        JSONObject jsonDataObject = new JSONObject(jsonDataString);
                        int code = jsonDataObject.getInt("code");
                        if ( code != 200 )
                        {
                            Log.e(TAG,"Unsuccessful API Response occurred"+jsonDataString);
                            Log.e(TAG,beacon.toString());
                            return;
                        }
                        beaconId = jsonDataObject.getJSONObject("data").getString("beaconId");
                    }
                    catch( JSONException e)
                    {
                        Log.e(TAG,"Failed to parse JSON"+e.getMessage());
                    }
                }

                try
                {
                    if( delegate != null && beaconId != null )
                    {
                        BISLocationListener.assignLocationToBeaconId(uniqueIdentifierForBeacon(beacon));
                        delegate.onBeaconEnter(beaconId);
                    }
                }
                catch( Exception e )
                {
                    Log.e( TAG,"Exception: " + e.getMessage() );
                }
            }
        };
        thread.start();
    }

    private void handleBeaconExit(Beacon b)
    {
        final Beacon beacon = b;
        Thread thread = new Thread()
        {
            public void run()
            {
//                        Log.d(TAG, "BEACON EXIT ::: " + beacon.toString());
                String jsonDataString = BISDetectorREST.notifyAboutBeaconExit(beacon);
                String beaconId = null;

                if ( jsonDataString != null )
                {
                    try
                    {
                        JSONObject jsonDataObject = new JSONObject(jsonDataString);
                        int code = jsonDataObject.getInt("code");
                        if ( code != 200 )
                        {
                            Log.e(TAG,"Unsuccessful API Response occurred"+jsonDataString);
                            Log.e(TAG,beacon.toString());
                            return;
                        }
                        beaconId = jsonDataObject.getJSONObject("data").getString("beaconId");
                    }
                    catch( JSONException e)
                    {
                        Log.e(TAG,"Failed to parse JSON"+e.getMessage());
                    }
                }

                try
                {
                    if( delegate != null && beaconId != null )
                    {
                        delegate.onBeaconExit(beaconId);
                    }
                }
                catch( Exception e )
                {
                    Log.e( TAG,"Exception: " + e.getMessage() );
                }
            }
        };
        thread.start();
    }

    static public String[] idsForBeacon(Beacon beacon) {

        if(beacon == null) {
            return new String[]{"", "", ""};
        }


        Identifier id1 = null;
        Identifier id2 = null;
        Identifier id3 = null;
        try { id1 = beacon.getIdentifiers().size() > 0 ? beacon.getId1() : null; } catch( Exception e ){}
        try { id2 = beacon.getIdentifiers().size() > 1 ? beacon.getId2() : null; } catch( Exception e ){}
        try { id3 = beacon.getIdentifiers().size() > 2 ? beacon.getId3() : null; } catch( Exception e ){}

        String ID1 = id1 != null ? beaconIdentifierToString(id1) : "";
        String ID2 = id2 != null ? beaconIdentifierToString(id2) : "";
        String ID3 = id3 != null ? beaconIdentifierToString(id3) : "";

        return new String[]{ID1,ID2,ID3};
    }

    static public String uniqueIdentifierForBeacon(Beacon beacon)
    {

        if(beacon == null) {
            return "";
        }

        String s = null;
        try
        {
            String[] strings = idsForBeacon(beacon);

            String id1 = strings[0];
            String id2 = strings[1];
            String id3 = strings[2];

            String uniqueId = id1 + "_" + id2 + "_" + id3;

            s = id1+":"+uniqueId.hashCode();
        }
        catch( Exception e )
        {
            Log.e( TAG,"Exception: " + e.getMessage() );
        }
        return s;
    }


    @Override
    public Context getApplicationContext() { return context; }

    @Override
    public void unbindService(ServiceConnection serviceConnection)
    {
        try
        {
            context.unbindService(serviceConnection);
        }
        catch( Exception e )
        {
            Log.e(TAG,e.toString());
        }
    }

    @Override
    public boolean bindService(Intent intent, ServiceConnection serviceConnection, int i)
    {
        try
        {
            return context.bindService(intent, serviceConnection, i);
        }
        catch( Exception e )
        {
            Log.e(TAG,e.toString());
        }
        return false;
    }
}
