/*
 * Copyright (c) 2008-2016 Afilias Technologies Limited. All rights reserved.
 */

package com.beaconsinspace.android.beacon.detector.DeviceAtlas;

import android.app.Activity;
import android.util.Log;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.TimeZone;

/**
 * Stats to gather:
 *  - Vendor, model et
 *  - CPU data, type, cores etc
 *  - GPU data
 *  - memory data
 *  - sensors
 *  - sim card data
 *  - screen props
 *  - camera props
 *  - battery
 *  - feature support, wifi, bluetooth, gps nfc etc
 *  - storage internal/external
 *  - system+webview UAs
 *  - browser properties
 *
 * @author Afilias Technologies Ltd
 */
public class DataCollector extends GpuCallback {
    private static final String TAG = DataCollector.class.getName();

    private static final DateFormat DATE_FORMAT;
    static {
        DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mmZ");
        DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("UTC"));
    }


    public static final String INFO = "info";
    private static final String VERSION = "version";
    private static final String SOURCE = "source";
    private static final String VERSION_NUM = "0.1";
    private static final String DATE = "date";

    public static final String DATA = "data";
    private static final String BUILD = "build";
    private static final String BUILD_VERSION = "buildVersion";
    private static final String WEB = "web";
    private static final String CPU = "cpu";
    private static final String MEMORY = "memory";
    private static final String GPU = "gpu";
    private static final String TELEPHONY = "telephony";
    private static final String STORAGE = "storage";
    private static final String SENSORS = "sensors";
    private static final String DISPLAY = "display";
    private static final String BATTERY = "battery";
    private static final String KEYBOARD = "keyboard";
    private static final String CAMERA = "camera";


    private Activity appContext;
    private HandleDataCallback callback;



    public DataCollector(Activity activity) {
        this(activity, null);
    }

    public DataCollector(Activity activity, HandleDataCallback callback) {
        Log.i(TAG, "DataCollector");
        this.appContext = activity;
        this.callback = callback;

        GpuLoader gpuLoader = new GpuLoader();
        gpuLoader.load(activity, this);
    }



    @Override
    protected void collectStats(GpuProperties gpuProps) {

        try {
            JSONObject data = new JSONObject();

            // TODO: filter BUILD properties
            data.put(BUILD, getBuildProperties());
            data.put(BUILD_VERSION, getBuildVersionProperties());
            data.put(WEB, WebProperties.getProperties(appContext));
            data.put(CPU, CpuProperties.getProperties());
            data.put(GPU, gpuProps.getProperties());
            data.put(MEMORY, MemoryProperties.getProperties());
            data.put(DISPLAY, DisplayProperties.getProperties(appContext));
            data.put(TELEPHONY, TelephonyProperties.getProperties(appContext));
            data.put(STORAGE, StorageProperties.getProperties());
            data.put(SENSORS, SensorProperties.getProperties(appContext));


            JSONObject all = new JSONObject();
            all.put(INFO, getInfo());
            all.put(DATA, data);

            if (callback != null) {
                callback.handleData(all);
            }
        } catch(JSONException ex) {
            Log.d(TAG, ex.toString());
        }
    }



    private JSONObject getInfo() throws JSONException {
        JSONObject info = new JSONObject();
        info.put(VERSION, VERSION_NUM);
        info.put(DATE, DATE_FORMAT.format(new Date()));
        info.put(SOURCE, getAppName());

        return info;
    }


    private String getAppName() {
        return appContext.getApplicationInfo().loadLabel(appContext.getPackageManager()).toString();
    }



    /**
     *
     * @return
     */
    private JSONObject getBuildProperties() throws JSONException {
        Field[] fields = android.os.Build.class.getDeclaredFields();
        return getPropertiesFromFields(fields, null);
    }


    /**
     *
     * @return
     */
    private JSONObject getBuildVersionProperties() throws JSONException {
        Field[] fields = android.os.Build.VERSION.class.getDeclaredFields();

        Set<String> exclude = new HashSet<>();
        exclude.add("SDK"); // deprecated in API 4 (old!) and superseded by SDK_INT

        return getPropertiesFromFields(fields, exclude);
    }



    /**
     *
     * @param fields
     * @param exclude
     * @return
     */
    private JSONObject getPropertiesFromFields(Field[] fields, Set<String> exclude) throws JSONException {
        JSONObject props = new JSONObject();

            for(Field field : fields) {
                //field.setAccessible(true);
                String key = field.getName();
                if(exclude != null && exclude.contains(key)) {
                    continue;
                }

                try {
                    key = StrUtil.toCamelCase(field.getName());
                    String value = StrUtil.asString(field.get(key));
                    props.put(key, value);
                } catch (IllegalAccessException e) {
                    Log.d(TAG, "*** Can't access: "+key);
                }
            }


        return props;
    }



}
