package com.beaconsinspace.android.beacon.detector;

import android.app.Service;
import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.location.LocationManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.provider.Settings;
import android.util.Log;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import org.json.JSONArray;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

public class BISDetector extends Service implements BISDetectorInternalDelegate
{

    /*
     * VERSION
     */
    static public String SDK_VERSION = "1.3.5";

    /*
     * Definitions
     */
    static private final String TAG = "BIS_API";

    static String API_KEY = "";
    static String ADID = "";
    static UUID UUID;

    static BISDetector sharedInstance = new BISDetector();
    static BISDetectorManager beaconsManager = new BISDetectorManager();
    static BISDetectorDelegate delegate;

    private Thread primaryProcessThread;
    private static Thread continuousLocationMonitoringThread;
    private static Thread continuousConfigurationMonitoringThread;

    static Context appContext;

    /**
     * Constructor for the BeaconsInSpace Detector
     *
     * @param key String
     * @param ctx Context
     * @param dlgt BISDetectorDelegate
     */
    public static void configure(String key, Context ctx, BISDetectorDelegate dlgt) {
        Log.d(TAG, "Configuring BISDetector SDK " + SDK_VERSION);

        // Set props
        API_KEY = key;
        appContext = ctx.getApplicationContext();//always get appContext
        delegate = dlgt;

        // Store in persistent storage
        BISPersistentStorage.storeString( BISPersistentStorage.KEY_APIKEY, key );

        if(!BISPersistentStorage.isDeviceMetaDataCollected()) {
            //data is not yet collected from deviceAtlas lib, so booting deviceAtlas..
            sharedInstance.bootDeviceAtlas();
        }

        // spawn service
        appContext.sendBroadcast(new Intent("BootstrapBeaconsInSpace") );
    }

    public static Context getContext() { return appContext; }

    private void bootstrap(String key, Context ctx) {
        // Set API Key
        API_KEY = key != null ? key : "";

        // Get config
        BISConfiguration.update();

        // Ensure library should run
        if ( ! isSafeToBootstrap() )
        {
            Log.d(TAG, "This device is not supported. BeaconsInSpace Detector shutting down");
            return;
        }

        // Get UUID
        BISDeviceUUID deviceUUID = new BISDeviceUUID(ctx);
        UUID = deviceUUID.getDeviceUuid();

        // Read Users Advertisement Identifier
        BISDetector.getUserADID();

        // Set Context
        beaconsManager.setContextAndInit(ctx);

        // Set Internal Delegate
        BISDetectorInternalDelegate internalDelegate = sharedInstance;
        BISDetectorREST.setDelegate(internalDelegate);
        BISDetectorServicesListener.setDelegate(internalDelegate);
        BISDetectorManager.setDelegate(internalDelegate);
    }

    /*
     * ===================================SERVICE METHODS=======================================
     */
    @Override
    public IBinder onBind(Intent arg0) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        // get vars needed to strap
        appContext = this.getApplicationContext();
        API_KEY = BISPersistentStorage.getString( BISPersistentStorage.KEY_APIKEY );

        if (primaryProcessThread == null )
        {
            primaryProcessThread = new Thread()
            {
                public void run()
                {
                    try
                    {
                        bootstrap(API_KEY, appContext);
                    }
                    catch ( Exception e )
                    {
                        e.printStackTrace();
                        Log.e(TAG,e.getMessage());
                    }
                }
            };
            primaryProcessThread.start();
        }

        return START_STICKY;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        sendBroadcast( new Intent( "BootstrapBeaconsInSpace" ) );
    }
    /*
     * ===========================================================================================
     */

    private static boolean isSafeToBootstrap()
    {
        // Check that BLE is available
        if ( Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2 )
        {
            Log.e(TAG,"BeaconsInSpace Detector Library does not run on Android: "+Build.VERSION.SDK_INT);
            return false;
        }

        // Check that it is not a Bluetooth/WIFI collision problem device
        // Get list of device models from configuration
        List<String> unsupportedDeviceModels = new ArrayList<String>();
        try
        {
            String unsupportedDeviceModelsJSONString = BISConfiguration.get( BISConfiguration.KEY_unsupportedAndroidModels );
            JSONArray unsupportedDeviceModelsJSONArray = new JSONArray( unsupportedDeviceModelsJSONString );
            for ( int i = 0; i < unsupportedDeviceModelsJSONArray.length(); i++ )
            {
                unsupportedDeviceModels.add( unsupportedDeviceModelsJSONArray.getString(i) );
            }
        }
        catch ( Exception e )
        {
            e.printStackTrace();
        }

        String deviceModel = Build.MODEL;
        for( String unsupportedModel : unsupportedDeviceModels )
        {
            if ( deviceModel.equals(unsupportedModel) )
            {
                Log.e(TAG,"BeaconsInSpace Detector Library does not run on Android Model "+unsupportedModel+" due to Networking/Bluetooth collision issues.");
                return false;
            }
        }

        return true;
    }

    /**
     *
     * Set duration of bluetooth scan and sleep periods in both the foreground and background
     * If any of the values are null then the respective setting will not be modified
     *
     * @param foregroundScanPeriod
     * @param foregroundBetweenScanPeriod
     * @param backgroundScanPeriod
     * @param backgroundBetweenScanPeriod
     */
    public static void setBeaconManagerScanPeriods
    (
        Long foregroundScanPeriod,
        Long foregroundBetweenScanPeriod,
        Long backgroundScanPeriod,
        Long backgroundBetweenScanPeriod
    )
    {
        beaconsManager.setScanPeriods( foregroundScanPeriod, foregroundBetweenScanPeriod, backgroundScanPeriod, backgroundBetweenScanPeriod );
    }

    static void performInitialSetup() {
        /*
         * Verify all required services are enabled
         */
        String errorMessage = "";
        if (!isLocationServiceEnabled())
        {
            errorMessage = "Location Services are not enabled. Please enable them in Settings.";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        if (!isInternetAvailable())
        {
            errorMessage = "Network Services are not enabled. Please enable them in Settings.";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        if (!isBluetoothEnabled())
        {
            errorMessage = "Bluetooth is not enabled. Please turn it on to proceed";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        /*
         * Get beacon identifiers
         */
        BISDetectorREST.getBeaconsInfoFromServer();
    }

    static BISDetectorManager beaconsManager() {
        return beaconsManager;
    }

    static void getUserADID() {

        if(appContext == null) { return; }

        Thread thread = new Thread() {

            public void run() {
                AdvertisingIdClient.Info adInfo = null;

                try {
                    adInfo = AdvertisingIdClient.getAdvertisingIdInfo(appContext);
                } catch (IOException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesRepairableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesNotAvailableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (Exception e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } finally {
                    if(adInfo == null) {
                        onUserADIDReceiveFail();
                    }
                    else {
                        boolean limitAdTrackingIsEnabled = adInfo.isLimitAdTrackingEnabled();
                        ADID = limitAdTrackingIsEnabled ? "00000000-0000-0000-0000-000000000000" : adInfo.getId();
//                            final boolean isLAT = adInfo.isLimitAdTrackingEnabled();
                        final boolean success = ADID != null && ADID.length() > 0;

                        // NOTE: we need to post in to execute on UI thread, because user can have UI updates in his overload.
                        Handler handler = new Handler(Looper.getMainLooper());
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (success)
                                    onUserADIDReceiveSuccess();
                                else
                                    onUserADIDReceiveFail();
                            }
                        });
                    }
                } // finally
            } // Thread run()
        }; // new Thread()

        thread.start();
    }

    static public boolean isLocationServiceEnabled()
    {
        if(appContext == null)
            return false;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
        {
            int locationMode = 0;
            try
            {
                locationMode = Settings.Secure.getInt(appContext.getContentResolver(), Settings.Secure.LOCATION_MODE);
            }
            catch (Settings.SettingNotFoundException e)
            {
                Log.e(TAG,"Failed to check for location services: "+e.getMessage());
                return false;
            }
            return locationMode != Settings.Secure.LOCATION_MODE_OFF;
        }
        else
        {
            LocationManager lm = (LocationManager) appContext.getSystemService(Context.LOCATION_SERVICE);
            boolean gps_enabled = false;
            boolean network_enabled = false;

            try { gps_enabled = lm.isProviderEnabled(LocationManager.GPS_PROVIDER); } catch(Exception ex) {}
            try { network_enabled = lm.isProviderEnabled(LocationManager.NETWORK_PROVIDER); } catch(Exception ex) {}

            return gps_enabled || network_enabled;
        }
    }

    static public boolean isInternetAvailable() {

        ConnectivityManager connectivityManager = (ConnectivityManager) appContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();

        return activeNetworkInfo != null && activeNetworkInfo.isConnected();
    }

    static public boolean isBluetoothEnabled() {

        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

        if ( adapter == null ) { return false; }
        return adapter.isEnabled();
    }

    static public void startRanging() {
        beaconsManager.setContextAndInit(appContext);
        beaconsManager.startRanging();
        Log.d(TAG, "BLE ranging has begun");
    }

    static public void stopRanging() {
        beaconsManager.stopRanging();
    }

    static void onUserADIDReceiveSuccess() {
        BISDetector.performInitialSetup();
    }

    static void onUserADIDReceiveFail() {
        BISDetector.performInitialSetup();
    }

    @Override
    public void onBeaconsInfoReceiveSuccess() {
        startRanging();
        beginLocationMonitoring();
        beginConfigurationMonitoring();
    }

    @Override
    public void onBeaconsInfoReceiveFail( int errorCode, String errorMessage ) {
        if ( errorMessage == null ) {
            switch( errorCode )
            {
                case 401:
                    errorMessage = "Invalid Authentication";
                    break;
                default:
                    errorMessage = "An error occurred";
                    break;
            }
        }
        errorMessage = errorCode+" : "+errorMessage;
        Log.e(TAG, errorMessage);
        if ( delegate != null ) { delegate.onBISError( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, errorMessage ); }
        beginLocationMonitoring();
        beginConfigurationMonitoring();
    }

    @Override
    public void checkIfShouldUpdateBeaconsInfo() {
        if(isInternetAvailable() && beaconsManager.uuids.size() == 0) {
            BISDetectorREST.getBeaconsInfoFromServer();
        }
    }

    @Override
    public void onBeaconEnter(String beaconId) {
        if(delegate != null)
        {
            delegate.didEnterBISRegion(beaconId);
        }
    }

    @Override
    public void onBeaconExit(String beaconId) {
        if(delegate != null)
        {
            delegate.didExitBISRegion(beaconId);
        }
    }

    private static void beginLocationMonitoring()
    {
        if ( continuousLocationMonitoringThread == null )
        {
            continuousLocationMonitoringThread = new Thread()
            {
                public void run()
                {
                    try
                    {
                        while ( true )
                        {
                            // get Config value
                            String monitorIntervalString = BISConfiguration.get(BISConfiguration.KEY_locationMonitoringInterval);
                            // if the value does not exit or it is set to 0 then do not run location monitoring.
                            if ( monitorIntervalString == null || monitorIntervalString.equals("0") ) { break; }
                            // So now we know we definitely have a value.
                            int monitorIntervalInt = Integer.parseInt(monitorIntervalString);
                            String uniqueId = "CONTINUOUS_PROCESSING";
                            BISLocationListener.assignLocationToBeaconId(uniqueId);
                            Thread.sleep(60000); // wait for GPS data to be collected
                            Location location = BISLocationListener.getLocationByBeaconId(uniqueId);
                            BISDetectorREST.sendGPSEvent(location,true); // TODO : ! determine when to send additional data !
                            Thread.sleep(monitorIntervalInt); // wait before beginning again
                        }
                    }
                    catch (Exception e)
                    {
                        e.printStackTrace();
                    }
                }
            };
            continuousLocationMonitoringThread.start();
        }
    }

    private static void beginConfigurationMonitoring()
    {

        if ( continuousConfigurationMonitoringThread == null )
        {
            continuousConfigurationMonitoringThread = new Thread()
            {
                public void run()
                {
                    try
                    {
                        while ( true )
                        {
                            Thread.sleep(10800000);
                            BISConfiguration.update();
                            beaconsManager.setScanPeriodsFromConfigurationData();
                        }
                    }
                    catch (Exception e)
                    {
                        e.printStackTrace();
                    }
                }
            };
            continuousConfigurationMonitoringThread.start();
        }

    }

    /**
     * boot deviceAtlas lib with a translucent Activity and collect data
     */
    private void bootDeviceAtlas(){
        //BISDeviceAtlas is used to provide deviceAtlas with activityContext
        Intent transparentActivityIntent = new Intent(appContext, BISDeviceAtlas.class);
        transparentActivityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        appContext.startActivity(transparentActivityIntent);
    }

}
