package com.beaconsinspace.android.beacon.detector;

import android.content.Context;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Looper;
import android.util.Log;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by johnlfoleyiii on 6/2/16.
 */
class BISLocationListener extends AsyncTask<String, String, Location> implements LocationListener {

    private static final String TAG = "BIS_LOCATION_LISTENER";

    private static final int LOCATION_HALF_LIFE = 1000 * 20; // seconds until location is deemed stale/invalid
    private static final int NETWORK_PROVIDER_LOWER_ACCURACY_BOUND = 8; // Accuracy bound that is good enough to stop location updates
    private static final int GPS_PROVIDER_LOWER_ACCURACY_BOUND = 8; // Accuracy bound that is good enough to stop location updates
    private static final int GPS_PROVIDER_UPPER_ACCURACY_BOUND = 12; // Accuracy bound that would require trying network location updates and extending SatelliteGPS updates
    private static final long MAX_MONITORING_PERIOD = 1000 * 15;
    private static final HashMap<String, Location> beaconLocations = new HashMap<String, Location>();
    private static boolean isCurrentlyMonitoring = false;
    private static Location currentBestLocation;
    private static boolean isMonitoringSatelliteGPS=false;
    private static boolean isMonitoringNetworkGPS=false;

    private Context context;
    private LocationManager locationManager;

    BISLocationListener(Context ctx)
    {
        context = ctx;
    }

    public static void assignLocationToBeaconId(String beaconId)
    {
        if ( locationIsStillValid(currentBestLocation ) )
        {
            setBeaconIdLocation(beaconId, currentBestLocation);
        }
        else
        {
            setBeaconIdLocation(beaconId, null);
            if ( ! isCurrentlyMonitoring )
            {
                currentBestLocation=null;
                BISLocationListener locationListener = new BISLocationListener(BISDetector.context);
                locationListener.execute(beaconId);
            }
        }
    }

    public static Location getLocationByBeaconId(String beaconId)
    {
        Location location = beaconLocations.get(beaconId);
        beaconLocations.remove(beaconId);
        if ( location == null && currentBestLocation != null ) { location = currentBestLocation; } // if we dont have a location and the app is monitoring and has a current best location then get it
        return location;
    }

    private static void setBeaconIdLocation(String beaconId, Location location) {
        beaconLocations.put(beaconId, location);
    }

    @Override
    protected Location doInBackground(String... beaconIdArr)
    {
        beginLocationMonitoring();
        while (isCurrentlyMonitoring) {}
        if ( currentBestLocation != null )
        {
            String l = currentBestLocation.toString();
        }
        return currentBestLocation;
    }

    @Override
    protected void onPostExecute(Location location)
    {
        try
        {
            for (Map.Entry<String, Location> entry : beaconLocations.entrySet())
            {
                // Set valid location on all of the beaconLocation entries with null locations
                String entryBeaconId = entry.getKey();
                Location entryLocation = entry.getValue();
                if (entryLocation == null) {
                    setBeaconIdLocation(entryBeaconId, location);
                }
            }
            super.onPostExecute(location);
        }
        catch( Exception e )
        {
            Log.e(TAG,"ERROR "+e.getMessage());
        }
    }

    static boolean locationIsStillValid(Location loc)
    {
        if (loc == null) {
            return false;
        }
        long locTime = loc.getTime();
        long currentTime = System.currentTimeMillis();
        return currentTime - locTime < LOCATION_HALF_LIFE;
    }

    void beginLocationMonitoring()
    {
        try {
            /*
             * Verify if its possible to search for GPS via GPS Satellite or Network Provider, and act accordingly
             */
            locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            boolean satelliteGPSAvailable = false;
            boolean networkGPSAvailable = false;

            try { satelliteGPSAvailable = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER); } catch (Exception ex) { }
            try { networkGPSAvailable = locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER); } catch (Exception ex) { }

            if (!satelliteGPSAvailable && !networkGPSAvailable)
            {
                endLocationMonitoring();
                return;
            }

            isCurrentlyMonitoring = true;

            if (satelliteGPSAvailable) {
                beginSatelliteGPSMonitoring();
            } else {
                beginNetworkGPSMonitoring();
            }

            monitorLocationUpdates();
        }
        catch (SecurityException e)
        {
            Log.e(TAG, "LOCATION EXCEPTION" + e.getMessage());
            endLocationMonitoring();
        }
        catch ( Exception e )
        {
            Log.e(TAG, "EXCEPTION " + e.getMessage());
        }
    }

    void beginSatelliteGPSMonitoring()
    {
        try
        {
            locationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, 0, 0, this, Looper.getMainLooper());
            isMonitoringSatelliteGPS=true;
        }
        catch(SecurityException e )
        {
            Log.e(TAG,"Failed to monitor satellite GPS: "+e.getMessage());
        }
        catch ( Exception e )
        {
            Log.e(TAG, "EXCEPTION " + e.getMessage());
        }
    }
    void beginNetworkGPSMonitoring()
    {
        try
        {
            locationManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, 0, 0, this, Looper.getMainLooper());
            isMonitoringNetworkGPS=true;
        }
        catch( SecurityException e )
        {
            Log.e(TAG,"Failed to monitor network GPS: "+e.getMessage());
        }
        catch ( Exception e )
        {
            Log.e(TAG, "EXCEPTION " + e.getMessage());
        }
    }

    void monitorLocationUpdates()
    {
        Thread thread = new Thread()
        {
            public void run()
            {
                try { Thread.sleep( MAX_MONITORING_PERIOD ); }
                catch (InterruptedException e) { e.printStackTrace(); }
            }
        };
        thread.start();
        try { thread.join(); }
        catch (InterruptedException e) { e.printStackTrace(); }

        if ( isCurrentlyMonitoring )
        {
            if ( ! isMonitoringNetworkGPS && ( currentBestLocation == null || currentBestLocation.getAccuracy() >= GPS_PROVIDER_UPPER_ACCURACY_BOUND ) )
            {
                beginNetworkGPSMonitoring();
                monitorLocationUpdates();
            }
            else
            {
                endLocationMonitoring();
            }
        }
    }

    void endLocationMonitoring()
    {
        try
        {
            locationManager.removeUpdates(this);
            isMonitoringNetworkGPS=false;
            isMonitoringSatelliteGPS=false;
            isCurrentlyMonitoring=false;
        }
        catch( SecurityException e )
        {
            Log.e(TAG,"FAILED TO END LOCATION MONITORING"+e.getMessage());
        }
        catch ( Exception e )
        {
            Log.e(TAG, "EXCEPTION " + e.getMessage());
        }
    }

    @Override
    public void onLocationChanged(Location loc)
    {
        makeUseOfNewLocation(loc);
    }

    /**
     * This method modify the last know good location according to the arguments.
     *
     * @param location The possible new location.
     */
    void makeUseOfNewLocation(Location location)
    {
        if ( isBetterLocation( location, currentBestLocation ) )
        {
            currentBestLocation = location;
            if
            (
                (
                    location.getProvider().equals(LocationManager.GPS_PROVIDER)
                    &&
                    location.getAccuracy() <= GPS_PROVIDER_LOWER_ACCURACY_BOUND
                )
                ||
                (
                    location.getProvider().equals(LocationManager.NETWORK_PROVIDER)
                    &&
                    location.getAccuracy() <= NETWORK_PROVIDER_LOWER_ACCURACY_BOUND
                )
            )
            {
                endLocationMonitoring();
            }
        }
    }

    /**
     * Determines whether one location reading is better than the current location fix
     * @param location  The new location that you want to evaluate
     * @param currentBestLocation  The current location fix, to which you want to compare the new one.
     */
    protected boolean isBetterLocation(Location location, Location currentBestLocation)
    {
        // A new location is always better than no location
        if (currentBestLocation == null) { return true; }

        // Check whether the new location fix is newer or older
        long timeDelta = location.getTime() - currentBestLocation.getTime();
        boolean isSignificantlyNewer = timeDelta > LOCATION_HALF_LIFE;
        boolean isSignificantlyOlder = timeDelta < -LOCATION_HALF_LIFE;
        boolean isNewer = timeDelta > 0;

        // If the location is old we don't want it
        if (isSignificantlyOlder) { return false; }

        // Check whether the new location fix is more or less accurate
        int accuracyDelta = (int) (location.getAccuracy() - currentBestLocation.getAccuracy());
        boolean isLessAccurate = accuracyDelta > 0;
        boolean isMoreAccurate = accuracyDelta < 0;

        // Determine location quality using a combination of timeliness and accuracy
        if ( isMoreAccurate ) { return true; } // More Accurate
        if ( isNewer && !isLessAccurate ) { return true; } // Newer and same accuracy
        return false;
    }

    @Override
    public void onStatusChanged(String var1, int var2, Bundle var3) { }

    @Override
    public void onProviderEnabled(String var1)
    {
        beginLocationMonitoring();
    }

    @Override
    public void onProviderDisabled(String var1)
    {
        endLocationMonitoring();
    }

}
