package com.beaconsinspace.android.beacon.detector;

import android.location.Location;
import android.os.Build;
import android.util.Base64;
import android.util.Log;

import org.altbeacon.beacon.Beacon;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;


class BISDetectorREST {

    // Defines
    static final String TAG = "BIS_REST";
    static String BIS_URL_EVENT = "https://api.beaconsinspace.com/v1/event";
    static String BIS_URL_BEACON = "https://api.beaconsinspace.com/v1/beacon?userId=%s&userIdType=%s&userId2=%s&userIdType2=%s&os=%s&device=%s&tz=%s&language=%s&country=%s&sdkVersion=%s&createdAt=%f";

    // Local
    static BISDetectorInternalDelegate delegate;

    // Setup
    static public void setDelegate(BISDetectorInternalDelegate _delegate) {
        delegate = _delegate;
    }

    static public ArrayList<String> getBeaconsInfoFromServer() {

        Log.i( TAG, "Getting beacon identifiers from BeaconsInSpace API" );

        String responseString = getBeaconsList();

        /*
         * Verify API returned JSON
         */
        if(responseString == null || responseString.length() == 0)
        {
            BISDetector.sharedInstance.onBeaconsInfoReceiveFail( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, null );
            return null;
        }

        /*
        *   As response we receive JSON.
        *   We need to parse it and get array of UUIDs of iBeacons to be monitored for this user.
        *
        *   JSON structure of response in format (key : value):
        *
        *   code        :   value (Int)
        *   message     :   value (String)
        *   data        :   value (Array of Dictionaries)
        *   |
        *   |-> single element of 'data'
        *       protocol    :   value (String)
        *       uuid        :   value (String)
        *
        */

        ArrayList<String> uuids = new ArrayList<>();

        try {
            JSONObject jsonObject = new JSONObject(responseString);

            /*
             * Verify status code of request
             */
            int responseCode = jsonObject.getInt("code");
            if ( responseCode != 200 )
            {
                BISDetector.sharedInstance.onBeaconsInfoReceiveFail( responseCode, null );
                return null;
            }

            JSONArray jsonArrayData = jsonObject.getJSONArray("data");
            if (jsonArrayData == null) return null;


            for (int i = 0; i < jsonArrayData.length(); i++) {

                JSONObject jsonObjectBeacon = jsonArrayData.getJSONObject(i);

                String protocol = jsonObjectBeacon.getString("protocol");

                if(protocol.equalsIgnoreCase("ibeacon")) {
                    String uuid = jsonObjectBeacon.getString("uuid");

                    if(uuid != null)
                        uuids.add(uuid);
                }
            }

//            Log.d(TAG, "Received array of UUIDs: " + uuids.toString());

            // If there are some uuids - start monitoring
            if (uuids.size() > 0) {
                BISDetector.beaconsManager().updateUUIDs(uuids);

                if(delegate != null)
                    delegate.onBeaconsInfoReceiveSuccess();
            }
        }
        catch (JSONException e) {
            Log.e(TAG, "JSONException: " + e.getMessage());
            BISDetector.sharedInstance.onBeaconsInfoReceiveFail( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, null );
        }

        return uuids;
    }

    static private String getBeaconsList() {

        // Perform URL call to BISDetector API

        String os = "";
        String device = "";
        String tz = "";
        String sdkVersion="";
        String language = "";
        String country = "";

        try {

            os = "ANDROID" + URLEncoder.encode(" " + Build.VERSION.RELEASE, "UTF-8");
            device = URLEncoder.encode(Build.BRAND + " " + Build.DEVICE, "UTF-8");
            tz = TimeZone.getDefault().getID();
            sdkVersion = BISDetector.SDK_VERSION;
            language = Locale.getDefault().toString();
            country = Locale.getDefault().toString();
        }
        catch (UnsupportedEncodingException e) {
            Log.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
        }

        String urlToCall = String.format(BISDetectorREST.BIS_URL_BEACON,
                BISDetector.UUID.toString(), "AUUID", BISDetector.ADID, "ADID", os, device, tz, language, country, sdkVersion, System.currentTimeMillis() / 1000.0);

        BISDetectorAsyncRequest urlCaller = new BISDetectorAsyncRequest();

        String responseString = null;

        try {
            urlCaller.execute(urlToCall);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "IllegalStateException: " + e.getMessage());
        }
        finally {

            try {
                responseString = urlCaller.get();
//                Log.d(TAG, "Response: " + responseString);
            }
            catch (InterruptedException e) {
                Log.e(TAG, "InterruptedException: " + e.getMessage());
            }
            catch (ExecutionException e) {
                Log.e(TAG, "ExecutionException: " + e.getMessage());
            }
        }

        return responseString;
    }


    static public String notifyAboutBeaconEnter(Beacon beacon) {
        return callEventURL(beacon, true);
    }

    static public String notifyAboutBeaconExit(Beacon beacon)
    {
        return callEventURL(beacon, false);
    }

    static private String callEventURL(Beacon beacon, boolean isEnter) {

        String[] beaconIds = BISDetectorManager.idsForBeacon(beacon);
        String beaconId = BISDetectorManager.uniqueIdentifierForBeacon(beacon);

        if(beaconIds == null || beaconIds.length < 3)
            return null;

        String tz = TimeZone.getDefault().getID();

        // Build data string
        String dataString = String.format("uuid=%s&major=%s&minor=%s&userId=%s&userIdType=%s&userId2=%s&userIdType2=%s&detect=%s&tz=%s&createdAt=%f",
                beaconIds[0], beaconIds[1], beaconIds[2],
                (BISDetector.UUID.toString() != null ? BISDetector.UUID.toString() : ""), "AUUID",
                (BISDetector.ADID != null ? BISDetector.ADID : ""), "ADID",
                (isEnter ? "enter" : "exit"),
                tz, System.currentTimeMillis()/1000.0);

        // Attempt to add GPS coordinates
        Location location = BISLocationListener.getLocationByBeaconId(beaconId);
        if ( location != null )
        {
            float latitude = (float) location.getLatitude();
            float longitude = (float) location.getLongitude();
            dataString += "&gpsLatitude="+latitude+"&gpsLongitude="+longitude;
        }

        final String finalDataString = dataString;
        final String[] resultStringData = new String[1];

//        Log.i( TAG, "URL STRING: "+ finalDataString );

        Thread thread = new Thread() {

            public void run() {
                resultStringData[0] = sendData(finalDataString);
//                Log.d(TAG, "JSON sent, response: " + resultStringData[0]);
            }
        };

        try
        {
            thread.start();
            thread.join();
        }
        catch( InterruptedException e )
        {
            Log.e( TAG, "callEventURL thread error:"+ e.getMessage() );
        }
        return resultStringData[0];
    }

    static String getAuthorizationHeader()
    {
        String USER_KEY = BISDetector.API_KEY;
        String PACKAGE_NAME = BISDetector.sharedInstance.context != null ? BISDetector.sharedInstance.context.getPackageName() : "";
        String authString = USER_KEY +":"+ PACKAGE_NAME;
        byte[] authBytes = authString.getBytes();
        String base64AuthString = Base64.encodeToString( authBytes, Base64.DEFAULT, authString.length(), Base64.NO_WRAP );
        String authorizationHeader = "Basic " + base64AuthString;
        return authorizationHeader;
    }

    static String sendData(String data) {

        if(data == null) { return null; }

        HttpURLConnection urlConnection;
        String url = BISDetectorREST.BIS_URL_EVENT;
        String result = null;

        // prepare auth header
        String authorizationHeader = getAuthorizationHeader();

        try {
            urlConnection = (HttpURLConnection) new URL(url).openConnection();
            urlConnection.setDoOutput(true);
            urlConnection.setRequestProperty("Authorization", authorizationHeader);
            urlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            urlConnection.setRequestProperty("Accept", "application/json");
            urlConnection.setRequestMethod("POST");
            urlConnection.connect();

            // Write
            OutputStream outputStream = urlConnection.getOutputStream();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outputStream, "UTF-8"));
            writer.write(data);
            writer.close();
            outputStream.close();

            // Read
            BufferedReader reader = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "UTF-8"));

            String line;
            StringBuilder sb = new StringBuilder();

            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }

            reader.close();
            result = sb.toString();
        }
        catch (UnsupportedEncodingException e) {
            Log.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
        }
        catch (IOException e) {
            Log.e(TAG, "IOException: " + e.getMessage());
        }

        return result;
    }
}
