package com.beaconsinspace.android.beacon.detector;

import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.RemoteException;
import android.util.Log;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.MonitorNotifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.startup.RegionBootstrap;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;


class BISDetectorManager implements BeaconConsumer {

    static private final String TAG = "BIS_BEACONS_MANAGER";
    static private final Long BEACON_EXIT_THRESHOLD = 30000l;

    static BISDetectorInternalDelegate delegate;

    Context context = null;
    BeaconManager beaconManager = null;

    ArrayList <Region> beaconsBeeingRanged = new ArrayList<>();
    ConcurrentHashMap<String, Long> beaconsInRangeMap = new ConcurrentHashMap<>();
    ConcurrentHashMap<String, ArrayList<Beacon>> beaconsInRangeByUUID = new ConcurrentHashMap<>();

    public ArrayList<String> uuids = new ArrayList<>();

    ArrayList <RegionBootstrap> bootstraps = new ArrayList<>();

    static public void setDelegate(BISDetectorInternalDelegate _delegate) {
        delegate = _delegate;
    }

    public boolean setContextAndInit(Context ctx) {

        if(ctx == null) {
            Log.e(TAG, "Context is null");
            return false;
        }

        context = ctx;

        if(beaconManager == null) {

            beaconManager = BeaconManager.getInstanceForApplication(context);

//            BeaconManager.setDebug(true);

//            beaconManager.setForegroundBetweenScanPeriod(1100l);
//            beaconManager.setBackgroundScanPeriod(1100l);
//            beaconManager.setBackgroundBetweenScanPeriod(5000l);

            String iBeaconLayout = "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24";
            beaconManager.getBeaconParsers().add(new BeaconParser(). setBeaconLayout(iBeaconLayout));

            String altBeaconLayout = "m:2-3=beac,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25";
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(altBeaconLayout));

//            String eddystoneBeaconLayout = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";
//            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(eddystoneBeaconLayout));

            doBind();
        }

        return true;
    }

    public void updateUUIDs(ArrayList<String> _uuids) {
        uuids = new ArrayList<>(_uuids);
//        startRanging();
    }

    public void startRanging()
    {
        // We need to stop ranging of old beacons
        stopRanging();
//        Log.d(TAG, "Starting Ranging of UUIDs: " + uuids.toString());
        doBind();
    }

    public void stopRanging()
    {
        try
        {
            for(Region region : beaconsBeeingRanged)
            {
                beaconManager.stopRangingBeaconsInRegion(region);
            }
            undoBind();
        }
        catch (RemoteException e)
        {
            Log.e(TAG, "RemoteException: " + e.toString());
        }
    }

    public void doBind() {

        if(beaconManager != null && !beaconManager.isBound(this)) {
            beaconManager.bind(this);
        }
    }

    public void undoBind() {

        if(beaconManager != null && beaconManager.isBound(this)) {
            beaconManager.unbind(this);
        }
    }

    @Override
    public void onBeaconServiceConnect() {

        beaconManager.setRangeNotifier(

                new RangeNotifier() {

                    @Override
                    public void didRangeBeaconsInRegion(Collection<Beacon> collection, org.altbeacon.beacon.Region region) {
//                        Log.d(TAG, "didRangeBeacons: " + collection + "\nInRegion: " + region.toString());

                        // 1. Check for EXIT event

                        String uuidRegion = region.getId1().toString();
                        ArrayList<Beacon> beaconsInRange = beaconsInRangeByUUID.get(uuidRegion);

                        if(beaconsInRange == null) {
                            beaconsInRange = new ArrayList<>();
                            beaconsInRangeByUUID.put(uuidRegion, beaconsInRange);
                        }

                        ArrayList<Beacon> _beaconsInRange = new ArrayList<>(beaconsInRange);
                        ArrayList<Beacon> _collection = new ArrayList<>(collection);

                        for (Beacon beaconInRange : _beaconsInRange) {

                            String uidBeaconInRange = BISDetectorManager.uniqueIdentifierForBeacon(beaconInRange);
//                            Log.d(TAG, "beaconInRange :: " + uidBeaconInRange);

                            boolean isFound = false;
                            Beacon foundBeacon = null;

                            for (Beacon beaconInCollection : _collection) {

                                String uidBeaconInCollection = BISDetectorManager.uniqueIdentifierForBeacon(beaconInCollection);
//                                Log.d(TAG, "beaconInCollection :: " + uidBeaconInCollection);

                                if(uidBeaconInRange.equalsIgnoreCase(uidBeaconInCollection)) {
//                                    Log.d(TAG, "Found");
                                    isFound = true;
                                    foundBeacon = beaconInRange;
                                    break;
                                }
                            }

                            if (isFound && foundBeacon != null)
                            {
                                // reset counter
                                Log.d(TAG, "beaconInRange :: " + uidBeaconInRange);
                                beaconsInRangeMap.put(uidBeaconInRange, System.currentTimeMillis());
                            }
                            else
                            {
                                // ranging beacon not found - compare counter to threshold and increment
//                                Log.d(TAG, "Beacon: " + uidBeaconInRange );
                                Long timeout = beaconsInRangeMap.get(uidBeaconInRange);
                                Long diff = timeout > 0 ? System.currentTimeMillis() - timeout : 0;

                                Log.d(TAG, "beaconOutOfRange: " + uidBeaconInRange + " ::: " + diff);

                                if(diff >= BEACON_EXIT_THRESHOLD) {

                                    beaconsInRange.remove(beaconInRange);
                                    beaconsInRangeMap.remove(uidBeaconInRange);
                                    handleBeaconExit(beaconInRange);
                                }
                            }
                        } // for (Beacon beaconInRange : _beaconsInRange)


                        // 2. Check for ENTER event
                        for(Beacon beaconInCollection : _collection) {

                            String uidBeaconInCollection = BISDetectorManager.uniqueIdentifierForBeacon(beaconInCollection);
//                            Log.d(TAG, "beaconInCollection :: " + uidBeaconInCollection);

                            boolean isKnown = false;

                            for ( Beacon beaconInRange : _beaconsInRange) {

                                String uidBeaconInRange = BISDetectorManager.uniqueIdentifierForBeacon(beaconInRange);

                                if(uidBeaconInCollection.equalsIgnoreCase(uidBeaconInRange)) {
//                                    Log.d(TAG, "FOUND");
                                    isKnown = true;
                                    break;
                                }
                            }

                            if(!isKnown) {

//                                Log.d(TAG, "NEW BEACON :: " + uidBeaconInCollection);

                                // add it to the list
                                beaconsInRange.add(beaconInCollection);
                                _beaconsInRange = new ArrayList<>(beaconsInRange);
                                beaconsInRangeMap.put(uidBeaconInCollection, 0l);
                                handleBeaconEnter(beaconInCollection);
                            }

                        } // for(Beacon beaconInCollection : _collection)
                    } // didRangeBeaconsInRegion
                } // new RangeNotifier()
        ); // beaconManager.setRangeNotifier(

        beaconManager.setMonitorNotifier(new MonitorNotifier() {
            @Override
            public void didEnterRegion(Region region) {
//                Log.e(TAG, "BEACON ENTER 2 ::: " + region.toString());
            }

            @Override
            public void didExitRegion(Region region) {
//                Log.e(TAG, "BEACON EXIT 2 ::: " + region.toString());
            }

            @Override
            public void didDetermineStateForRegion(int state, Region region) {
//                Log.e(TAG, "BEACON STATE 2 ::: " + region.toString() + " STATE : " + state);
            }
        });


        if(uuids.size() > 0) {

            beaconsBeeingRanged.clear();
            bootstraps.clear();

            try {

                for(int n = 0; n < uuids.size(); n++) {

                    String uuid = uuids.get(n);

                    Region region = new org.altbeacon.beacon.Region("BISDetector" + n, Identifier.parse(uuid.toLowerCase()), null, null);
                    beaconsBeeingRanged.add(region);
                    beaconManager.startRangingBeaconsInRegion(region);

//                    beaconManager.startMonitoringBeaconsInRegion(region);

//                    RegionBootstrap regionBootstrap = new RegionBootstrap(this, region);
//                    bootstraps.add(regionBootstrap);
                }
            }
            catch (RemoteException e) {
                Log.e(TAG, "RemoteException: " + e.toString());
            }

        } // if(uuids.size() > 0)
    }

    private void handleBeaconEnter(Beacon b) {

        final Beacon beacon = b;
        Thread thread = new Thread()
        {
            public void run()
            {
                //        Log.d(TAG, "BEACON EXIT ::: " + beacon.toString());
                String jsonDataString = BISDetectorREST.notifyAboutBeaconEnter(beacon);
                String beaconId = null;

                try
                {
                    JSONObject jsonDataObject = new JSONObject(jsonDataString);
                    int code = jsonDataObject.getInt("code");
                    if ( code != 200 )
                    {
                        Log.e(TAG,"Unsuccessful API Response occurred"+jsonDataString);
                        return;
                    }
                    beaconId = jsonDataObject.getJSONObject("data").getString("beaconId");
                }
                catch( JSONException e)
                {
                    Log.e(TAG,"Failed to parse JSON"+e.getMessage());
                }

                if( delegate != null && beaconId != null )
                {
                    BISLocationListener.assignLocationToBeaconId(uniqueIdentifierForBeacon(beacon));
                    delegate.onBeaconEnter(beaconId);
                }
            }
        };
        thread.start();
    }

    private void handleBeaconExit(Beacon b)
    {
        final Beacon beacon = b;
        Thread thread = new Thread()
        {
            public void run()
            {
                //        Log.d(TAG, "BEACON EXIT ::: " + beacon.toString());
                String jsonDataString = BISDetectorREST.notifyAboutBeaconExit(beacon);
                String beaconId = null;

                try
                {
                    JSONObject jsonDataObject = new JSONObject(jsonDataString);
                    int code = jsonDataObject.getInt("code");
                    if ( code != 200 )
                    {
                        Log.e(TAG,"Unsuccessful API Response occurred"+jsonDataString);
                        return;
                    }
                    beaconId = jsonDataObject.getJSONObject("data").getString("beaconId");
                }
                catch( JSONException e)
                {
                    Log.e(TAG,"Failed to parse JSON"+e.getMessage());
                }

                if( delegate != null && beaconId != null )
                {
                    delegate.onBeaconExit(beaconId);
                }
            }
        };
        thread.start();
    }

    static public String[] idsForBeacon(Beacon beacon) {

        if(beacon == null) {
            return new String[]{"", "", ""};
        }

        Identifier id1 = beacon.getIdentifiers().size() > 0 ? beacon.getId1() : null;
        Identifier id2 = beacon.getIdentifiers().size() > 1 ? beacon.getId2() : null;
        Identifier id3 = beacon.getIdentifiers().size() > 2 ? beacon.getId3() : null;

        String uuid = id1 != null ? id1.toString() : "";
        String major = id2 != null ? id2.toString() : "";
        String minor = id3 != null ? id3.toString() : "";

        return new String[]{uuid, major, minor};
    }

    static public String uniqueIdentifierForBeacon(Beacon beacon) {

        if(beacon == null) {
            return "";
        }

        String[] strings = idsForBeacon(beacon);

        String uuid = strings[0];
        String major = strings[1];
        String minor = strings[2];

        if(uuid.contains("0x"))
            uuid = uuid.replace("0x", "");
        if(major.contains("0x"))
            major = uuid.replace("0x", "");
        if(minor.contains("0x"))
            minor = uuid.replace("0x", "");

        String uniqueId = uuid + "_" + major + "_" + minor;

        return uuid+":"+uniqueId.hashCode();
    }


    @Override
    public Context getApplicationContext() { return context; }

    @Override
    public void unbindService(ServiceConnection serviceConnection) {
        context.unbindService(serviceConnection);
    }

    @Override
    public boolean bindService(Intent intent, ServiceConnection serviceConnection, int i) {
        return context.bindService(intent, serviceConnection, i);
    }
}
