package com.beaconsinspace.android.beacon.detector;

import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Handler;
import android.os.Looper;
import android.provider.Settings;
import android.util.Log;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.util.UUID;

public class BISDetector implements BISDetectorInternalDelegate {

    /*
     * VERSION
     */
    static public String SDK_VERSION = "1.1.2";

    /*
     * Definitions
     */
    static private final String TAG = "BIS_API";

    static String API_KEY = "";
    static String ADID = "";
    static UUID UUID;

    static BISDetector sharedInstance = new BISDetector();
    static Context context;
    static BISDetectorManager beaconsManager = new BISDetectorManager();
    static BISDetectorDelegate delegate;

    /**
     * Constructor for the BeaconsInSpace Detector
     *
     * @param key String
     * @param ctx Context
     * @param dlgt BISDetectorDelegate
     */
    static public void configure(String key, Context ctx, BISDetectorDelegate dlgt) {

        // Log occurrance
        Log.d(TAG, "Configuring BISDetector SDK " + SDK_VERSION);

        // 0. Set Delegate
        delegate = dlgt;

        // 1. Set API Key
        API_KEY = key != null ? key : "";

        // 2. Set Context
        context = ctx;
        beaconsManager.setContextAndInit(ctx);

        // 3. Set Internal Delegate
        BISDetectorInternalDelegate internalDelegate = sharedInstance;
        BISDetectorREST.setDelegate(internalDelegate);
        BISDetectorServicesListener.setDelegate(internalDelegate);
        BISDetectorManager.setDelegate(internalDelegate);

        // 4. Get UUID
        BISDeviceUUIDFactory UUIDFactory = new BISDeviceUUIDFactory(context);
        UUID = UUIDFactory.getDeviceUuid();

        // 4. Read Users Advertisement Identifier
        BISDetector.getUserADID();
    }

    static void performInitialSetup()
    {
        /*
         * Verify all required services are enabled
         */
        String errorMessage = "";
        if (!isLocationServiceEnabled())
        {
            errorMessage = "Location Services are not enabled. Please enable them in Settings.";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        if (!isInternetAvailable())
        {
            errorMessage = "Network Services are not enabled. Please enable them in Settings.";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        if (!isBluetoothEnabled())
        {
            errorMessage = "Bluetooth is not enabled. Please turn it on to proceed";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        /*
         * Get beacon identifiers
         */
        BISDetectorREST.getBeaconsInfoFromServer();
    }

    static BISDetectorManager beaconsManager() {
        return beaconsManager;
    }

    static void getUserADID() {

        if(context == null) { return; }

        Thread thread = new Thread() {

            public void run() {
                AdvertisingIdClient.Info adInfo = null;

                try {
                    adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                }
                catch (IOException | GooglePlayServicesRepairableException | GooglePlayServicesNotAvailableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                }
                finally {
                    if(adInfo == null) {
                        onUserADIDReceiveFail();
                    }
                    else {
                        ADID = adInfo.getId();
//                            final boolean isLAT = adInfo.isLimitAdTrackingEnabled();
                        final boolean success = ADID != null && ADID.length() > 0;

                        // NOTE: we need to post in to execute on UI thread, because user can have UI updates in his overload.
                        Handler handler = new Handler(Looper.getMainLooper());
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (success)
                                    onUserADIDReceiveSuccess();
                                else
                                    onUserADIDReceiveFail();
                            }
                        });
                    }
                } // finally
            } // Thread run()
        }; // new Thread()

        thread.start();
    }

    static public boolean isLocationServiceEnabled() {

        if(context == null)
            return false;

        int locationMode = 0;

        try {
            locationMode = Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.LOCATION_MODE);
        }
        catch (Settings.SettingNotFoundException e) {
            e.printStackTrace();
        }

        return (locationMode != Settings.Secure.LOCATION_MODE_OFF);
    }

    static public boolean isInternetAvailable() {

        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();

        return activeNetworkInfo != null && activeNetworkInfo.isConnected();
    }

    static public boolean isBluetoothEnabled() {

        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

        return adapter.isEnabled();
    }

    static public void startRanging() {
        beaconsManager.setContextAndInit(context);
        beaconsManager.startRanging();
    }

    static public void stopRanging() {
        beaconsManager.stopRanging();
    }

    static void onUserADIDReceiveSuccess() {
        BISDetector.performInitialSetup();
    }

    static void onUserADIDReceiveFail() {
        BISDetector.performInitialSetup();
    }

    @Override
    public void onBeaconsInfoReceiveSuccess() {
        Log.d(TAG, "BLE ranging has begun");
        startRanging();
    }

    @Override
    public void onBeaconsInfoReceiveFail( int errorCode, String errorMessage ) {
        if ( errorMessage == null ) {
            switch( errorCode )
            {
                case 401:
                    errorMessage = "Invalid Authentication";
                    break;
                default:
                    errorMessage = "An error occurred";
                    break;
            }
        }
        errorMessage = errorCode+" : "+errorMessage;
        Log.e(TAG, errorMessage);
        if ( delegate != null ) { delegate.onBISError( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, errorMessage ); }
    }

    @Override
    public void checkIfShouldUpdateBeaconsInfo() {
        if(isInternetAvailable() && beaconsManager.uuids.size() == 0) {
            BISDetectorREST.getBeaconsInfoFromServer();
        }
    }

    @Override
    public void onBeaconEnter(String beaconId) {
        if(delegate != null)
        {
            delegate.didEnterBISRegion(beaconId);
        }
    }

    @Override
    public void onBeaconExit(String beaconId) {
        if(delegate != null)
        {
            delegate.didExitBISRegion(beaconId);
        }
    }
}
