/*
 * Copyright (C) 2020 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.core_av.usb_camera.capturer.video.provider

import android.view.Surface
import com.bandyer.core_av.capturer.video.CaptureFrameListener
import com.bandyer.core_av.capturer.video.VideoFrame.Companion.create
import com.bandyer.core_av.capturer.video.dispatcher.FrameDispatcher
import com.bandyer.core_av.capturer.video.provider.FrameProvider
import com.bandyer.core_av.capturer.video.provider.FrameQuality
import com.bandyer.core_av.capturer.video.provider.fail
import com.bandyer.core_av.usb_camera.UsbData
import com.bandyer.core_av.usb_camera.internal.IFrameCallback
import com.bandyer.core_av.usb_camera.internal.UVCCamera
import java.nio.ByteBuffer


internal class BaseUsbFrameProvider(val usbData: UsbData) : UsbFrameProvider, IFrameCallback {

    override var frameQuality = FrameQuality(UVCCamera.DEFAULT_PREVIEW_WIDTH, UVCCamera.DEFAULT_PREVIEW_HEIGHT, UVCCamera.DEFAULT_PREVIEW_MIN_FPS)
        set(value) {
            if (field == value) return
            field = value
            camera?.setPreviewSize(value.width, value.height, value.fps)
        }

    private var camera: UVCCamera? = null

    private var frameDispatcher: FrameDispatcher? = null

    override fun onInit(frameDispatcher: FrameDispatcher) {
        this.frameDispatcher = frameDispatcher
        try {
            camera = UVCCamera().apply {
                setPreviewDisplay(frameDispatcher.surfaceTexture)
                setFrameCallback(this@BaseUsbFrameProvider)
                startPreview()
            }
            camera!!.connect(usbData)
        } catch (e: Exception) {
            // Log error
            fail(FrameProvider.Exception("Camera is already in use."))
            return
        }
    }

    override fun onStart(width: Int, height: Int, fps: Int) {
        frameDispatcher ?: return
        camera?.initialize(Surface(frameDispatcher!!.surfaceTexture), this@BaseUsbFrameProvider)
        frameQuality = FrameQuality(width, height, fps)
        camera?.startPreview()
    }

    @Throws(InterruptedException::class)
    override fun onStop() {
        camera?.stopPreview()
    }

    override fun onDispose() {
        camera?.destroy()
    }

    override fun onFailure(exception: FrameProvider.Exception) = onDispose()

    override fun captureFrame(onCaptureFrameListener: CaptureFrameListener) = Unit

    override fun onFrame(buffer: ByteBuffer) {
        val yuvFrame = create(buffer, frameQuality.width, frameQuality.height)
        frameDispatcher?.dispatch(yuvFrame)
    }

}