/*
 *  UVCCamera
 *  library and sample to access to UVC web camera on non-rooted Android device
 *
 * Copyright (c) 2014-2017 saki t_saki@serenegiant.com
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 *  All files in the folder are under this Apache License, Version 2.0.
 *  Files in the libjpeg-turbo, libusb, libuvc, rapidjson folder
 *  may have a different license, see the respective files.
 */
@file:Suppress("JniMissingFunction")

package com.bandyer.core_av.usb_camera.internal

import android.graphics.SurfaceTexture
import android.hardware.usb.UsbDevice
import android.text.TextUtils
import android.util.Log
import android.view.Surface
import android.view.SurfaceHolder
import com.bandyer.core_av.usb_camera.UsbData
import org.json.JSONException
import org.json.JSONObject
import java.util.*
import kotlin.math.max

/**
 * the constructor of this class should be call within the thread that has a looper
 * (UI thread or a thread that called Looper.prepare)
 */
internal class UVCCamera {

    companion object {
        private val DEBUG = false // TODO set false when releasing
        private val TAG = UVCCamera::class.java.simpleName
        private val DEFAULT_USBFS = "/dev/bus/usb"
        val DEFAULT_PREVIEW_WIDTH = 640
        val DEFAULT_PREVIEW_HEIGHT = 480
        val DEFAULT_PREVIEW_MIN_FPS = 1
        val DEFAULT_PREVIEW_MAX_FPS = 30
        val DEFAULT_BANDWIDTH = 1.0f

        //--------------------------------------------------------------------------------
        val CTRL_SCANNING = 0x00000001 // D0:  Scanning Mode
        val CTRL_AE = 0x00000002 // D1:  Auto-Exposure Mode
        val CTRL_AE_PRIORITY = 0x00000004 // D2:  Auto-Exposure Priority
        val CTRL_AE_ABS = 0x00000008 // D3:  Exposure Time (Absolute)
        val CTRL_AR_REL = 0x00000010 // D4:  Exposure Time (Relative)
        val CTRL_FOCUS_ABS = 0x00000020 // D5:  Focus (Absolute)
        val CTRL_FOCUS_REL = 0x00000040 // D6:  Focus (Relative)
        val CTRL_IRIS_ABS = 0x00000080 // D7:  Iris (Absolute)
        val CTRL_IRIS_REL = 0x00000100 // D8:  Iris (Relative)
        val CTRL_ZOOM_ABS = 0x00000200 // D9:  Zoom (Absolute)
        val CTRL_ZOOM_REL = 0x00000400 // D10: Zoom (Relative)
        val CTRL_PANTILT_ABS = 0x00000800 // D11: PanTilt (Absolute)
        val CTRL_PANTILT_REL = 0x00001000 // D12: PanTilt (Relative)
        val CTRL_ROLL_ABS = 0x00002000 // D13: Roll (Absolute)
        val CTRL_ROLL_REL = 0x00004000 // D14: Roll (Relative)
        val CTRL_FOCUS_AUTO = 0x00020000 // D17: Focus, Auto
        val CTRL_PRIVACY = 0x00040000 // D18: Privacy
        val CTRL_FOCUS_SIMPLE = 0x00080000 // D19: Focus, Simple
        val CTRL_WINDOW = 0x00100000 // D20: Window
        val PU_BRIGHTNESS = -0x7fffffff // D0: Brightness
        val PU_CONTRAST = -0x7ffffffe // D1: Contrast
        val PU_HUE = -0x7ffffffc // D2: Hue
        val PU_SATURATION = -0x7ffffff8 // D3: Saturation
        val PU_SHARPNESS = -0x7ffffff0 // D4: Sharpness
        val PU_GAMMA = -0x7fffffe0 // D5: Gamma
        val PU_WB_TEMP = -0x7fffffc0 // D6: White Balance Temperature
        val PU_WB_COMPO = -0x7fffff80 // D7: White Balance Component
        val PU_BACKLIGHT = -0x7fffff00 // D8: Backlight Compensation
        val PU_GAIN = -0x7ffffe00 // D9: Gain
        val PU_POWER_LF = -0x7ffffc00 // D10: Power Line Frequency
        val PU_HUE_AUTO = -0x7ffff800 // D11: Hue, Auto
        val PU_WB_TEMP_AUTO = -0x7ffff000 // D12: White Balance Temperature, Auto
        val PU_WB_COMPO_AUTO = -0x7fffe000 // D13: White Balance Component, Auto
        val PU_DIGITAL_MULT = -0x7fffc000 // D14: Digital Multiplier
        val PU_DIGITAL_LIMIT = -0x7fff8000 // D15: Digital Multiplier Limit
        val PU_AVIDEO_STD = -0x7fff0000 // D16: Analog Video Standard
        val PU_AVIDEO_LOCK = -0x7ffe0000 // D17: Analog Video Lock Status
        val PU_CONTRAST_AUTO = -0x7ffc0000 // D18: Contrast, Auto

        // uvc_status_class from libuvc.h
        val STATUS_CLASS_CONTROL = 0x10
        val STATUS_CLASS_CONTROL_CAMERA = 0x11
        val STATUS_CLASS_CONTROL_PROCESSING = 0x12

        // uvc_status_attribute from libuvc.h
        val STATUS_ATTRIBUTE_VALUE_CHANGE = 0x00
        val STATUS_ATTRIBUTE_INFO_CHANGE = 0x01
        val STATUS_ATTRIBUTE_FAILURE_CHANGE = 0x02
        val STATUS_ATTRIBUTE_UNKNOWN = 0xff
        private var isLoaded = false

        @Throws(JSONException::class)
        private fun addSize(format: JSONObject, formatType: Int, frameType: Int, size_list: MutableList<Size>) {
            val size = format.getJSONArray("size")
            val size_nums = size.length()
            for (j in 0 until size_nums) {
                val sz = size.getString(j).split("x".toRegex()).toTypedArray()
                try {
                    size_list.add(Size(formatType, frameType, j, sz[0].toInt(), sz[1].toInt()))
                } catch (e: Exception) {
                    break
                }
            }
        }

        private val SUPPORTS_CTRL = arrayOf(
                "D0:  Scanning Mode",
                "D1:  Auto-Exposure Mode",
                "D2:  Auto-Exposure Priority",
                "D3:  Exposure Time (Absolute)",
                "D4:  Exposure Time (Relative)",
                "D5:  Focus (Absolute)",
                "D6:  Focus (Relative)",
                "D7:  Iris (Absolute)",
                "D8:  Iris (Relative)",
                "D9:  Zoom (Absolute)",
                "D10: Zoom (Relative)",
                "D11: PanTilt (Absolute)",
                "D12: PanTilt (Relative)",
                "D13: Roll (Absolute)",
                "D14: Roll (Relative)",
                "D15: Reserved",
                "D16: Reserved",
                "D17: Focus, Auto",
                "D18: Privacy",
                "D19: Focus, Simple",
                "D20: Window",
                "D21: Region of Interest",
                "D22: Reserved, set to zero",
                "D23: Reserved, set to zero")
        private val SUPPORTS_PROC = arrayOf(
                "D0: Brightness",
                "D1: Contrast",
                "D2: Hue",
                "D3: Saturation",
                "D4: Sharpness",
                "D5: Gamma",
                "D6: White Balance Temperature",
                "D7: White Balance Component",
                "D8: Backlight Compensation",
                "D9: Gain",
                "D10: Power Line Frequency",
                "D11: Hue, Auto",
                "D12: White Balance Temperature, Auto",
                "D13: White Balance Component, Auto",
                "D14: Digital Multiplier",
                "D15: Digital Multiplier Limit",
                "D16: Analog Video Standard",
                "D17: Analog Video Lock Status",
                "D18: Contrast, Auto",
                "D19: Reserved. Set to zero",
                "D20: Reserved. Set to zero",
                "D21: Reserved. Set to zero",
                "D22: Reserved. Set to zero",
                "D23: Reserved. Set to zero")

        private fun dumpControls(controlSupports: Long) {
            Log.i(TAG, String.format("controlSupports=%x", controlSupports))
            for (i in SUPPORTS_CTRL.indices) {
                Log.i(TAG, SUPPORTS_CTRL[i] + if (controlSupports and (0x1 shl i).toLong() != 0L) "=enabled" else "=disabled")
            }
        }

        private fun dumpProc(procSupports: Long) {
            Log.i(TAG, String.format("procSupports=%x", procSupports))
            for (i in SUPPORTS_PROC.indices) {
                Log.i(TAG, SUPPORTS_PROC[i] + if (procSupports and (0x1 shl i).toLong() != 0L) "=enabled" else "=disabled")
            }
        }

        init {
            if (!isLoaded) {
                System.loadLibrary("UVCCamera")
                isLoaded = true
            }
        }
    }

    var usbData: UsbData? = null
        private set

    protected
    var mControlSupports // カメラコントロールでサポートしている機能フラグ
            : Long = 0
    var mProcSupports // プロセッシングユニットでサポートしている機能フラグ
            : Long = 0
    var mCurrentWidth: Int = DEFAULT_PREVIEW_WIDTH
    var mCurrentHeight: Int = DEFAULT_PREVIEW_HEIGHT
    var mCurrentBandwidthFactor: Float = DEFAULT_BANDWIDTH
    var mSupportedSize: String? = null
    var mCurrentSizeList: List<Size>? = null

    // these fields from here are accessed from native code and do not change name and remove
    var mNativePtr: Long = nativeCreate()
    val mScanningModeMin = 0
    val mScanningModeMax = 0
    val mScanningModeDef = 0
    val mExposureModeMin = 0
    val mExposureModeMax = 0
    val mExposureModeDef = 0
    val mExposurePriorityMin = 0
    val mExposurePriorityMax = 0
    val mExposurePriorityDef = 0
    val mExposureMin = 0
    val mExposureMax = 0
    val mExposureDef = 0
    val mAutoFocusMin = 0
    val mAutoFocusMax = 0
    val mAutoFocusDef = 0
    val mFocusMin = 0
    val mFocusMax = 0
    val mFocusDef = 0
    val mFocusRelMin = 0
    val mFocusRelMax = 0
    val mFocusRelDef = 0
    val mFocusSimpleMin = 0
    val mFocusSimpleMax = 0
    val mFocusSimpleDef = 0
    val mIrisMin = 0
    val mIrisMax = 0
    val mIrisDef = 0
    val mIrisRelMin = 0
    val mIrisRelMax = 0
    val mIrisRelDef = 0
    val mPanMin = 0
    val mPanMax = 0
    val mPanDef = 0
    val mTiltMin = 0
    val mTiltMax = 0
    val mTiltDef = 0
    val mRollMin = 0
    val mRollMax = 0
    val mRollDef = 0
    val mPanRelMin = 0
    val mPanRelMax = 0
    val mPanRelDef = 0
    val mTiltRelMin = 0
    val mTiltRelMax = 0
    val mTiltRelDef = 0
    val mRollRelMin = 0
    val mRollRelMax = 0
    val mRollRelDef = 0
    val mPrivacyMin = 0
    val mPrivacyMax = 0
    val mPrivacyDef = 0
    val mAutoWhiteBlanceMin = 0
    val mAutoWhiteBlanceMax = 0
    val mAutoWhiteBlanceDef = 0
    val mAutoWhiteBlanceCompoMin = 0
    val mAutoWhiteBlanceCompoMax = 0
    val mAutoWhiteBlanceCompoDef = 0
    val mWhiteBlanceMin = 0
    val mWhiteBlanceMax = 0
    val mWhiteBlanceDef = 0
    val mWhiteBlanceCompoMin = 0
    val mWhiteBlanceCompoMax = 0
    val mWhiteBlanceCompoDef = 0
    val mWhiteBlanceRelMin = 0
    val mWhiteBlanceRelMax = 0
    val mWhiteBlanceRelDef = 0
    val mBacklightCompMin = 0
    val mBacklightCompMax = 0
    val mBacklightCompDef = 0
    val mBrightnessMin = 0
    val mBrightnessMax = 0
    val mBrightnessDef = 0
    val mContrastMin = 0
    val mContrastMax = 0
    val mContrastDef = 0
    val mSharpnessMin = 0
    val mSharpnessMax = 0
    val mSharpnessDef = 0
    val mGainMin = 0
    val mGainMax = 0
    val mGainDef = 0
    val mGammaMin = 0
    val mGammaMax = 0
    val mGammaDef = 0
    val mSaturationMin = 0
    val mSaturationMax = 0
    val mSaturationDef = 0
    val mHueMin = 0
    val mHueMax = 0
    val mHueDef = 0
    val mZoomMin = 0
    val mZoomMax = 0
    val mZoomDef = 0
    val mZoomRelMin = 0
    val mZoomRelMax = 0
    val mZoomRelDef = 0
    val mPowerlineFrequencyMin = 0
    val mPowerlineFrequencyMax = 0
    val mPowerlineFrequencyDef = 0
    val mMultiplierMin = 0
    val mMultiplierMax = 0
    val mMultiplierDef = 0
    val mMultiplierLimitMin = 0
    val mMultiplierLimitMax = 0
    val mMultiplierLimitDef = 0
    val mAnalogVideoStandardMin = 0
    val mAnalogVideoStandardMax = 0
    val mAnalogVideoStandardDef = 0
    val mAnalogVideoLockStateMin = 0
    val mAnalogVideoLockStateMax = 0
    val mAnalogVideoLockStateDef = 0

    /**
     * connect to a UVC camera
     * USB permission is necessary before this method is called
     * @param usbData
     */
    @Synchronized
    fun connect(data: UsbData) {
        var result: Int
        try {
            usbData = data.clone()
            result = nativeConnect(mNativePtr,
                    usbData!!.venderId,
                    usbData!!.productId,
                    usbData!!.fileDescriptor,
                    usbData!!.busNum,
                    usbData!!.devNum,
                    getUSBFSName(usbData!!))
        } catch (e: Exception) {
            Log.w(TAG, e)
            result = -1
        }
        if (result != 0) {
            throw UnsupportedOperationException("open failed:result=$result")
        }
        if (mNativePtr != 0L && TextUtils.isEmpty(mSupportedSize)) {
            mSupportedSize = nativeGetSupportedSize(mNativePtr);
        }
        nativeSetPreviewSize(mNativePtr, DEFAULT_PREVIEW_WIDTH, DEFAULT_PREVIEW_HEIGHT, DEFAULT_PREVIEW_MIN_FPS, DEFAULT_PREVIEW_MAX_FPS, DEFAULT_BANDWIDTH);
    }

    @Synchronized
    fun initialize(surface: Surface, frameCallback: IFrameCallback?) {
        if (mNativePtr == 0L) return
        nativeInitialize(mNativePtr, surface, frameCallback)
    }

    /**
     * set status callback
     * @param callback
     */
    fun setStatusCallback(callback: IStatusCallback?) {
        if (mNativePtr == 0L) return
        nativeSetStatusCallback(mNativePtr, callback);
    }

    /**
     * set button callback
     * @param callback
     */
    fun setButtonCallback(callback: IButtonCallback?) {
        if (mNativePtr == 0L) return
        nativeSetButtonCallback(mNativePtr, callback)
    }

    @Synchronized
    fun getSupportedSize(): String? {
        return if (!TextUtils.isEmpty(mSupportedSize)) mSupportedSize else nativeGetSupportedSize(mNativePtr).also { mSupportedSize = it }
    }

    val previewSize: Size?
        get() {
            var result: Size? = null
            val list: List<Size> = supportedSizeList
            for (sz in list) {
                if (sz.width == mCurrentWidth
                        || sz.height == mCurrentHeight) {
                    result = sz
                    break
                }
            }
            return result
        }

    /**
     * Set preview size and preview mode
     * @param width
     * @param height
     * @param min_fps
     * @param max_fps
     * @param bandwidthFactor
     */
    fun setPreviewSize(width: Int, height: Int,
                       min_fps: Int = DEFAULT_PREVIEW_MIN_FPS,
                       max_fps: Int = DEFAULT_PREVIEW_MAX_FPS,
                       bandwidthFactor: Float = mCurrentBandwidthFactor) {
        require(!(width == 0 || height == 0)) { "invalid preview size" }
        if (mNativePtr == 0L) return
        val maxFps = max(min_fps, max_fps)
        val result = nativeSetPreviewSize(mNativePtr, width, height, min_fps, maxFps, bandwidthFactor)
        if (result != 0) throw IllegalArgumentException("Failed to set preview size");
        mCurrentWidth = width
        mCurrentHeight = height
        mCurrentBandwidthFactor = bandwidthFactor
    }

    val supportedSizeList: List<Size>
        get() {
            val result: MutableList<Size> = ArrayList()
            if (!TextUtils.isEmpty(mSupportedSize)) try {
                val json = JSONObject(mSupportedSize)
                val formats = json.getJSONArray("formats")
                val format_nums = formats.length()
                for (i in 0 until format_nums) {
                    val format = formats.getJSONObject(i)
                    if (format.has("type") && format.has("size")) {
                        val format_type = format.getInt("type")
                        if (format_type == 4) {
                            addSize(format, format_type, 0, result)
                        }
                    }
                }
            } catch (e: JSONException) {
                e.printStackTrace()
            }
            return result
        }


    /**
     * set preview surface with SurfaceHolder
     * you can use SurfaceHolder came from SurfaceView/GLSurfaceView
     * @param holder
     */
    @Synchronized
    fun setPreviewDisplay(holder: SurfaceHolder) = setPreviewDisplay(holder.surface)

    /**
     * set preview surface with SurfaceTexture.
     * this method require API >= 14
     * @param texture
     */
    @Synchronized
    fun setPreviewDisplay(texture: SurfaceTexture) = setPreviewDisplay(Surface(texture))

    /**
     * set preview surface with Surface
     * @param surface
     */
    @Synchronized
    fun setPreviewDisplay(surface: Surface) = nativeSetPreviewDisplay(mNativePtr, surface)

    /**
     * set frame callback
     * @param callback
     * @param pixelFormat
     */
    fun setFrameCallback(callback: IFrameCallback?) {
        if (mNativePtr == 0L) return
        nativeSetFrameCallback(mNativePtr, callback)
    }

    /**
     * start preview
     */
    @Synchronized
    fun startPreview() {
        if (mNativePtr == 0L) return
        nativeStartPreview(mNativePtr)
    }

    /**
     * stop preview
     */
    @Synchronized
    fun stopPreview() {
        usbData ?: return
        nativeStopPreview(mNativePtr)
    }

    /**
     * destroy UVCCamera object
     */
    @Synchronized
    fun destroy() {
        stopPreview()
        if (mNativePtr != 0L) {
            nativeDestroy(mNativePtr);
            mNativePtr = 0;    // nativeDestroyを呼ぶのでここでクリアしちゃダメ
        }
        usbData?.close()
        usbData = null
        mProcSupports = 0
        mControlSupports = mProcSupports
        mCurrentBandwidthFactor = 0f
        mSupportedSize = null
        mCurrentSizeList = null
        if (DEBUG) Log.v(TAG, "close:finished")
    }

    // wrong result may return when you call this just after camera open.
    // it is better to wait several hundreads millseconds.
    fun checkSupportFlag(flag: Long): Boolean {
        updateCameraParams()
        return if (flag and -0x80000000 == -0x80000000L) mProcSupports and flag == flag and 0x7ffffffF else mControlSupports and flag == flag
    }//    		nativeSetAutoFocus(mNativePtr, autoFocus);

    //================================================================================
    //    		result = nativeGetAutoFocus(mNativePtr) > 0;
    @get:Synchronized
    @set:Synchronized
    var autoFocus: Boolean
        get() {
            if (mNativePtr != 0L) return nativeGetAutoFocus(mNativePtr) > 0
            return true
        }
        set(autoFocus) {
            if (mNativePtr != 0L) {
                nativeSetAutoFocus(mNativePtr, autoFocus);
            }
        }
    //================================================================================
    /**
     * @param focus [%]
     */
    @Synchronized
    fun setFocus(focus: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mFocusMax - mFocusMin).toFloat()
            if (range > 0)
                nativeSetFocus(mNativePtr, (focus / 100f * range).toInt() + mFocusMin);
        }
    }

    /**
     * @param focus_abs
     * @return focus[%]
     */
    @Synchronized
    fun getFocus(focus_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateFocusLimit(mNativePtr);
            val range = Math.abs(mFocusMax - mFocusMin).toFloat()
            if (range > 0) {
                result = ((focus_abs - mFocusMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return focus[%]
     */
    @Synchronized
    open fun getFocus(): Int {
        return getFocus(nativeGetFocus(mNativePtr))
    }

    @Synchronized
    fun resetFocus() {
        if (mNativePtr != 0L) {
            nativeSetFocus(mNativePtr, mFocusDef);
        }
    }//    		nativeSetAutoWhiteBlance(mNativePtr, autoWhiteBlance);

    //================================================================================
    //    		result = nativeGetAutoWhiteBlance(mNativePtr) > 0;
    @get:Synchronized
    @set:Synchronized
    var autoWhiteBlance: Boolean
        get() {
            if (mNativePtr != 0L) return nativeGetAutoWhiteBlance(mNativePtr) > 0
            return true
        }
        set(autoWhiteBlance) {
            if (mNativePtr != 0L) {
                nativeSetAutoWhiteBlance(mNativePtr, autoWhiteBlance);
            }
        }
    //================================================================================
    /**
     * @param whiteBlance [%]
     */
    @Synchronized
    fun setWhiteBlance(whiteBlance: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mWhiteBlanceMax - mWhiteBlanceMin).toFloat()
            if (range > 0)
                nativeSetWhiteBlance(mNativePtr, (whiteBlance / 100f * range).toInt() + mWhiteBlanceMin);
        }
    }

    /**
     * @param whiteBlance_abs
     * @return whiteBlance[%]
     */
    @Synchronized
    fun getWhiteBlance(whiteBlance_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateWhiteBlanceLimit(mNativePtr);
            val range = Math.abs(mWhiteBlanceMax - mWhiteBlanceMin).toFloat()
            if (range > 0) {
                result = ((whiteBlance_abs - mWhiteBlanceMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return white blance[%]
     */
    @Synchronized
    open fun getWhiteBlance(): Int {
        return getFocus(nativeGetWhiteBlance(mNativePtr))
    }

    @Synchronized
    fun resetWhiteBlance() {
        if (mNativePtr != 0L) {
            nativeSetWhiteBlance(mNativePtr, mWhiteBlanceDef);
        }
    }
    //================================================================================
    /**
     * @param brightness [%]
     */
    @Synchronized
    fun setBrightness(brightness: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mBrightnessMax - mBrightnessMin).toFloat()
            if (range > 0)
                nativeSetBrightness(mNativePtr, (brightness / 100f * range).toInt() + mBrightnessMin);
        }
    }

    /**
     * @param brightness_abs
     * @return brightness[%]
     */
    @Synchronized
    fun getBrightness(brightness_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateBrightnessLimit(mNativePtr);
            val range = Math.abs(mBrightnessMax - mBrightnessMin).toFloat()
            if (range > 0) {
                result = ((brightness_abs - mBrightnessMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return brightness[%]
     */
    @Synchronized
    open fun getBrightness(): Int {
        return getBrightness(nativeGetBrightness(mNativePtr))
    }

    @Synchronized
    fun resetBrightness() {
        if (mNativePtr != 0L) {
            nativeSetBrightness(mNativePtr, mBrightnessDef);
        }
    }
    //================================================================================
    /**
     * @param contrast [%]
     */
    @Synchronized
    fun setContrast(contrast: Int) {
        if (mNativePtr != 0L) {
            nativeUpdateContrastLimit(mNativePtr);
            val range = Math.abs(mContrastMax - mContrastMin).toFloat()
            if (range > 0)
                nativeSetContrast(mNativePtr, (contrast / 100f * range).toInt() + mContrastMin);
        }
    }

    /**
     * @param contrast_abs
     * @return contrast[%]
     */
    @Synchronized
    fun getContrast(contrast_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            val range = Math.abs(mContrastMax - mContrastMin).toFloat()
            if (range > 0) {
                result = ((contrast_abs - mContrastMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return contrast[%]
     */
    @Synchronized
    fun getSharpness(sharpness_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateSharpnessLimit(mNativePtr);
            val range = Math.abs(mSharpnessMax - mSharpnessMin).toFloat()
            if (range > 0) {
                result = ((sharpness_abs - mSharpnessMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return sharpness[%]
     */
    @Synchronized
    open fun getSharpness(): Int {
        return getSharpness(nativeGetSharpness(mNativePtr))
    }

    @Synchronized
    fun resetSharpness() {
        if (mNativePtr != 0L) {
            nativeSetSharpness(mNativePtr, mSharpnessDef);
        }
    }
    //================================================================================
    /**
     * @param gain [%]
     */
    @Synchronized
    fun setGain(gain: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mGainMax - mGainMin).toFloat()
            if (range > 0)
                nativeSetGain(mNativePtr, (gain / 100f * range).toInt() + mGainMin);
        }
    }

    /**
     * @param gain_abs
     * @return gain[%]
     */
    @Synchronized
    fun getGain(gain_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateGainLimit(mNativePtr);
            val range = Math.abs(mGainMax - mGainMin).toFloat()
            if (range > 0) {
                result = ((gain_abs - mGainMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return gain[%]
     */
    @Synchronized
    open fun getGain(): Int {
        return getGain(nativeGetGain(mNativePtr))
    }

    @Synchronized
    fun resetGain() {
        if (mNativePtr != 0L) {
            nativeSetGain(mNativePtr, mGainDef);
        }
    }
    //================================================================================
    /**
     * @param gamma [%]
     */
    @Synchronized
    fun setGamma(gamma: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mGammaMax - mGammaMin).toFloat()
            if (range > 0)
                nativeSetGamma(mNativePtr, (gamma / 100f * range).toInt() + mGammaMin);
        }
    }

    /**
     * @param gamma_abs
     * @return gamma[%]
     */
    @Synchronized
    fun getGamma(gamma_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateGammaLimit(mNativePtr);
            val range = Math.abs(mGammaMax - mGammaMin).toFloat()
            if (range > 0) {
                result = ((gamma_abs - mGammaMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return gamma[%]
     */
    @Synchronized
    open fun getGamma(): Int {
        return getGamma(nativeGetGamma(mNativePtr))
    }

    @Synchronized
    fun resetGamma() {
        if (mNativePtr != 0L) {
            nativeSetGamma(mNativePtr, mGammaDef);
        }
    }
    //================================================================================
    /**
     * @param saturation [%]
     */
    @Synchronized
    open fun getContrast(): Int {
        return getContrast(nativeGetContrast(mNativePtr))
    }

    @Synchronized
    fun resetContrast() {
        if (mNativePtr != 0L) {
            nativeSetContrast(mNativePtr, mContrastDef);
        }
    }
    //================================================================================
    /**
     * @param sharpness [%]
     */
    @Synchronized
    fun setSharpness(sharpness: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mSharpnessMax - mSharpnessMin).toFloat()
            if (range > 0)
                nativeSetSharpness(mNativePtr, (sharpness / 100f * range).toInt() + mSharpnessMin);
        }
    }

    /**
     * @param sharpness_abs
     * @return sharpness[%]
     */

    fun setSaturation(saturation: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mSaturationMax - mSaturationMin).toFloat()
            if (range > 0)
                nativeSetSaturation(mNativePtr, (saturation / 100f * range).toInt() + mSaturationMin);
        }
    }

    /**
     * @param saturation_abs
     * @return saturation[%]
     */
    @Synchronized
    fun getSaturation(saturation_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateSaturationLimit(mNativePtr);
            val range = Math.abs(mSaturationMax - mSaturationMin).toFloat()
            if (range > 0) {
                result = ((saturation_abs - mSaturationMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return saturation[%]
     */
    @Synchronized
    open fun getSaturation(): Int {
        return getSaturation(nativeGetSaturation(mNativePtr))
    }

    @Synchronized
    fun resetSaturation() {
        if (mNativePtr != 0L) {
            nativeSetSaturation(mNativePtr, mSaturationDef);
        }
    }
    //================================================================================
    /**
     * @param hue [%]
     */
    @Synchronized
    fun setHue(hue: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mHueMax - mHueMin).toFloat()
            if (range > 0)
                nativeSetHue(mNativePtr, (hue / 100f * range).toInt() + mHueMin);
        }
    }

    /**
     * @param hue_abs
     * @return hue[%]
     */
    @Synchronized
    fun getHue(hue_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateHueLimit(mNativePtr);
            val range = Math.abs(mHueMax - mHueMin).toFloat()
            if (range > 0) {
                result = ((hue_abs - mHueMin) * 100f / range).toInt()
            }
        }
        return result
    }

    /**
     * @return hue[%]
     */
    @Synchronized
    open fun getHue(): Int {
        return getHue(nativeGetHue(mNativePtr))
    }

    @Synchronized
    fun resetHue() {
        if (mNativePtr != 0L) {
            nativeSetHue(mNativePtr, mSaturationDef);
        }
    }

    //================================================================================
    fun setPowerlineFrequency(frequency: Int) {
        if (mNativePtr != 0L)
            nativeSetPowerlineFrequency(mNativePtr, frequency);
    }

    fun getPowerlineFrequency(): Int {
        return nativeGetPowerlineFrequency(mNativePtr)
    }
    //================================================================================
    /**
     * this may not work well with some combination of camera and device
     * @param zoom [%]
     */
    @Synchronized
    fun setZoom(zoom: Int) {
        if (mNativePtr != 0L) {
            val range = Math.abs(mZoomMax - mZoomMin).toFloat()
            if (range > 0) {
                val z = (zoom / 100f * range).toInt() + mZoomMin
                // 			   Log.d(TAG, "setZoom:zoom=" + zoom + " ,value=" + z);
                nativeSetZoom(mNativePtr, z);
            }
        }
    }

    /**
     * @param zoom_abs
     * @return zoom[%]
     */
    @Synchronized
    fun getZoom(zoom_abs: Int): Int {
        var result = 0
        if (mNativePtr != 0L) {
            nativeUpdateZoomLimit(mNativePtr);
            val range = Math.abs(mZoomMax - mZoomMin).toFloat()
            if (range > 0) {
                result = ((zoom_abs - mZoomMin) * 100f / range).toInt()
            }
        }
        return result
    }


    /**
     * @return zoom[%]
     */
    @Synchronized
    fun getZoom(): Int {
        return getZoom(nativeGetZoom(mNativePtr))
    }

    @Synchronized
    fun resetZoom() {
        if (mNativePtr != 0L) {
            nativeSetZoom(mNativePtr, mZoomDef);
        }
    }

    //================================================================================
    @Synchronized
    fun updateCameraParams() {
        if (mNativePtr != 0L) {
            if (mControlSupports == 0L || mProcSupports == 0L) {
                // サポートしている機能フラグを取得
                if (mControlSupports == 0L)
                    mControlSupports = nativeGetCtrlSupports(mNativePtr);
                if (mProcSupports == 0L)
                    mProcSupports = nativeGetProcSupports(mNativePtr);
//                 設定値を取得
                if ((mControlSupports != 0L) && (mProcSupports != 0L)) {
                    nativeUpdateBrightnessLimit(mNativePtr);
                    nativeUpdateContrastLimit(mNativePtr);
                    nativeUpdateSharpnessLimit(mNativePtr);
                    nativeUpdateGainLimit(mNativePtr);
                    nativeUpdateGammaLimit(mNativePtr);
                    nativeUpdateSaturationLimit(mNativePtr);
                    nativeUpdateHueLimit(mNativePtr);
                    nativeUpdateZoomLimit(mNativePtr);
                    nativeUpdateWhiteBlanceLimit(mNativePtr);
                    nativeUpdateFocusLimit(mNativePtr);
                }
                if (DEBUG) {
                    dumpControls(mControlSupports)
                    dumpProc(mProcSupports)
                    Log.v(TAG, String.format("Brightness:min=%d,max=%d,def=%d", mBrightnessMin, mBrightnessMax, mBrightnessDef))
                    Log.v(TAG, String.format("Contrast:min=%d,max=%d,def=%d", mContrastMin, mContrastMax, mContrastDef))
                    Log.v(TAG, String.format("Sharpness:min=%d,max=%d,def=%d", mSharpnessMin, mSharpnessMax, mSharpnessDef))
                    Log.v(TAG, String.format("Gain:min=%d,max=%d,def=%d", mGainMin, mGainMax, mGainDef))
                    Log.v(TAG, String.format("Gamma:min=%d,max=%d,def=%d", mGammaMin, mGammaMax, mGammaDef))
                    Log.v(TAG, String.format("Saturation:min=%d,max=%d,def=%d", mSaturationMin, mSaturationMax, mSaturationDef))
                    Log.v(TAG, String.format("Hue:min=%d,max=%d,def=%d", mHueMin, mHueMax, mHueDef))
                    Log.v(TAG, String.format("Zoom:min=%d,max=%d,def=%d", mZoomMin, mZoomMax, mZoomDef))
                    Log.v(TAG, String.format("WhiteBlance:min=%d,max=%d,def=%d", mWhiteBlanceMin, mWhiteBlanceMax, mWhiteBlanceDef))
                    Log.v(TAG, String.format("Focus:min=%d,max=%d,def=%d", mFocusMin, mFocusMax, mFocusDef))
                }
            }
        } else {
            mProcSupports = 0
            mControlSupports = mProcSupports
        }
    }

    fun getUSBFSName(usbData: UsbData): String? {
        var result: String? = null
        val name = usbData.deviceName
        val v = if (!TextUtils.isEmpty(name)) name.split("/".toRegex()).toTypedArray() else null
        if (v != null && v.size > 2) {
            val sb = StringBuilder(v[0])
            for (i in 1 until v.size - 2) sb.append("/").append(v[i])
            result = sb.toString()
        }
        if (TextUtils.isEmpty(result)) {
            Log.w(TAG, "failed to get USBFS path, try to use default path:$name")
            result = DEFAULT_USBFS
        }
        return result
    }
    //**********************************************************************
    /**
     * start movie capturing(this should call while previewing)
     * @param surface
     */
    fun startCapture(surface: Surface?) {
        if (this.usbData != null && surface != null) {
            nativeSetCaptureDisplay(mNativePtr, surface);
        } else throw NullPointerException("startCapture")
    }

    /**
     * stop movie capturing
     */
    fun stopCapture() {
        if (this.usbData != null) {
            nativeSetCaptureDisplay(mNativePtr, null);
        }
    }

    // until here


    // #nativeCreate and #nativeDestroy are not static methods.
    private external fun nativeCreate(): Long
    private external fun nativeInitialize(id_camera: Long, surface: Surface?, callback: IFrameCallback?): Int
    private external fun nativeDestroy(id_camera: Long)

    private external fun nativeConnect(id_camera: Long, venderId: Int, productId: Int, fileDescriptor: Int, busNum: Int, devAddr: Int, usbfs: String?): Int
    private external fun nativeRelease(id_camera: Long): Int

    private external fun nativeSetStatusCallback(mNativePtr: Long, callback: IStatusCallback?): Int
    private external fun nativeSetButtonCallback(mNativePtr: Long, callback: IButtonCallback?): Int

    private external fun nativeSetPreviewSize(id_camera: Long, width: Int, height: Int, min_fps: Int, max_fps: Int, bandwidth: Float): Int
    private external fun nativeGetSupportedSize(id_camera: Long): String?
    private external fun nativeStartPreview(id_camera: Long): Int
    private external fun nativeStopPreview(id_camera: Long): Int
    private external fun nativeSetPreviewDisplay(id_camera: Long, surface: Surface?): Int
    private external fun nativeSetFrameCallback(id_camera: Long, callback: IFrameCallback?): Int

    private external fun nativeSetCaptureDisplay(id_camera: Long, surface: Surface?): Int

    private external fun nativeGetCtrlSupports(id_camera: Long): Long
    private external fun nativeGetProcSupports(id_camera: Long): Long

    private external fun nativeUpdateScanningModeLimit(id_camera: Long): Int
    private external fun nativeSetScanningMode(id_camera: Long, scanning_mode: Int): Int
    private external fun nativeGetScanningMode(id_camera: Long): Int

    private external fun nativeUpdateExposureModeLimit(id_camera: Long): Int
    private external fun nativeSetExposureMode(id_camera: Long, exposureMode: Int): Int
    private external fun nativeGetExposureMode(id_camera: Long): Int

    private external fun nativeUpdateExposurePriorityLimit(id_camera: Long): Int
    private external fun nativeSetExposurePriority(id_camera: Long, priority: Int): Int
    private external fun nativeGetExposurePriority(id_camera: Long): Int

    private external fun nativeUpdateExposureLimit(id_camera: Long): Int
    private external fun nativeSetExposure(id_camera: Long, exposure: Int): Int
    private external fun nativeGetExposure(id_camera: Long): Int

    private external fun nativeUpdateExposureRelLimit(id_camera: Long): Int
    private external fun nativeSetExposureRel(id_camera: Long, exposure_rel: Int): Int
    private external fun nativeGetExposureRel(id_camera: Long): Int

    private external fun nativeUpdateAutoFocusLimit(id_camera: Long): Int
    private external fun nativeSetAutoFocus(id_camera: Long, autofocus: Boolean): Int
    private external fun nativeGetAutoFocus(id_camera: Long): Int

    private external fun nativeUpdateFocusLimit(id_camera: Long): Int
    private external fun nativeSetFocus(id_camera: Long, focus: Int): Int
    private external fun nativeGetFocus(id_camera: Long): Int

    private external fun nativeUpdateFocusRelLimit(id_camera: Long): Int
    private external fun nativeSetFocusRel(id_camera: Long, focus_rel: Int): Int
    private external fun nativeGetFocusRel(id_camera: Long): Int

    private external fun nativeUpdateIrisLimit(id_camera: Long): Int
    private external fun nativeSetIris(id_camera: Long, iris: Int): Int
    private external fun nativeGetIris(id_camera: Long): Int

    private external fun nativeUpdateIrisRelLimit(id_camera: Long): Int
    private external fun nativeSetIrisRel(id_camera: Long, iris_rel: Int): Int
    private external fun nativeGetIrisRel(id_camera: Long): Int

    private external fun nativeUpdatePanLimit(id_camera: Long): Int
    private external fun nativeSetPan(id_camera: Long, pan: Int): Int
    private external fun nativeGetPan(id_camera: Long): Int

    private external fun nativeUpdatePanRelLimit(id_camera: Long): Int
    private external fun nativeSetPanRel(id_camera: Long, pan_rel: Int): Int
    private external fun nativeGetPanRel(id_camera: Long): Int

    private external fun nativeUpdateTiltLimit(id_camera: Long): Int
    private external fun nativeSetTilt(id_camera: Long, tilt: Int): Int
    private external fun nativeGetTilt(id_camera: Long): Int

    private external fun nativeUpdateTiltRelLimit(id_camera: Long): Int
    private external fun nativeSetTiltRel(id_camera: Long, tilt_rel: Int): Int
    private external fun nativeGetTiltRel(id_camera: Long): Int

    private external fun nativeUpdateRollLimit(id_camera: Long): Int
    private external fun nativeSetRoll(id_camera: Long, roll: Int): Int
    private external fun nativeGetRoll(id_camera: Long): Int

    private external fun nativeUpdateRollRelLimit(id_camera: Long): Int
    private external fun nativeSetRollRel(id_camera: Long, roll_rel: Int): Int
    private external fun nativeGetRollRel(id_camera: Long): Int

    private external fun nativeUpdateAutoWhiteBlanceLimit(id_camera: Long): Int
    private external fun nativeSetAutoWhiteBlance(id_camera: Long, autoWhiteBlance: Boolean): Int
    private external fun nativeGetAutoWhiteBlance(id_camera: Long): Int

    private external fun nativeUpdateAutoWhiteBlanceCompoLimit(id_camera: Long): Int
    private external fun nativeSetAutoWhiteBlanceCompo(id_camera: Long, autoWhiteBlanceCompo: Boolean): Int
    private external fun nativeGetAutoWhiteBlanceCompo(id_camera: Long): Int

    private external fun nativeUpdateWhiteBlanceLimit(id_camera: Long): Int
    private external fun nativeSetWhiteBlance(id_camera: Long, whiteBlance: Int): Int
    private external fun nativeGetWhiteBlance(id_camera: Long): Int

    private external fun nativeUpdateWhiteBlanceCompoLimit(id_camera: Long): Int
    private external fun nativeSetWhiteBlanceCompo(id_camera: Long, whiteBlance_compo: Int): Int
    private external fun nativeGetWhiteBlanceCompo(id_camera: Long): Int

    private external fun nativeUpdateBacklightCompLimit(id_camera: Long): Int
    private external fun nativeSetBacklightComp(id_camera: Long, backlight_comp: Int): Int
    private external fun nativeGetBacklightComp(id_camera: Long): Int

    private external fun nativeUpdateBrightnessLimit(id_camera: Long): Int
    private external fun nativeSetBrightness(id_camera: Long, brightness: Int): Int
    private external fun nativeGetBrightness(id_camera: Long): Int

    private external fun nativeUpdateContrastLimit(id_camera: Long): Int
    private external fun nativeSetContrast(id_camera: Long, contrast: Int): Int
    private external fun nativeGetContrast(id_camera: Long): Int

    private external fun nativeUpdateAutoContrastLimit(id_camera: Long): Int
    private external fun nativeSetAutoContrast(id_camera: Long, autocontrast: Boolean): Int
    private external fun nativeGetAutoContrast(id_camera: Long): Int

    private external fun nativeUpdateSharpnessLimit(id_camera: Long): Int
    private external fun nativeSetSharpness(id_camera: Long, sharpness: Int): Int
    private external fun nativeGetSharpness(id_camera: Long): Int

    private external fun nativeUpdateGainLimit(id_camera: Long): Int
    private external fun nativeSetGain(id_camera: Long, gain: Int): Int
    private external fun nativeGetGain(id_camera: Long): Int

    private external fun nativeUpdateGammaLimit(id_camera: Long): Int
    private external fun nativeSetGamma(id_camera: Long, gamma: Int): Int
    private external fun nativeGetGamma(id_camera: Long): Int

    private external fun nativeUpdateSaturationLimit(id_camera: Long): Int
    private external fun nativeSetSaturation(id_camera: Long, saturation: Int): Int
    private external fun nativeGetSaturation(id_camera: Long): Int

    private external fun nativeUpdateHueLimit(id_camera: Long): Int
    private external fun nativeSetHue(id_camera: Long, hue: Int): Int
    private external fun nativeGetHue(id_camera: Long): Int

    private external fun nativeUpdateAutoHueLimit(id_camera: Long): Int
    private external fun nativeSetAutoHue(id_camera: Long, autohue: Boolean): Int
    private external fun nativeGetAutoHue(id_camera: Long): Int

    private external fun nativeUpdatePowerlineFrequencyLimit(id_camera: Long): Int
    private external fun nativeSetPowerlineFrequency(id_camera: Long, frequency: Int): Int
    private external fun nativeGetPowerlineFrequency(id_camera: Long): Int

    private external fun nativeUpdateZoomLimit(id_camera: Long): Int
    private external fun nativeSetZoom(id_camera: Long, zoom: Int): Int
    private external fun nativeGetZoom(id_camera: Long): Int

    private external fun nativeUpdateZoomRelLimit(id_camera: Long): Int
    private external fun nativeSetZoomRel(id_camera: Long, zoom_rel: Int): Int
    private external fun nativeGetZoomRel(id_camera: Long): Int

    private external fun nativeUpdateDigitalMultiplierLimit(id_camera: Long): Int
    private external fun nativeSetDigitalMultiplier(id_camera: Long, multiplier: Int): Int
    private external fun nativeGetDigitalMultiplier(id_camera: Long): Int

    private external fun nativeUpdateDigitalMultiplierLimitLimit(id_camera: Long): Int
    private external fun nativeSetDigitalMultiplierLimit(id_camera: Long, multiplier_limit: Int): Int
    private external fun nativeGetDigitalMultiplierLimit(id_camera: Long): Int

    private external fun nativeUpdateAnalogVideoStandardLimit(id_camera: Long): Int
    private external fun nativeSetAnalogVideoStandard(id_camera: Long, standard: Int): Int
    private external fun nativeGetAnalogVideoStandard(id_camera: Long): Int

    private external fun nativeUpdateAnalogVideoLockStateLimit(id_camera: Long): Int
    private external fun nativeSetAnalogVideoLoackState(id_camera: Long, state: Int): Int
    private external fun nativeGetAnalogVideoLoackState(id_camera: Long): Int

    private external fun nativeUpdatePrivacyLimit(id_camera: Long): Int
    private external fun nativeSetPrivacy(id_camera: Long, privacy: Boolean): Int
    private external fun nativeGetPrivacy(id_camera: Long): Int
}