/*
 * Copyright (C) 2020 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.core_av.usb_camera.internal

import android.content.Context
import android.hardware.usb.UsbDevice
import android.hardware.usb.UsbDeviceConnection
import android.hardware.usb.UsbInterface
import android.hardware.usb.UsbManager
import android.os.Build
import android.text.TextUtils
import android.util.Log
import android.util.SparseArray
import androidx.annotation.RequiresApi
import com.bandyer.core_av.usb_camera.BuildConfig.DEBUG
import com.bandyer.core_av.usb_camera.UsbData
import com.bandyer.core_av.usb_camera.UsbDeviceInfo
import java.lang.ref.WeakReference
import java.util.*

/**
 * control class
 * never reuse the instance when it closed
 */
internal class BaseUsbData : UsbData {

    companion object {
        const val TAG = "UsbData"
    }

    private val mWeakMonitor: WeakReference<BaseUsbConnector>
    private val mWeakDevice: WeakReference<UsbDevice>

    /**
     * get UsbDeviceConnection
     *
     * @return
     */
    @get:Synchronized
    var connection: UsbDeviceConnection?
        protected set
    protected val mInfo: UsbDeviceInfo
    override val busNum: Int
    override val devNum: Int
    private val mInterfaces = SparseArray<SparseArray<UsbInterface>>()

    /**
     * this class needs permission to access USB device before constructing
     *
     * @param monitor
     * @param device
     */
    internal constructor(context: Context, monitor: BaseUsbConnector, device: UsbDevice) {
        if (DEBUG) Log.i(TAG, "UsbControlBlock:constructor")
        mWeakMonitor = WeakReference(monitor)
        mWeakDevice = WeakReference(device)
        val mUsbManager = context.getSystemService(Context.USB_SERVICE) as UsbManager
        connection = mUsbManager.openDevice(device)
        mInfo = monitor.getDeviceInfo(device)
        val name = device.deviceName
        val v = if (!TextUtils.isEmpty(name)) name.split("/".toRegex()).toTypedArray() else null
        var busnum = 0
        var devnum = 0
        if (v != null) {
            busnum = v[v.size - 2].toInt()
            devnum = v[v.size - 1].toInt()
        }
        busNum = busnum
        devNum = devnum
        //			if (DEBUG) {
        if (connection != null) {
            val desc = connection!!.fileDescriptor
            val rawDesc = connection!!.rawDescriptors
            Log.i(TAG, String.format(Locale.US, "name=%s,desc=%d,busnum=%d,devnum=%d,rawDesc=", name, desc, busnum, devnum) + rawDesc)
        } else {
            Log.e(TAG, "could not connect to device $name")
        }
        //			}
    }

    /**
     * copy constructor
     *
     * @param src
     * @throws IllegalStateException
     */
    private constructor(src: BaseUsbData) {
        val monitor = src.uSBMonitor
        val device = src.device ?: throw IllegalStateException("device may already be removed")
        connection = monitor!!.mUsbManager.openDevice(device)
        if (connection == null) {
            throw IllegalStateException("device may already be removed or have no permission")
        }
        mInfo = monitor.getDeviceInfo(device)
        mWeakMonitor = WeakReference(monitor)
        mWeakDevice = WeakReference(device)
        busNum = src.busNum
        devNum = src.devNum
        // FIXME USBMonitor.musbDatasに追加する(今はHashMapなので追加すると置き換わってしまうのでだめ, ListかHashMapにListをぶら下げる?)
    }

    /**
     * duplicate by clone
     * need permission
     * USBMonitor never handle cloned UsbControlBlock, you should release it after using it.
     *
     * @return
     * @throws CloneNotSupportedException
     */
    @Throws(CloneNotSupportedException::class)
    public override fun clone(): BaseUsbData {
        val usbData: BaseUsbData
        try {
            usbData = BaseUsbData(this)
        } catch (e: IllegalStateException) {
            throw CloneNotSupportedException(e.message)
        }
        return usbData
    }

    private val uSBMonitor: BaseUsbConnector?
        get() = mWeakMonitor.get()

    val device: UsbDevice?
        get() = mWeakDevice.get()

    /**
     * get device name
     *
     * @return
     */
    override val deviceName: String
        get() {
            val device = mWeakDevice.get()
            return device?.deviceName ?: ""
        }

    /**
     * get device id
     *
     * @return
     */
    val deviceId: Int
        get() {
            val device = mWeakDevice.get()
            return device?.deviceId ?: 0
        }

    /**
     * get device key string
     *
     * @return same value if the devices has same vendor id, product id, device class, device subclass and device protocol
     */
    val deviceKeyName: String
        get() = BaseUsbConnector.getDeviceKeyName(mWeakDevice.get())

    /**
     * get device key string
     *
     * @param useNewAPI if true, try to use serial number
     * @return
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun getDeviceKeyName(useNewAPI: Boolean): String {
        if (useNewAPI) checkConnection()
        return BaseUsbConnector.getDeviceKeyName(mWeakDevice.get(), mInfo.serial, useNewAPI)
    }

    /**
     * get device key
     *
     * @return
     * @throws IllegalStateException
     */
    @get:Throws(IllegalStateException::class)
    val deviceKey: Int
        get() {
            checkConnection()
            return BaseUsbConnector.getDeviceKey(mWeakDevice.get())
        }

    /**
     * get device key
     *
     * @param useNewAPI if true, try to use serial number
     * @return
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun getDeviceKey(useNewAPI: Boolean): Int {
        if (useNewAPI) checkConnection()
        return BaseUsbConnector.getDeviceKey(mWeakDevice.get(), mInfo.serial, useNewAPI)
    }

    /**
     * get device key string
     * if device has serial number, use it
     *
     * @return
     */
    val deviceKeyNameWithSerial: String
        get() = BaseUsbConnector.getDeviceKeyName(mWeakDevice.get(), mInfo.serial, false)

    /**
     * get device key
     * if device has serial number, use it
     *
     * @return
     */
    val deviceKeyWithSerial: Int
        get() = deviceKeyNameWithSerial.hashCode()

    /**
     * get file descriptor to access USB device
     *
     * @return
     * @throws IllegalStateException
     */
    @get:Throws(IllegalStateException::class)
    @get:Synchronized
    override val fileDescriptor: Int
        get() {
            checkConnection()
            return connection!!.fileDescriptor
        }

    /**
     * get raw descriptor for the USB device
     *
     * @return
     * @throws IllegalStateException
     */
    @get:Throws(IllegalStateException::class)
    @get:Synchronized
    val rawDescriptors: ByteArray
        get() {
            checkConnection()
            return connection!!.rawDescriptors
        }

    /**
     * get vendor id
     *
     * @return
     */
    override val venderId: Int
        get() {
            val device = mWeakDevice.get()
            return device?.vendorId ?: 0
        }

    /**
     * get product id
     *
     * @return
     */
    override val productId: Int
        get() {
            val device = mWeakDevice.get()
            return device?.productId ?: 0
        }

    /**
     * get version string of USB
     *
     * @return
     */
    val usbVersion: String?
        get() = mInfo.usb_version

    /**
     * get manufacture
     *
     * @return
     */
    val manufacture: String?
        get() = mInfo.manufacturer

    /**
     * get product name
     *
     * @return
     */
    val productName: String?
        get() = mInfo.productName

    /**
     * get version
     *
     * @return
     */
    val version: String?
        get() = mInfo.version

    /**
     * get serial number
     *
     * @return
     */
    val serial: String?
        get() = mInfo.serial

    /**
     * get interface
     *
     * @param interface_id
     * @throws IllegalStateException
     */
    @Synchronized
    @Throws(IllegalStateException::class)
    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun getInterface(interface_id: Int): UsbInterface? {
        return getInterface(interface_id, 0)
    }

    /**
     * get interface
     *
     * @param interface_id
     * @param altsetting
     * @return
     * @throws IllegalStateException
     */
    @Synchronized
    @Throws(IllegalStateException::class)
    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun getInterface(interface_id: Int, altsetting: Int): UsbInterface? {
        checkConnection()
        var intfs = mInterfaces[interface_id]
        if (intfs == null) {
            intfs = SparseArray()
            mInterfaces.put(interface_id, intfs)
        }
        var intf = intfs[altsetting]
        if (intf == null) {
            val device = mWeakDevice.get()
            val n = device!!.interfaceCount
            for (i in 0 until n) {
                val temp = device.getInterface(i)
                if ((temp.id == interface_id) && (temp.alternateSetting == altsetting)) {
                    intf = temp
                    break
                }
            }
            if (intf != null) {
                intfs.append(altsetting, intf)
            }
        }
        return intf
    }

    /**
     * open specific interface
     *
     * @param intf
     */
    @Synchronized
    fun claimInterface(intf: UsbInterface?) {
        claimInterface(intf, true)
    }

    @Synchronized
    fun claimInterface(intf: UsbInterface?, force: Boolean) {
        checkConnection()
        connection!!.claimInterface(intf, force)
    }

    /**
     * close interface
     *
     * @param intf
     * @throws IllegalStateException
     */
    @Synchronized
    @Throws(IllegalStateException::class)
    fun releaseInterface(intf: UsbInterface) {
        checkConnection()
        val intfs = mInterfaces[intf.id]
        if (intfs != null) {
            val index = intfs.indexOfValue(intf)
            intfs.removeAt(index)
            if (intfs.size() == 0) {
                mInterfaces.remove(intf.id)
            }
        }
        connection!!.releaseInterface(intf)
    }

    /**
     * Close device
     * This also close interfaces if they are opened in Java side
     */
    @Synchronized
    override fun close() {
        if (DEBUG) Log.i(TAG, "UsbControlBlock#close:")
        if (connection != null) {
            val n = mInterfaces.size()
            for (i in 0 until n) {
                val intfs = mInterfaces.valueAt(i)
                if (intfs != null) {
                    val m = intfs.size()
                    for (j in 0 until m) {
                        val intf = intfs.valueAt(j)
                        connection!!.releaseInterface(intf)
                    }
                    intfs.clear()
                }
            }
            mInterfaces.clear()
            connection!!.close()
            connection = null
            val monitor = mWeakMonitor.get() ?: return
            device ?: return
            if (monitor.musbDatas.isEmpty()) return
            monitor.musbDatas.remove(device!!)
            monitor.mOnDeviceConnectListener?.onDisconnect(device!!, this@BaseUsbData)
        }
    }

    override fun equals(other: Any?): Boolean {
        if (other == null) return false
        if (other is BaseUsbData) {
            val device = other.device
            return if (device == null) mWeakDevice.get() == null else (device == mWeakDevice.get())
        } else if (other is UsbDevice) {
            return (other == mWeakDevice.get())
        }
        return super.equals(other)
    }

    @Synchronized
    @Throws(IllegalStateException::class)
    private fun checkConnection() {
        if (connection == null) {
            throw IllegalStateException("already closed")
        }
    }

    override fun hashCode(): Int {
        return mWeakDevice.hashCode()
    }
}