/*
 *  UVCCamera
 *  library and sample to access to UVC web camera on non-rooted Android device
 *
 * Copyright (c) 2014-2017 saki t_saki@serenegiant.com
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 *  All files in the folder are under this Apache License, Version 2.0.
 *  Files in the libjpeg-turbo, libusb, libuvc, rapidjson folder
 *  may have a different license, see the respective files.
 */
package com.bandyer.core_av.usb_camera.internal

import android.annotation.SuppressLint
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.hardware.usb.UsbConstants
import android.hardware.usb.UsbDevice
import android.hardware.usb.UsbDeviceConnection
import android.hardware.usb.UsbManager
import android.os.Build
import android.os.Handler
import android.os.HandlerThread
import android.text.TextUtils
import android.util.Log
import android.util.SparseArray
import com.bandyer.core_av.usb_camera.*
import com.bandyer.core_av.usb_camera.internal.USBVendorId.vendorName
import java.io.UnsupportedEncodingException
import java.lang.ref.WeakReference
import java.nio.charset.Charset
import java.util.*
import java.util.concurrent.ConcurrentHashMap

internal class BaseUsbConnector(context: Context, listener: OnDeviceConnectListener): UsbConnector {

    private val ACTION_USB_PERMISSION = ACTION_USB_PERMISSION_BASE + hashCode()

    /**
     * openしているUsbControlBlock
     */
    val musbDatas = ConcurrentHashMap<UsbDevice?, BaseUsbData>()
    val mHasPermissions = SparseArray<WeakReference<UsbDevice?>?>()
    val mWeakContext: WeakReference<Context>
    val mUsbManager: UsbManager
    var mOnDeviceConnectListener: OnDeviceConnectListener? = null
    var mPermissionIntent: PendingIntent? = null
    val mDeviceFilters: MutableList<DeviceFilter> = ArrayList()

    /**
     * コールバックをワーカースレッドで呼び出すためのハンドラー
     */
    private lateinit var mAsyncHandler: Handler

    @Volatile
    private var destroyed: Boolean = false

    /**
     * Release all related resources,
     * never reuse again
     */
    override fun destroy() {
        if (DEBUG) Log.i(TAG, "destroy:")
        unregister()
        if (!destroyed) {
            destroyed = true
            // モニターしているUSB機器を全てcloseする
            val keys: Set<UsbDevice?> = musbDatas.keys
            if (keys != null) {
                var usbData: BaseUsbData?
                try {
                    for (key: UsbDevice? in keys) {
                        usbData = musbDatas.remove(key)
                        usbData?.close()
                    }
                } catch (e: Exception) {
                    Log.e(TAG, "destroy:", e)
                }
            }
            musbDatas.clear()
            try {
                mAsyncHandler.looper.quit()
            } catch (e: Exception) {
                Log.e(TAG, "destroy:", e)
            }
        }
    }

    /**
     * register BroadcastReceiver to monitor USB events
     *
     * @throws IllegalStateException
     */
    @Synchronized
    @Throws(IllegalStateException::class)
    override fun register() {
        if (destroyed) throw IllegalStateException("already destroyed")
        if (mPermissionIntent == null) {
            if (DEBUG) Log.i(TAG, "register:")
            val context = mWeakContext.get()
            if (context != null) {
                mPermissionIntent = PendingIntent.getBroadcast(context, 0, Intent(ACTION_USB_PERMISSION), 0)
                val filter = IntentFilter(ACTION_USB_PERMISSION)
                // ACTION_USB_DEVICE_ATTACHED never comes on some devices so it should not be added here
                filter.addAction(UsbManager.ACTION_USB_DEVICE_DETACHED)
                context.registerReceiver(mUsbReceiver, filter)
            }
            // start connection check
            mDeviceCounts = 0
            mAsyncHandler.postDelayed(mDeviceCheckRunnable, 1000)
        }
    }

    /**
     * unregister BroadcastReceiver
     *
     * @throws IllegalStateException
     */
    @Synchronized
    @Throws(IllegalStateException::class)
    override fun unregister() {
        // 接続チェック用Runnableを削除
        mDeviceCounts = 0
        if (!destroyed) {
            mAsyncHandler.removeCallbacks(mDeviceCheckRunnable)
        }
        if (mPermissionIntent != null) {
//			if (DEBUG) Log.i(TAG, "unregister:");
            val context = mWeakContext.get()
            try {
                context?.unregisterReceiver(mUsbReceiver)
            } catch (e: Exception) {
                Log.w(TAG, e)
            }
            mPermissionIntent = null
        }
    }

    @get:Synchronized
    val isRegistered: Boolean
        get() = !destroyed && (mPermissionIntent != null)

    /**
     * set device filter
     *
     * @param filter
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun setDeviceFilter(filter: DeviceFilter) {
        if (destroyed) throw IllegalStateException("already destroyed")
        mDeviceFilters.clear()
        mDeviceFilters.add(filter)
    }

    /**
     * デバイスフィルターを追加
     *
     * @param filter
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun addDeviceFilter(filter: DeviceFilter) {
        if (destroyed) throw IllegalStateException("already destroyed")
        mDeviceFilters.add(filter)
    }

    /**
     * デバイスフィルターを削除
     *
     * @param filter
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun removeDeviceFilter(filter: DeviceFilter) {
        if (destroyed) throw IllegalStateException("already destroyed")
        mDeviceFilters.remove(filter)
    }

    /**
     * set device filters
     *
     * @param filters
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun setDeviceFilter(filters: List<DeviceFilter>?) {
        if (destroyed) throw IllegalStateException("already destroyed")
        mDeviceFilters.clear()
        mDeviceFilters.addAll((filters)!!)
    }

    /**
     * add device filters
     *
     * @param filters
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun addDeviceFilter(filters: List<DeviceFilter>?) {
        if (destroyed) throw IllegalStateException("already destroyed")
        mDeviceFilters.addAll((filters)!!)
    }

    /**
     * remove device filters
     *
     * @param filters
     */
    @Throws(IllegalStateException::class)
    fun removeDeviceFilter(filters: List<DeviceFilter>?) {
        if (destroyed) throw IllegalStateException("already destroyed")
        mDeviceFilters.removeAll((filters)!!)
    }

    /**
     * return the number of connected USB devices that matched device filter
     *
     * @return
     * @throws IllegalStateException
     */
    @get:Throws(IllegalStateException::class)
    val deviceCount: Int
        get() {
            if (destroyed) throw IllegalStateException("already destroyed")
            return deviceList.size
        }

    /**
     * return device list, return empty list if no device matched
     *
     * @return
     * @throws IllegalStateException
     */
    @get:Throws(IllegalStateException::class)
    val deviceList: List<UsbDevice>
        get() {
            if (destroyed) throw IllegalStateException("already destroyed")
            return getDeviceList(mDeviceFilters)
        }

    /**
     * return device list, return empty list if no device matched
     *
     * @param filters
     * @return
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun getDeviceList(filters: List<DeviceFilter>?): List<UsbDevice> {
        if (destroyed) throw IllegalStateException("already destroyed")
        val deviceList = mUsbManager.deviceList
        val result: MutableList<UsbDevice> = ArrayList()
        if (deviceList != null) {
            if ((filters == null) || filters.isEmpty()) {
                result.addAll(deviceList.values)
            } else {
                for (device: UsbDevice in deviceList.values) {
                    for (filter: DeviceFilter? in filters) {
                        if ((filter != null) && filter.matches(device)) {
                            // when filter matches
                            if (!filter.isExclude) {
                                result.add(device)
                            }
                            break
                        }
                    }
                }
            }
        }
        return result
    }

    /**
     * return device list, return empty list if no device matched
     *
     * @param filter
     * @return
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun getDeviceList(filter: DeviceFilter?): List<UsbDevice> {
        if (destroyed) throw IllegalStateException("already destroyed")
        val deviceList = mUsbManager.deviceList
        val result: MutableList<UsbDevice> = ArrayList()
        if (deviceList != null) {
            for (device: UsbDevice in deviceList.values) {
                if ((filter == null) || (filter.matches(device) && !filter.isExclude)) {
                    result.add(device)
                }
            }
        }
        return result
    }

    /**
     * get USB device list, without filter
     *
     * @return
     * @throws IllegalStateException
     */
    @get:Throws(IllegalStateException::class)
    val devices: Iterator<UsbDevice>?
        get() {
            if (destroyed) throw IllegalStateException("already destroyed")
            var iterator: Iterator<UsbDevice>? = null
            val list = mUsbManager.deviceList
            if (list != null) iterator = list.values.iterator()
            return iterator
        }

    /**
     * output device list to LogCat
     */
    fun dumpDevices() {
        val list = mUsbManager.deviceList
        if (list != null) {
            val keys: Set<String> = list.keys
            if (keys != null && keys.size > 0) {
                val sb = StringBuilder()
                for (key: String in keys) {
                    val device = list[key]
                    val num_interface = device?.interfaceCount ?: 0
                    sb.setLength(0)
                    for (i in 0 until num_interface) {
                        sb.append(String.format(Locale.US, "interface%d:%s", i, device!!.getInterface(i).toString()))
                    }
                    Log.i(TAG, "key=$key:$device:$sb")
                }
            } else {
                Log.i(TAG, "no device")
            }
        } else {
            Log.i(TAG, "no device")
        }
    }

    /**
     * return whether the specific Usb device has permission
     *
     * @param device
     * @return true: 指定したUsbDeviceにパーミッションがある
     * @throws IllegalStateException
     */
    @Throws(IllegalStateException::class)
    fun hasPermission(device: UsbDevice?): Boolean {
        if (destroyed) throw IllegalStateException("already destroyed")
        return updatePermission(device, device != null && mUsbManager.hasPermission(device))
    }

    /**
     * 内部で保持しているパーミッション状態を更新
     *
     * @param device
     * @param hasPermission
     * @return hasPermission
     */
    private fun updatePermission(device: UsbDevice?, hasPermission: Boolean): Boolean {
        val deviceKey = getDeviceKey(device, false)
        synchronized(mHasPermissions) {
            if (hasPermission) {
                if (mHasPermissions.get(deviceKey) == null) {
                    mHasPermissions.put(deviceKey, WeakReference(device))
                }
            } else {
                mHasPermissions.remove(deviceKey)
            }
        }
        return hasPermission
    }

    /**
     * request permission to access to USB device
     *
     * @param device
     * @return true if fail to request permission
     */
    @Synchronized
    override fun requestPermission(device: UsbDevice): Boolean {
//		if (DEBUG) Log.v(TAG, "requestPermission:device=" + device);
        var result = false
        if (isRegistered) {
            if (device != null) {
                if (mUsbManager.hasPermission(device)) {
                    // call onConnect if app already has permission
                    processConnect(device)
                } else {
                    try {
                        // パーミッションがなければ要求する
                        mUsbManager.requestPermission(device, mPermissionIntent)
                    } catch (e: Exception) {
                        // Android5.1.xのGALAXY系でandroid.permission.sec.MDM_APP_MGMTという意味不明の例外生成するみたい
                        Log.w(TAG, e)
                        processCancel(device)
                        result = true
                    }
                }
            } else {
                processCancel(device)
                result = true
            }
        } else {
            processCancel(device)
            result = true
        }
        return result
    }

    /**
     * 指定したUsbDeviceをopenする
     *
     * @param device
     * @return
     * @throws SecurityException パーミッションがなければSecurityExceptionを投げる
     */
    @Throws(SecurityException::class)
    fun openDevice(device: UsbDevice): UsbData {
        if (hasPermission(device)) {
            var result = musbDatas[device]
            if (result == null) {
                result = BaseUsbData(mWeakContext.get()!!,this@BaseUsbConnector, device) // この中でopenDeviceする
                musbDatas[device] = result
            }
            return result
        } else {
            throw SecurityException("has no permission")
        }
    }

    /**
     * BroadcastReceiver for USB permission
     */
    private val mUsbReceiver: BroadcastReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            if (destroyed) return
            val action = intent.action
            if ((ACTION_USB_PERMISSION == action)) {
                // when received the result of requesting USB permission
                synchronized(this@BaseUsbConnector) {
                    val device: UsbDevice? = intent.getParcelableExtra(UsbManager.EXTRA_DEVICE)
                    if (intent.getBooleanExtra(UsbManager.EXTRA_PERMISSION_GRANTED, false)) {
                        if (device != null) {
                            // get permission, call onConnect
                            processConnect(device)
                        }
                    } else {
                        // failed to get permission
                        processCancel(device)
                    }
                }
            } else if ((UsbManager.ACTION_USB_DEVICE_ATTACHED == action)) {
                val device = intent.getParcelableExtra<UsbDevice>(UsbManager.EXTRA_DEVICE)!!
                updatePermission(device, hasPermission(device))
                processAttach(device)
            } else if ((UsbManager.ACTION_USB_DEVICE_DETACHED == action)) {
                // when device removed
                val device = intent.getParcelableExtra<UsbDevice>(UsbManager.EXTRA_DEVICE)
                if (device != null ) {
                    val usbData = musbDatas[device]
                    usbData?.close()
                    mDeviceCounts = 0
                    processDettach(device)
                }
            }
        }
    }

    /**
     * number of connected & detected devices
     */
    @Volatile
    private var mDeviceCounts = 0

    /**
     * periodically check connected devices and if it changed, call onAttach
     */
    private val mDeviceCheckRunnable: Runnable = object : Runnable {
        override fun run() {
            if (destroyed) return
            val devices = deviceList
            val n = devices.size
            var hasPermissionCounts: Int
            var m: Int
            synchronized(mHasPermissions) {
                hasPermissionCounts = mHasPermissions.size()
                mHasPermissions.clear()
                for (device: UsbDevice? in devices) {
                    hasPermission(device)
                }
                m = mHasPermissions.size()
            }
            if ((n > mDeviceCounts) || (m > hasPermissionCounts)) {
                mDeviceCounts = n
                mOnDeviceConnectListener?.let {
                    (0 until n).forEach { i ->
                        val device = devices[i]
                        if (device.deviceClass != UsbConstants.USB_CLASS_VIDEO && device.deviceClass != UsbConstants.USB_CLASS_MISC) return@forEach
                        mAsyncHandler.post { it.onAttach(device) }
                    }
                }

            }
            mAsyncHandler.postDelayed(this, 2000) // confirm every 2 seconds
        }
    }

    /**
     * open specific USB device
     *
     * @param device
     */
    private fun processConnect(device: UsbDevice) {
        if (destroyed) return
        updatePermission(device, true)
        mAsyncHandler.post {
            if (DEBUG) Log.v(TAG, "processConnect:device=$device")
            var usbData: UsbData?
            val createNew: Boolean
            usbData = musbDatas[device]
            if (usbData == null) {
                usbData = BaseUsbData(mWeakContext.get()!!, this@BaseUsbConnector, device)
                musbDatas[device] = usbData
            }
            mOnDeviceConnectListener?.onDeviceUsageGranted(device, usbData)
        }
    }

    private fun processCancel(device: UsbDevice?) {
        if (destroyed) return
        if (DEBUG) Log.v(TAG, "processCancel:")
        updatePermission(device, false)
        mOnDeviceConnectListener ?: return
        mAsyncHandler.post { mOnDeviceConnectListener?.onDeviceUsageDenied(device) }
    }

    private fun processAttach(device: UsbDevice) {
        if (destroyed) return
        if (DEBUG) Log.v(TAG, "processAttach:")
        mOnDeviceConnectListener ?: return
        if (device.deviceClass != UsbConstants.USB_CLASS_VIDEO && device.deviceClass != UsbConstants.USB_CLASS_MISC) return
        mAsyncHandler.post { mOnDeviceConnectListener?.onAttach(device) }
    }

    private fun processDettach(device: UsbDevice) {
        if (destroyed) return
        if (DEBUG) Log.v(TAG, "processDettach:")
        mOnDeviceConnectListener ?: return
        mAsyncHandler.post { mOnDeviceConnectListener?.onDetach(device) }
    }

    /**
     * ベンダー名・製品名・バージョン・シリアルを取得する
     *
     * @param device
     * @return
     */
    fun getDeviceInfo(device: UsbDevice) = updateDeviceInfo(mUsbManager, device)

    companion object {
        private val DEBUG = false // TODO set false on production
        private val TAG = "USBMonitor"
        private val ACTION_USB_PERMISSION_BASE = "com.serenegiant.USB_PERMISSION."
        val ACTION_USB_DEVICE_ATTACHED = "android.hardware.usb.action.USB_DEVICE_ATTACHED"

        /**
         * USB機器毎の設定保存用にデバイスキー名を生成する。
         * ベンダーID, プロダクトID, デバイスクラス, デバイスサブクラス, デバイスプロトコルから生成
         * 同種の製品だと同じキー名になるので注意
         *
         * @param device nullなら空文字列を返す
         * @return
         */
        fun getDeviceKeyName(device: UsbDevice?): String {
            return getDeviceKeyName(device, null, false)
        }

        /**
         * USB機器毎の設定保存用にデバイスキー名を生成する。
         * useNewAPI=falseで同種の製品だと同じデバイスキーになるので注意
         *
         * @param device
         * @param useNewAPI
         * @return
         */
        fun getDeviceKeyName(device: UsbDevice?, useNewAPI: Boolean): String {
            return getDeviceKeyName(device, null, useNewAPI)
        }

        /**
         * USB機器毎の設定保存用にデバイスキー名を生成する。この機器名をHashMapのキーにする
         * UsbDeviceがopenしている時のみ有効
         * ベンダーID, プロダクトID, デバイスクラス, デバイスサブクラス, デバイスプロトコルから生成
         * serialがnullや空文字でなければserialを含めたデバイスキー名を生成する
         * useNewAPI=trueでAPIレベルを満たしていればマニュファクチャ名, バージョン, コンフィギュレーションカウントも使う
         *
         * @param device    nullなら空文字列を返す
         * @param serial    UsbDeviceConnection#getSerialで取得したシリアル番号を渡す, nullでuseNewAPI=trueでAPI>=21なら内部で取得
         * @param useNewAPI API>=21またはAPI>=23のみで使用可能なメソッドも使用する(ただし機器によってはnullが返ってくるので有効かどうかは機器による)
         * @return
         */
        @SuppressLint("NewApi")
        fun getDeviceKeyName(device: UsbDevice?, serial: String?, useNewAPI: Boolean): String {
            if (device == null) return ""
            val sb = StringBuilder()
            sb.append(device.vendorId)
            sb.append("#") // API >= 12
            sb.append(device.productId)
            sb.append("#") // API >= 12
            sb.append(device.deviceClass)
            sb.append("#") // API >= 12
            sb.append(device.deviceSubclass)
            sb.append("#") // API >= 12
            sb.append(device.deviceProtocol) // API >= 12
            if (!TextUtils.isEmpty(serial)) {
                sb.append("#")
                sb.append(serial)
            }
            if (useNewAPI && Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                sb.append("#")
                if (TextUtils.isEmpty(serial)) {
                    sb.append(device.serialNumber)
                    sb.append("#") // API >= 21
                }
                sb.append(device.manufacturerName)
                sb.append("#") // API >= 21
                sb.append(device.configurationCount)
                sb.append("#") // API >= 21
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    sb.append(device.version)
                    sb.append("#") // API >= 23
                }
            }
            //		if (DEBUG) Log.v(TAG, "getDeviceKeyName:" + sb.toString());
            return sb.toString()
        }

        /**
         * デバイスキーを整数として取得
         * getDeviceKeyNameで得られる文字列のhasCodeを取得
         * ベンダーID, プロダクトID, デバイスクラス, デバイスサブクラス, デバイスプロトコルから生成
         * 同種の製品だと同じデバイスキーになるので注意
         *
         * @param device nullなら0を返す
         * @return
         */
        fun getDeviceKey(device: UsbDevice?): Int {
            return if (device != null) getDeviceKeyName(device, null, false).hashCode() else 0
        }

        /**
         * デバイスキーを整数として取得
         * getDeviceKeyNameで得られる文字列のhasCodeを取得
         * useNewAPI=falseで同種の製品だと同じデバイスキーになるので注意
         *
         * @param device
         * @param useNewAPI
         * @return
         */
        fun getDeviceKey(device: UsbDevice?, useNewAPI: Boolean): Int {
            return if (device != null) getDeviceKeyName(device, null, useNewAPI).hashCode() else 0
        }

        /**
         * デバイスキーを整数として取得
         * getDeviceKeyNameで得られる文字列のhasCodeを取得
         * serialがnullでuseNewAPI=falseで同種の製品だと同じデバイスキーになるので注意
         *
         * @param device    nullなら0を返す
         * @param serial    UsbDeviceConnection#getSerialで取得したシリアル番号を渡す, nullでuseNewAPI=trueでAPI>=21なら内部で取得
         * @param useNewAPI API>=21またはAPI>=23のみで使用可能なメソッドも使用する(ただし機器によってはnullが返ってくるので有効かどうかは機器による)
         * @return
         */
        fun getDeviceKey(device: UsbDevice?, serial: String?, useNewAPI: Boolean): Int {
            return if (device != null) getDeviceKeyName(device, serial, useNewAPI).hashCode() else 0
        }



        /**
         * ベンダー名・製品名・バージョン・シリアルを取得する
         *
         * @param manager
         * @param device
         * @return
         */
        fun updateDeviceInfo(manager: UsbManager, device: UsbDevice) = device.info(manager)
    }

    init {
        if (DEBUG) Log.v(TAG, "USBMonitor:Constructor")
        if (listener == null) throw IllegalArgumentException("OnDeviceConnectListener should not null.")
        mWeakContext = WeakReference(context)
        mUsbManager = context.getSystemService(Context.USB_SERVICE) as UsbManager
        mOnDeviceConnectListener = listener
        val thread = HandlerThread("UsbMonitor").also { it.start() }
        mAsyncHandler = Handler(thread.looper)
        destroyed = false
        if (DEBUG) Log.v(TAG, "USBMonitor:mUsbManager=$mUsbManager")
    }
}