/*
 * Copyright (C) 2020 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.core_av.usb_camera

import android.content.Context
import android.hardware.usb.UsbDevice
import android.hardware.usb.UsbDeviceConnection
import android.hardware.usb.UsbManager
import android.os.Build
import com.bandyer.core_av.usb_camera.internal.USBVendorId
import com.bandyer.core_av.usb_camera.internal.UsbConstants.USB_DT_STRING
import com.bandyer.core_av.usb_camera.internal.UsbConstants.USB_REQ_GET_DESCRIPTOR
import com.bandyer.core_av.usb_camera.internal.UsbConstants.USB_REQ_STANDARD_DEVICE_GET
import java.io.UnsupportedEncodingException
import java.nio.charset.Charset

/**
 * Get device info
 *
 * @param context Context
 * @return info of the device
 */
fun UsbDevice.info(context: Context): UsbDeviceInfo {
    val manager = context.applicationContext.getSystemService(Context.USB_SERVICE) as UsbManager
    return info(manager)
}

internal fun UsbDevice.info(manager: UsbManager) = UsbDeviceInfo().apply {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
        manufacturer = this@info.manufacturerName
        productName = this@info.productName
        serial = this@info.serialNumber
    }
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) usb_version = this@info.version
    if (manager.hasPermission(this@info)) {
        val connection = manager.openDevice(this@info)
        val desc = connection.rawDescriptors
        if (usb_version.isNullOrEmpty()) usb_version = String.format("%x.%02x", (desc[3].toInt() and 0xff), (desc[2].toInt() and 0xff))
        if (version.isNullOrEmpty()) version = String.format("%x.%02x", (desc[13].toInt() and 0xff), (desc[12].toInt() and 0xff))
        if (serial.isNullOrEmpty()) serial = connection.serial
        val languages = ByteArray(256)
        var languageCount = 0
        // controlTransfer(int requestType, int request, int value, int index, byte[] buffer, int length, int timeout)
        try {
            val result = connection.controlTransfer(
                    USB_REQ_STANDARD_DEVICE_GET,
                    USB_REQ_GET_DESCRIPTOR,
                    (USB_DT_STRING shl 8) or 0, 0, languages, 256, 0)
            if (result > 0) {
                languageCount = (result - 2) / 2
            }
            if (languageCount > 0) {
                if (manufacturer.isNullOrEmpty()) manufacturer = getString(connection, desc[14].toInt(), languageCount, languages)
                if (productName.isNullOrEmpty()) productName = getString(connection, desc[15].toInt(), languageCount, languages)
                if (serial.isNullOrEmpty()) serial = getString(connection, desc[16].toInt(), languageCount, languages)
            }
        } finally {
            connection.close()
        }
    }
    if (manufacturer.isNullOrEmpty()) manufacturer = USBVendorId.vendorName(this@info.vendorId)
    if (manufacturer.isNullOrEmpty()) manufacturer = String.format("%04x", this@info.vendorId)
    if (productName.isNullOrEmpty()) productName = String.format("%04x", this@info.productId)
}

private fun getString(connection: UsbDeviceConnection, id: Int, languageCount: Int, languages: ByteArray): String? {
    val work = ByteArray(256)
    var result: String? = null
    for (i in 1..languageCount) {
        val ret = connection.controlTransfer(
                USB_REQ_STANDARD_DEVICE_GET,
                USB_REQ_GET_DESCRIPTOR,
                (USB_DT_STRING shl 8) or id, languages[i].toInt(), work, 256, 0)
        if ((ret > 2) && (work[0].toInt() == ret) && (work[1].toInt() == USB_DT_STRING)) {
            // skip first two bytes(bLength & bDescriptorType), and copy the rest to the string
            try {
                result = String(work, 2, ret - 2, Charset.forName("UTF-16LE"))
                if ("Љ" != result) {    // 変なゴミが返ってくる時がある
                    break
                } else {
                    result = null
                }
            } catch (e: UnsupportedEncodingException) {
                // ignore
            }
        }
    }
    return result
}