package com.bandyer.communication_center.live_pointer

import com.bandyer.communication_center.call.Call
import com.bandyer.communication_center.live_pointer.model.PointerEvent
import com.bandyer.communication_center.live_pointer.utils.PointerObserver
import com.bandyer.communication_center.live_pointer.utils.sampleWithTimeout
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel

internal class CallAwareLivePointerInteractor(
        private val call: Call,
        private val pointerEventListener: PointerEventListener,
        private var samplingPeriod: Long?,
        private var idleTimeout: Long?) {

    private val mainScope = MainScope()
    private val ioScope = CoroutineScope(SupervisorJob() + Dispatchers.IO)
    private val listeners = hashMapOf<String, Channel<PointerEvent>>()

    private var pointerObserver: PointerObserver = object : PointerObserver(call) {
        override fun onEvent(event: PointerEvent) {
            val userAlias = event.requester.user.userAlias
            listeners[userAlias!!] = listeners[userAlias] ?: pointerChannel()
            ioScope.launch { listeners[userAlias]!!.send(event) }
        }
    }

    private fun pointerChannel() = Channel<PointerEvent>(Channel.UNLIMITED).apply {
        var lastPointer: PointerEvent? = null
        ioScope.launch {
            sampleWithTimeout(
                    onSample = {
                        mainScope.launch {
                            if (lastPointer != null && lastPointer!!.streamId != it.streamId) pointerEventListener.onPointerIdle(call, lastPointer!!)
                            lastPointer = it
                            pointerEventListener.onPointerEvent(call, it)
                        }
                    },
                    onIdle = {
                        mainScope.launch { if (lastPointer == it) pointerEventListener.onPointerIdle(call, it) }
                    }, samplingPeriod, idleTimeout)
        }
    }

    fun dispose() {
        pointerObserver.dispose()
        kotlin.runCatching {
            listeners.values.forEach { it.close() }
            listeners.clear()
        }
    }
}