// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.search.documents.indexes.models;

import com.azure.core.annotation.Fluent;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSetter;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonTypeName;

import java.util.Arrays;
import java.util.List;

/**
 * Allows you to take control over the process of converting text into
 * indexable/searchable tokens. It's a user-defined configuration consisting of
 * a single predefined tokenizer and one or more filters. The tokenizer is
 * responsible for breaking text into tokens, and the filters for modifying
 * tokens emitted by the tokenizer.
 */
@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.PROPERTY, property = "@odata.type")
@JsonTypeName("#Microsoft.Azure.Search.CustomAnalyzer")
@Fluent
public final class CustomAnalyzer extends LexicalAnalyzer {
    /*
     * The name of the tokenizer to use to divide continuous text into a
     * sequence of tokens, such as breaking a sentence into words. Possible
     * values include: 'Classic', 'EdgeNGram', 'Keyword', 'Letter',
     * 'Lowercase', 'MicrosoftLanguageTokenizer',
     * 'MicrosoftLanguageStemmingTokenizer', 'NGram', 'PathHierarchy',
     * 'Pattern', 'Standard', 'UaxUrlEmail', 'Whitespace'
     */
    @JsonProperty(value = "tokenizer", required = true)
    private LexicalTokenizerName tokenizerName;

    /*
     * A list of token filters used to filter out or modify the tokens
     * generated by a tokenizer. For example, you can specify a lowercase
     * filter that converts all characters to lowercase. The filters are run in
     * the order in which they are listed.
     */
    @JsonProperty(value = "tokenFilters")
    private List<TokenFilterName> tokenFilters;

    /*
     * A list of character filters used to prepare input text before it is
     * processed by the tokenizer. For instance, they can replace certain
     * characters or symbols. The filters are run in the order in which they
     * are listed.
     */
    @JsonProperty(value = "charFilters")
    private List<CharFilterName> charFilters;

    /**
     * Constructor of {@link LexicalAnalyzer}.
     *
     * @param name The name of the analyzer. It must only contain letters, digits, spaces,
     * dashes or underscores, can only start and end with alphanumeric
     * characters, and is limited to 128 characters.
     * @param tokenizerName The name of the tokenizer to use to divide continuous text into a
     * sequence of tokens, such as breaking a sentence into words. Possible
     * values include: 'Classic', 'EdgeNGram', 'Keyword', 'Letter',
     * 'Lowercase', 'MicrosoftLanguageTokenizer',
     * 'MicrosoftLanguageStemmingTokenizer', 'NGram', 'PathHierarchy',
     * 'Pattern', 'Standard', 'UaxUrlEmail', 'Whitespace'
     */
    public CustomAnalyzer(String name, LexicalTokenizerName tokenizerName) {
        super(name);
        this.tokenizerName = tokenizerName;
    }

    /**
     * Get the tokenizer property: The name of the tokenizer to use to divide
     * continuous text into a sequence of tokens, such as breaking a sentence
     * into words. Possible values include: 'Classic', 'EdgeNGram', 'Keyword',
     * 'Letter', 'Lowercase', 'MicrosoftLanguageTokenizer',
     * 'MicrosoftLanguageStemmingTokenizer', 'NGram', 'PathHierarchy',
     * 'Pattern', 'Standard', 'UaxUrlEmail', 'Whitespace'.
     *
     * @return the tokenizer value.
     */
    public LexicalTokenizerName getTokenizer() {
        return this.tokenizerName;
    }

    /**
     * Get the tokenFilters property: A list of token filters used to filter
     * out or modify the tokens generated by a tokenizer. For example, you can
     * specify a lowercase filter that converts all characters to lowercase.
     * The filters are run in the order in which they are listed.
     *
     * @return the tokenFilters value.
     */
    public List<TokenFilterName> getTokenFilters() {
        return this.tokenFilters;
    }

    /**
     * Set the tokenFilters property: A list of token filters used to filter
     * out or modify the tokens generated by a tokenizer. For example, you can
     * specify a lowercase filter that converts all characters to lowercase.
     * The filters are run in the order in which they are listed.
     *
     * @param tokenFilters the tokenFilters value to set.
     * @return the CustomAnalyzer object itself.
     */
    public CustomAnalyzer setTokenFilters(TokenFilterName... tokenFilters) {
        this.tokenFilters = (tokenFilters == null) ? null : Arrays.asList(tokenFilters);
        return this;
    }

    /**
     * Set the tokenFilters property: A list of token filters used to filter
     * out or modify the tokens generated by a tokenizer. For example, you can
     * specify a lowercase filter that converts all characters to lowercase.
     * The filters are run in the order in which they are listed.
     *
     * @param tokenFilters the tokenFilters value to set.
     * @return the CustomAnalyzer object itself.
     */
    @JsonSetter
    public CustomAnalyzer setTokenFilters(List<TokenFilterName> tokenFilters) {
        this.tokenFilters = tokenFilters;
        return this;
    }

    /**
     * Get the charFilters property: A list of character filters used to
     * prepare input text before it is processed by the tokenizer. For
     * instance, they can replace certain characters or symbols. The filters
     * are run in the order in which they are listed.
     *
     * @return the charFilters value.
     */
    public List<CharFilterName> getCharFilters() {
        return this.charFilters;
    }

    /**
     * Set the charFilters property: A list of character filters used to
     * prepare input text before it is processed by the tokenizer. For
     * instance, they can replace certain characters or symbols. The filters
     * are run in the order in which they are listed.
     *
     * @param charFilters the charFilters value to set.
     * @return the CustomAnalyzer object itself.
     */
    public CustomAnalyzer setCharFilters(CharFilterName... charFilters) {
        this.charFilters = (charFilters == null) ? null : Arrays.asList(charFilters);
        return this;
    }

    /**
     * Set the charFilters property: A list of character filters used to
     * prepare input text before it is processed by the tokenizer. For
     * instance, they can replace certain characters or symbols. The filters
     * are run in the order in which they are listed.
     *
     * @param charFilters the charFilters value to set.
     * @return the CustomAnalyzer object itself.
     */
    @JsonSetter
    public CustomAnalyzer setCharFilters(List<CharFilterName> charFilters) {
        this.charFilters = charFilters;
        return this;
    }
}
