// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.core.implementation.http.rest;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * A range replace substitution is a substitution that replaces a range of characters in a String with the supplied
 * value. This type of substitution is commonly used for host and path replacements where the Swagger interface will
 * have a definition such as {@code @Host("{url}")} which will be replaced with a value such as
 * {@code https://myhost.com}.
 * <p>
 * Before the introduction of this replacement {@link String#replace(CharSequence, CharSequence)} was used which would
 * generate a {@code Pattern} to perform replacing.
 */
public final class RangeReplaceSubstitution extends Substitution implements Comparable<RangeReplaceSubstitution> {
    private final int start;
    private final int end;

    /**
     * Create a new Substitution.
     *
     * @param urlParameterName The name that is used between curly quotes as a placeholder in the target URL.
     * @param methodParameterIndex The index of the parameter in the original interface method where the value for the
     * placeholder is.
     * @param shouldEncode Whether the value from the method's argument should be encoded when the substitution is
     * taking place.
     * @param start The starting index of the range replacement substitution.
     * @param end The ending index of the range replacement substitution.
     */
    private RangeReplaceSubstitution(String urlParameterName, int methodParameterIndex, boolean shouldEncode, int start,
        int end) {
        super(urlParameterName, methodParameterIndex, shouldEncode);
        this.start = start;
        this.end = end;
    }

    /**
     * Gets the range replacement substitutions for the provided urlParameterName.
     *
     * @param urlParameterName The name that is used between curly quotes as a placeholder in the target URL.
     * @param methodParameterIndex The index of the parameter in the original interface method where the value for the
     * placeholder is.
     * @param shouldEncode Whether the value from the method's argument should be encoded when the substitution is
     * taking place.
     * @param substitutionBase The base string that will be used to find the range replacement substitutions.
     * @return The range replacement substitutions for the provided urlParameterName.
     */
    public static List<RangeReplaceSubstitution> getRangeReplaceSubstitutions(String urlParameterName,
        int methodParameterIndex, boolean shouldEncode, String substitutionBase) {
        List<RangeReplaceSubstitution> substitutions = new ArrayList<>();

        String placeholder = "{" + urlParameterName + "}";
        int indexOf = 0;
        while (true) {
            indexOf = substitutionBase.indexOf(placeholder, indexOf);

            if (indexOf == -1) {
                break;
            }

            substitutions.add(new RangeReplaceSubstitution(urlParameterName, methodParameterIndex, shouldEncode,
                indexOf, indexOf + placeholder.length()));
            indexOf = indexOf + placeholder.length();
        }

        return substitutions;
    }

    /**
     * Gets the starting index of the range replacement substitution.
     *
     * @return The starting index of the range replacement substitution.
     */
    public int getStart() {
        return start;
    }

    /**
     * Gets the ending index of the range replacement substitution.
     *
     * @return The ending index of the range replacement substitution.
     */
    public int getEnd() {
        return end;
    }

    /**
     * Gets the size of the range replacement substitution.
     *
     * @return The size of the range replacement substitution.
     */
    public int getSize() {
        return end - start;
    }

    @Override
    public int compareTo(RangeReplaceSubstitution o) {
        if (start < o.start) {
            return -1;
        } else if (start > o.start) {
            return 1;
        } else {
            return Integer.compare(end, o.end);
        }
    }

    @Override
    public int hashCode() {
        return Objects.hash(start, end);
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof RangeReplaceSubstitution)) {
            return false;
        }

        RangeReplaceSubstitution other = (RangeReplaceSubstitution) obj;

        return start == other.start && end == other.end;
    }
}
