// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.core.util;

import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.serializer.JsonSerializerProviders;
import com.azure.core.util.serializer.ObjectSerializer;
import com.azure.core.util.serializer.JsonSerializer;
import com.azure.core.util.serializer.TypeReference;
import static com.azure.core.util.FluxUtil.monoError;

import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;

/**
 * This class is an abstraction over many different ways that binary data can be represented. The {@link BinaryData}
 * can be created from {@link InputStream}, {@link Flux} of {@link ByteBuffer}, {@link String}, {@link Object}, or byte
 * array.
 * <p><strong>Immutable data</strong></p>
 * {@link BinaryData} is constructed by copying the given data. Once {@link BinaryData} is instantiated, it can not be
 * changed. It provides various convenient APIs to get data out of {@link BinaryData}, they all start with the 'to'
 * prefix, for example {@link BinaryData#toBytes()}.
 * <p>
 * Code samples are presented below.
 *
 * <p><strong>Create an instance from Bytes</strong></p>
 * <pre>
 * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
 * BinaryData binaryData = BinaryData.fromBytes&#40;data&#41;;
 * System.out.println&#40;new String&#40;binaryData.toBytes&#40;&#41;, StandardCharsets.UTF_8&#41;&#41;;
 * </pre>
 *
 * <p><strong>Create an instance from String</strong></p>
 * <pre>
 * final String data = &quot;Some Data&quot;;
 * &#47;&#47; Following will use default character set as StandardCharsets.UTF_8
 * BinaryData binaryData = BinaryData.fromString&#40;data&#41;;
 * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
 * </pre>
 *
 * <p><strong>Create an instance from InputStream</strong></p>
 * <pre>
 * final ByteArrayInputStream inputStream = new ByteArrayInputStream&#40;&quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;&#41;;
 * BinaryData binaryData = BinaryData.fromStream&#40;inputStream&#41;;
 * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
 * </pre>
 *
 * <p><strong>Create an instance from Object</strong></p>
 * <pre>
 * class Person &#123;
 *     &#123;@literal @&#125;JsonProperty
 *     private String name;
 * 
 *     &#123;@literal @&#125;JsonSetter
 *     public Person setName&#40;String name&#41; &#123;
 *         this.name = name;
 *         return this;
 *     &#125;
 * 
 *     &#123;@literal @&#125;JsonGetter
 *     public String getName&#40;&#41; &#123;
 *         return name;
 *     &#125;
 * &#125;
 * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
 * 
 * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
 * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
 * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
 * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
 * 
 * </pre>
 *
 * @see ObjectSerializer
 * @see JsonSerializer
 * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
 */
public final class  BinaryData {
    private static final ClientLogger LOGGER = new ClientLogger(BinaryData.class);
    private static final BinaryData EMPTY_DATA = new BinaryData(new byte[0]);

    private static final Object LOCK = new Object();

    private final byte[] data;

    private static volatile JsonSerializer defaultJsonSerializer;

    /**
     * Create an instance of {@link BinaryData} from the given data.
     *
     * @param data to represent as bytes.
     */
    BinaryData(byte[] data) {
        this.data = data;
    }

    /**
     * Creates a {@link BinaryData} instance with given {@link InputStream} as source of data. The {@link InputStream}
     * is not closed by this function.
     *
     * <p><strong>Create an instance from InputStream</strong></p>
     * <pre>
     * final ByteArrayInputStream inputStream = new ByteArrayInputStream&#40;&quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;&#41;;
     * BinaryData binaryData = BinaryData.fromStream&#40;inputStream&#41;;
     * System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     * </pre>
     *
     * @param inputStream The {@link InputStream} to use as data backing the instance of {@link BinaryData}.
     * @throws UncheckedIOException If any error in reading from {@link InputStream}.
     * @throws NullPointerException If {@code inputStream} is null.
     * @return {@link BinaryData} representing the binary data.
     */
    public static BinaryData fromStream(InputStream inputStream) {
        if (Objects.isNull(inputStream)) {
            return EMPTY_DATA;
        }

        final int bufferSize = 1024;
        try {
            ByteArrayOutputStream dataOutputBuffer = new ByteArrayOutputStream();
            int nRead;
            byte[] data = new byte[bufferSize];
            while ((nRead = inputStream.read(data, 0, data.length)) != -1) {
                dataOutputBuffer.write(data, 0, nRead);
            }
            dataOutputBuffer.flush();

            return new BinaryData(dataOutputBuffer.toByteArray());

        } catch (IOException ex) {
            throw LOGGER.logExceptionAsError(new UncheckedIOException(ex));
        }
    }

    /**
     * Asynchronously creates a {@link BinaryData} instance with the given {@link InputStream} as source of data. The
     * {@link InputStream} is not closed by this function. If the {@link InputStream} is {@code null}, an empty
     * {@link BinaryData} will be returned.
     *
     * @param inputStream The {@link InputStream} to use as data backing the instance of {@link BinaryData}.
     * @return {@link Mono} of {@link BinaryData} representing the binary data.
     */
    public static Mono<BinaryData> fromStreamAsync(InputStream inputStream) {
        return Mono.fromCallable(() -> fromStream(inputStream));
    }

    /**
     * Creates a {@link BinaryData} instance with given {@link Flux} of {@link ByteBuffer} as source of data. It will
     * collect all the bytes from {@link ByteBuffer} into {@link BinaryData}. If the {@link Flux} is {@code null}, an
     * empty {@link BinaryData} will be returned.
     *
     * <p><strong>Create an instance from String</strong></p>
     * <pre>
     * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
     * final Flux&lt;ByteBuffer&gt; dataFlux = Flux.just&#40;ByteBuffer.wrap&#40;data&#41;&#41;;
     * 
     * Mono&lt;BinaryData&gt; binaryDataMono = BinaryData.fromFlux&#40;dataFlux&#41;;
     * 
     * Disposable subscriber = binaryDataMono
     *     .map&#40;binaryData -&gt; &#123;
     *         System.out.println&#40;binaryData.toString&#40;&#41;&#41;;
     *         return true;
     *     &#125;&#41;
     *     .subscribe&#40;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param data The byte buffer stream to use as data backing the instance of {@link BinaryData}.
     * @return {@link Mono} of {@link BinaryData} representing binary data.
     */
    public static Mono<BinaryData> fromFlux(Flux<ByteBuffer> data) {
        if (Objects.isNull(data)) {
            return Mono.just(EMPTY_DATA);
        }

        return FluxUtil.collectBytesInByteBufferStream(data)
            .flatMap(bytes -> Mono.just(new BinaryData(bytes)));
    }

    /**
     * Creates a {@link BinaryData} instance with given data. The {@link String} is converted into bytes using UTF_8
     * character set. If the String is {@code null}, an empty {@link BinaryData} will be returned.
     *
     * @param data The string to use as data backing the instance of {@link BinaryData}.
     * @return {@link BinaryData} representing binary data.
     */
    public static BinaryData fromString(String data) {
        if (Objects.isNull(data) || data.length() == 0) {
            return EMPTY_DATA;
        }

        return new BinaryData(data.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Creates a {@link BinaryData} instance with given byte array data. If the byte array is {@code null}, an empty
     * {@link BinaryData} will be returned.
     *
     * @param data The byte array to use as data backing the instance of {@link BinaryData}.
     * @return {@link BinaryData} representing the binary data.
     */
    public static BinaryData fromBytes(byte[] data) {
        if (Objects.isNull(data) || data.length == 0) {
            return EMPTY_DATA;
        }

        return new BinaryData(Arrays.copyOf(data, data.length));
    }

    /**
     * Serialize the given {@link Object} into {@link BinaryData} using json serializer which is available on classpath.
     * The serializer on classpath must implement {@link JsonSerializer} interface. If the given Object is {@code null},
     * an empty {@link BinaryData} will be returned.
     * <p><strong>Code sample</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * </pre>

     * @param data The object to use as data backing the instance of {@link BinaryData}.
     * @throws IllegalStateException If a {@link JsonSerializer} cannot be found on the classpath.
     * @return {@link BinaryData} representing the JSON serialized object.
     *
     * @see JsonSerializer
     * @see <a href="ObjectSerializer" target="_blank">More about serialization</a>
     */
    public static BinaryData fromObject(Object data) {
        if (Objects.isNull(data)) {
            return EMPTY_DATA;
        }
        final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        getDefaultSerializer().serialize(outputStream, data);

        return new BinaryData(outputStream.toByteArray());
    }

    /**
     * Serialize the given {@link Object} into {@link BinaryData} using json serializer which is available on classpath.
     * The serializer on classpath must implement {@link JsonSerializer} interface. If the given Object is {@code null},
     * an empty {@link BinaryData} will be returned.
     * <p><strong>Code sample</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * BinaryData.fromObjectAsync&#40;data&#41;
     *     .subscribe&#40;binaryData -&gt; sendToService&#40;binaryData&#41;&#41;;
     * 
     * </pre>

     * @param data The object to use as data backing the instance of {@link BinaryData}.
     * @throws IllegalStateException If a {@link JsonSerializer} cannot be found on the classpath.
     * @return {@link BinaryData} representing the JSON serialized object.
     *
     * @see JsonSerializer
     * @see <a href="ObjectSerializer" target="_blank">More about serialization</a>
     */
    public static Mono<BinaryData> fromObjectAsync(Object data) {
        return Mono.fromCallable(() -> fromObject(data));
    }

    /**
     * Serialize the given {@link Object} into {@link BinaryData} using the provided {@link ObjectSerializer}.
     * If the Object is {@code null}, an empty {@link BinaryData} will be returned.
     * <p>You can provide your custom implementation of {@link ObjectSerializer} interface or use one provided in Azure
     * SDK by adding them as dependency.
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">Gson serializer</a>.</li>
     * </ul>
     *
     * <p><strong>Create an instance from Object</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * </pre>
     *
     * @param data The object to use as data backing the instance of {@link BinaryData}.
     * @param serializer to use for serializing the object.
     * @throws NullPointerException If {@code serializer} is null.
     * @return {@link BinaryData} representing binary data.
     * @see ObjectSerializer
     * @see JsonSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public static BinaryData fromObject(Object data, ObjectSerializer serializer) {
        if (Objects.isNull(data)) {
            return EMPTY_DATA;
        }

        Objects.requireNonNull(serializer, "'serializer' cannot be null.");

        final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        serializer.serialize(outputStream, data);
        return new BinaryData(outputStream.toByteArray());
    }

    /**
     * Serialize the given {@link Object} into {@link Mono} {@link BinaryData} using the provided
     * {@link ObjectSerializer}. If the Object is {@code null}, an empty {@link BinaryData} will be returned.
     *
     * <p>You can provide your custom implementation of {@link ObjectSerializer} interface or use one provided in zure
     * SDK by adding them as dependency.
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">Gson serializer</a>.</li>
     * </ul>
     *
     * @param data The object to use as data backing the instance of {@link BinaryData}.
     * @param serializer to use for serializing the object.
     * @throws NullPointerException If {@code serializer} is null.
     * @return {@link Mono} of {@link BinaryData} representing the binary data.
     * @see ObjectSerializer
     * @see <a href="https://aka.ms/azsdk/java/docs/serialization" target="_blank">More about serialization</a>
     */
    public static Mono<BinaryData> fromObjectAsync(Object data, ObjectSerializer serializer) {
        if (Objects.isNull(serializer)) {
            return monoError(LOGGER, new NullPointerException("'serializer' cannot be null."));
        }
        return Mono.fromCallable(() -> fromObject(data, serializer));
    }

    /**
     * Provides byte array representation of this {@link BinaryData} object.
     *
     * @return byte array representation of the the data.
     */
    public byte[] toBytes() {
        return Arrays.copyOf(this.data, this.data.length);
    }

    /**
     * Provides {@link String} representation of this {@link BinaryData} object. The bytes are converted into
     * {@link String} using the UTF-8 character set.
     *
     * @return {@link String} representation of the data.
     */
    public String toString() {
        return new String(this.data, StandardCharsets.UTF_8);
    }

    /**
     * Deserialize the bytes into the {@link Object} of given type by applying the provided {@link ObjectSerializer} on
     * the data. The type, represented by {@link TypeReference}, can either be a regular class or a generic class that
     * retains the type information.
     *
     * <p>You can provide your custom implementation of {@link ObjectSerializer} interface or use one provided in zure
     * SDK by adding them as dependency.
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">Gson serializer</a>.</li>
     * </ul>
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing a regular class</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * Person person = binaryData.toObject&#40;TypeReference.createInstance&#40;Person.class&#41;, serializer&#41;;
     * System.out.println&#40;&quot;Name : &quot; + person.getName&#40;&#41;&#41;;
     * 
     * </pre>
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing generic types</strong></p>
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;personList, serializer&#41;;
     * 
     * &#47;&#47; Retains the type of the list when deserializing
     * List&lt;Person&gt; persons = binaryData.toObject&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;, serializer&#41;;
     * persons.forEach&#40;person -&gt; System.out.println&#40;&quot;Name : &quot; + person.getName&#40;&#41;&#41;&#41;;
     * </pre>
     *
     * @param typeReference representing the {@link TypeReference type} of the Object.
     * @param serializer to use deserialize data into type.
     * @param <T> Generic type that the data is deserialized into.
     * @throws NullPointerException If {@code serializer} or {@code typeReference} is null.
     * @return The {@link Object} of given type after deserializing the bytes.
     */
    public <T> T toObject(TypeReference<T> typeReference, ObjectSerializer serializer) {
        Objects.requireNonNull(typeReference, "'typeReference' cannot be null.");
        Objects.requireNonNull(serializer, "'serializer' cannot be null.");

        InputStream jsonStream = new ByteArrayInputStream(this.data);
        return serializer.deserialize(jsonStream, typeReference);
    }

    /**
     * Return a {@link Mono} by deserializing the bytes into the {@link Object} of given type after applying the
     * provided {@link ObjectSerializer} on the {@link BinaryData}. The type, represented by {@link TypeReference},
     * can either be a regular class or a generic class that retains the type information.
     *
     * <p>You can provide your custom implementation of {@link ObjectSerializer} interface or use one provided in zure
     * SDK by adding them as dependency.
     * <ul>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-jackson" target="_blank">Jackson serializer</a></li>
     * <li><a href="https://mvnrepository.com/artifact/com.azure/azure-core-serializer-json-gson" target="_blank">Gson serializer</a>.</li>
     * </ul>
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing a regular class</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Provide your custom serializer or use Azure provided serializers.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;data, serializer&#41;;
     * 
     * Disposable subscriber = binaryData
     *     .toObjectAsync&#40;TypeReference.createInstance&#40;Person.class&#41;, serializer&#41;
     *     .subscribe&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing generic types</strong></p>
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * final ObjectSerializer serializer =
     *     new MyJsonSerializer&#40;&#41;; &#47;&#47; Replace this with your Serializer
     * BinaryData binaryData = BinaryData.fromObject&#40;personList, serializer&#41;;
     * 
     * Disposable subscriber = binaryData
     *     .toObjectAsync&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;, serializer&#41; &#47;&#47; retains the generic type information
     *     .subscribe&#40;persons -&gt; persons.forEach&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;&#41;;
     * 
     * &#47;&#47; So that your program wait for above subscribe to complete.
     * TimeUnit.SECONDS.sleep&#40;5&#41;;
     * subscriber.dispose&#40;&#41;;
     * </pre>
     *
     * @param typeReference representing the {@link TypeReference type} of the Object.
     * @param serializer to use deserialize data into type.
     * @param <T> Generic type that the data is deserialized into.
     * @throws NullPointerException If {@code typeReference} or {@code serializer} is null.
     * @return The {@link Object} of given type after deserializing the bytes.
     */
    public <T> Mono<T> toObjectAsync(TypeReference<T> typeReference, ObjectSerializer serializer) {

        if (Objects.isNull(typeReference)) {
            return monoError(LOGGER, new NullPointerException("'typeReference' cannot be null."));
        } else if (Objects.isNull(serializer)) {
            return monoError(LOGGER, new NullPointerException("'serializer' cannot be null."));
        }
        return Mono.fromCallable(() -> toObject(typeReference, serializer));
    }

    /**
     * Deserialize the bytes into the {@link Object} of given type by using json serializer which is available in
     * classpath. The type, represented by {@link TypeReference}, can either be a regular class or a generic class that
     * retains the type information. This method assumes the data to be in JSON format and will use a default
     * implementation of {@link JsonSerializer}.
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing a regular class</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * Person person = binaryData.toObject&#40;TypeReference.createInstance&#40;Person.class&#41;&#41;;
     * System.out.println&#40;person.getName&#40;&#41;&#41;;
     * </pre>
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing generic types</strong></p>
     * <pre>
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;personList&#41;;
     * 
     * List&lt;Person&gt; persons = binaryData.toObject&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;&#41;;
     * persons.forEach&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * </pre>
     *
     * @param typeReference representing the {@link TypeReference type} of the Object.
     * @param <T> Generic type that the data is deserialized into.
     * @throws NullPointerException If {@code typeReference} is null.
     * @return The {@link Object} of given type after deserializing the bytes.
     */
    public <T> T toObject(TypeReference<T> typeReference) {
        Objects.requireNonNull(typeReference, "'typeReference' cannot be null.");

        InputStream jsonStream = new ByteArrayInputStream(this.data);
        return getDefaultSerializer().deserialize(jsonStream, typeReference);
    }

    /**
     * Return a {@link Mono} by deserializing the bytes into the {@link Object} of given type after applying the Json
     * serializer found on classpath. The type, represented by {@link TypeReference}, can either be a regular class
     * or a generic class that retains the type information. This method assumes the data to be in JSON format and will
     * use a default implementation of {@link JsonSerializer}.
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing a regular class</strong></p>
     * <pre>
     * class Person &#123;
     *     &#123;@literal @&#125;JsonProperty
     *     private String name;
     * 
     *     &#123;@literal @&#125;JsonSetter
     *     public Person setName&#40;String name&#41; &#123;
     *         this.name = name;
     *         return this;
     *     &#125;
     * 
     *     &#123;@literal @&#125;JsonGetter
     *     public String getName&#40;&#41; &#123;
     *         return name;
     *     &#125;
     * &#125;
     * final Person data = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * 
     * &#47;&#47; Ensure your classpath have the Serializer to serialize the object which implement implement
     * &#47;&#47; com.azure.core.util.serializer.JsonSerializer interface.
     * &#47;&#47; Or use Azure provided libraries for this.
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-jackson or
     * &#47;&#47; https:&#47;&#47;mvnrepository.com&#47;artifact&#47;com.azure&#47;azure-core-serializer-json-gson
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;data&#41;;
     * 
     * binaryData.toObjectAsync&#40;TypeReference.createInstance&#40;Person.class&#41;&#41;
     *     .subscribe&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;;
     * </pre>
     *
     * <p><strong>Code sample to demonstrate serializing and deserializing generic types</strong></p>
     * <pre>
     * 
     * final Person person1 = new Person&#40;&#41;.setName&#40;&quot;John&quot;&#41;;
     * final Person person2 = new Person&#40;&#41;.setName&#40;&quot;Jack&quot;&#41;;
     * 
     * List&lt;Person&gt; personList = new ArrayList&lt;&gt;&#40;&#41;;
     * personList.add&#40;person1&#41;;
     * personList.add&#40;person2&#41;;
     * 
     * BinaryData binaryData = BinaryData.fromObject&#40;personList&#41;;
     * 
     * binaryData.toObjectAsync&#40;new TypeReference&lt;List&lt;Person&gt;&gt;&#40;&#41; &#123; &#125;&#41;
     *     .subscribe&#40;persons -&gt; persons.forEach&#40;person -&gt; System.out.println&#40;person.getName&#40;&#41;&#41;&#41;&#41;;
     * </pre>
     *
     * @param typeReference representing the {@link TypeReference type} of the Object.
     * @param <T> Generic type that the data is deserialized into.
     * @throws NullPointerException If {@code typeReference} is null.
     * @return The {@link Object} of given type after deserializing the bytes.
     */
    public <T> Mono<T> toObjectAsync(TypeReference<T> typeReference) {
        if (Objects.isNull(typeReference)) {
            return monoError(LOGGER, new NullPointerException("'typeReference' cannot be null."));
        }
        return Mono.fromCallable(() -> toObject(typeReference));
    }

    /**
     * Provides {@link InputStream} for the data represented by this {@link BinaryData} object.
     *
     * <p><strong>Get InputStream from BinaryData</strong></p>
     * <pre>
     * final byte[] data = &quot;Some Data&quot;.getBytes&#40;StandardCharsets.UTF_8&#41;;
     * BinaryData binaryData = BinaryData.fromStream&#40;new ByteArrayInputStream&#40;data&#41;&#41;;
     * final byte[] bytes = new byte[data.length];
     * &#40;binaryData.toStream&#40;&#41;&#41;.read&#40;bytes, 0, data.length&#41;;
     * System.out.println&#40;new String&#40;bytes&#41;&#41;;
     * </pre>
     *
     * @return {@link InputStream} representing the binary data.
     */
    public InputStream toStream() {
        return new ByteArrayInputStream(this.data);
    }

    /* This will ensure lazy instantiation to avoid hard dependency on Json Serializer. */
    private static JsonSerializer getDefaultSerializer() {
        if (defaultJsonSerializer ==  null) {
            synchronized (LOCK) {
                if (defaultJsonSerializer == null) {
                    defaultJsonSerializer = JsonSerializerProviders.createInstance();
                }
            }
        }
        return defaultJsonSerializer;
    }
}
