/*
 * Decompiled with CFR 0.152.
 */
package com.atomikos.datasource.pool;

import com.atomikos.datasource.pool.ConnectionFactory;
import com.atomikos.datasource.pool.ConnectionPoolException;
import com.atomikos.datasource.pool.ConnectionPoolProperties;
import com.atomikos.datasource.pool.CreateConnectionException;
import com.atomikos.datasource.pool.PoolExhaustedException;
import com.atomikos.datasource.pool.Reapable;
import com.atomikos.datasource.pool.XPooledConnection;
import com.atomikos.datasource.pool.XPooledConnectionEventListener;
import com.atomikos.icatch.HeuristicMessage;
import com.atomikos.icatch.imp.thread.InterruptedExceptionHelper;
import com.atomikos.icatch.imp.thread.TaskManager;
import com.atomikos.logging.Logger;
import com.atomikos.logging.LoggerFactory;
import com.atomikos.timing.AlarmTimer;
import com.atomikos.timing.AlarmTimerListener;
import com.atomikos.timing.PooledAlarmTimer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class ConnectionPool
implements XPooledConnectionEventListener {
    private static final Logger LOGGER = LoggerFactory.createLogger(ConnectionPool.class);
    public static final int DEFAULT_MAINTENANCE_INTERVAL = 60;
    private List<XPooledConnection> connections = new ArrayList<XPooledConnection>();
    private ConnectionFactory connectionFactory;
    private ConnectionPoolProperties properties;
    private boolean destroyed;
    private PooledAlarmTimer maintenanceTimer;
    private String name;

    public ConnectionPool(ConnectionFactory connectionFactory, ConnectionPoolProperties properties) throws ConnectionPoolException {
        this.connectionFactory = connectionFactory;
        this.properties = properties;
        this.destroyed = false;
        this.name = properties.getUniqueResourceName();
        this.init();
    }

    private void assertNotDestroyed() throws ConnectionPoolException {
        if (this.destroyed) {
            throw new ConnectionPoolException("Pool was already destroyed - you can no longer use it");
        }
    }

    private void init() throws ConnectionPoolException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": initializing...");
        }
        this.addConnectionsIfMinPoolSizeNotReached();
        this.launchMaintenanceTimer();
    }

    private void launchMaintenanceTimer() {
        int maintenanceInterval = this.properties.getMaintenanceInterval();
        if (maintenanceInterval <= 0) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.logDebug(this + ": using default maintenance interval...");
            }
            maintenanceInterval = 60;
        }
        this.maintenanceTimer = new PooledAlarmTimer((long)(maintenanceInterval * 1000));
        this.maintenanceTimer.addAlarmTimerListener(new AlarmTimerListener(){

            public void alarm(AlarmTimer timer) {
                ConnectionPool.this.reapPool();
                ConnectionPool.this.removeConnectionsThatExceededMaxLifetime();
                ConnectionPool.this.addConnectionsIfMinPoolSizeNotReached();
                ConnectionPool.this.removeIdleConnectionsIfMinPoolSizeExceeded();
            }
        });
        TaskManager.getInstance().executeTask((Runnable)this.maintenanceTimer);
    }

    private synchronized void addConnectionsIfMinPoolSizeNotReached() {
        int connectionsToAdd = this.properties.getMinPoolSize() - this.totalSize();
        for (int i = 0; i < connectionsToAdd; ++i) {
            try {
                XPooledConnection xpc = this.connectionFactory.createPooledConnection();
                this.connections.add(xpc);
                xpc.registerXPooledConnectionEventListener(this);
                continue;
            }
            catch (Exception dbDown) {
                if (!LOGGER.isDebugEnabled()) continue;
                LOGGER.logDebug(this + ": could not establish initial connection", (Throwable)dbDown);
            }
        }
    }

    private Reapable recycleConnectionIfPossible(HeuristicMessage hmsg) throws Exception {
        Reapable ret = null;
        for (int i = 0; i < this.totalSize(); ++i) {
            XPooledConnection xpc = this.connections.get(i);
            if (!xpc.canBeRecycledForCallingThread()) continue;
            ret = xpc.createConnectionProxy(hmsg);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.logDebug(this + ": recycling connection from pool...");
            }
            return ret;
        }
        return ret;
    }

    public synchronized Reapable borrowConnection(HeuristicMessage hmsg) throws CreateConnectionException, PoolExhaustedException, ConnectionPoolException {
        this.assertNotDestroyed();
        Reapable ret = null;
        ret = this.findExistingOpenConnectionForCallingThread(hmsg);
        if (ret == null) {
            ret = this.findOrWaitForAnAvailableConnection(hmsg);
        }
        return ret;
    }

    private Reapable findOrWaitForAnAvailableConnection(HeuristicMessage hmsg) throws ConnectionPoolException {
        Reapable ret = null;
        long remainingTime = (long)this.properties.getBorrowConnectionTimeout() * 1000L;
        do {
            if ((ret = this.retrieveFirstAvailableConnectionAndGrowPoolIfNecessary(hmsg)) != null) continue;
            remainingTime = this.waitForAtLeastOneAvailableConnection(remainingTime);
            this.assertNotDestroyed();
        } while (ret == null);
        return ret;
    }

    private Reapable retrieveFirstAvailableConnectionAndGrowPoolIfNecessary(HeuristicMessage hmsg) throws CreateConnectionException {
        Reapable ret = this.retrieveFirstAvailableConnection(hmsg);
        if (ret == null && this.canGrow()) {
            this.growPool();
            ret = this.retrieveFirstAvailableConnection(hmsg);
        }
        return ret;
    }

    private Reapable findExistingOpenConnectionForCallingThread(HeuristicMessage hmsg) {
        Reapable recycledConnection = null;
        try {
            recycledConnection = this.recycleConnectionIfPossible(hmsg);
        }
        catch (Exception e) {
            LOGGER.logWarning(this + ": error while trying to recycle", (Throwable)e);
        }
        return recycledConnection;
    }

    private void logCurrentPoolSize() {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": current size: " + this.availableSize() + "/" + this.totalSize());
        }
    }

    private boolean canGrow() {
        return this.totalSize() < this.properties.getMaxPoolSize();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Reapable retrieveFirstAvailableConnection(HeuristicMessage hmsg) {
        Reapable ret = null;
        Iterator<XPooledConnection> it = this.connections.iterator();
        while (it.hasNext() && ret == null) {
            XPooledConnection xpc = it.next();
            if (!xpc.isAvailable()) continue;
            try {
                ret = xpc.createConnectionProxy(hmsg);
                if (!LOGGER.isDebugEnabled()) continue;
                LOGGER.logDebug(this + ": got connection from pool");
            }
            catch (CreateConnectionException ex) {
                String msg = this + ": error creating proxy of connection " + xpc;
                LOGGER.logWarning(msg, (Throwable)ex);
                it.remove();
                xpc.destroy();
            }
            finally {
                this.logCurrentPoolSize();
            }
        }
        return ret;
    }

    private synchronized void growPool() throws CreateConnectionException {
        XPooledConnection xpc = this.connectionFactory.createPooledConnection();
        this.connections.add(xpc);
        xpc.registerXPooledConnectionEventListener(this);
        this.logCurrentPoolSize();
    }

    private synchronized void removeIdleConnectionsIfMinPoolSizeExceeded() {
        if (this.connections == null || this.properties.getMaxIdleTime() <= 0) {
            return;
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": trying to shrink pool");
        }
        ArrayList<XPooledConnection> connectionsToRemove = new ArrayList<XPooledConnection>();
        int maxConnectionsToRemove = this.totalSize() - this.properties.getMinPoolSize();
        if (maxConnectionsToRemove > 0) {
            for (int i = 0; i < this.connections.size(); ++i) {
                XPooledConnection xpc = this.connections.get(i);
                long lastRelease = xpc.getLastTimeReleased();
                long maxIdle = this.properties.getMaxIdleTime();
                long now = System.currentTimeMillis();
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.logDebug(this + ": connection idle for " + (now - lastRelease) + "ms");
                }
                if (!xpc.isAvailable() || now - lastRelease < maxIdle * 1000L || connectionsToRemove.size() >= maxConnectionsToRemove) continue;
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.logDebug(this + ": connection idle for more than " + maxIdle + "s, closing it: " + xpc);
                }
                xpc.destroy();
                connectionsToRemove.add(xpc);
            }
        }
        this.connections.removeAll(connectionsToRemove);
        this.logCurrentPoolSize();
    }

    private synchronized void reapPool() {
        long maxInUseTime = this.properties.getReapTimeout();
        if (this.connections == null || maxInUseTime <= 0L) {
            return;
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": reaping old connections");
        }
        Iterator<XPooledConnection> it = this.connections.iterator();
        while (it.hasNext()) {
            XPooledConnection xpc = it.next();
            long lastTimeReleased = xpc.getLastTimeAcquired();
            boolean inUse = !xpc.isAvailable();
            long now = System.currentTimeMillis();
            if (!inUse || now - maxInUseTime * 1000L <= lastTimeReleased) continue;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.logDebug(this + ": connection in use for more than " + maxInUseTime + "s, reaping it: " + xpc);
            }
            xpc.reap();
            it.remove();
        }
        this.logCurrentPoolSize();
    }

    private synchronized void removeConnectionsThatExceededMaxLifetime() {
        long maxLifetime = this.properties.getMaxLifetime();
        if (this.connections == null || maxLifetime <= 0L) {
            return;
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": closing connections that exceeded maxLifetime");
        }
        Iterator<XPooledConnection> it = this.connections.iterator();
        while (it.hasNext()) {
            XPooledConnection xpc = it.next();
            long creationTime = xpc.getCreationTime();
            long now = System.currentTimeMillis();
            if (!xpc.isAvailable() || now - creationTime < maxLifetime * 1000L) continue;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.logDebug(this + ": connection in use for more than " + maxLifetime + "s, destroying it: " + xpc);
            }
            xpc.destroy();
            it.remove();
        }
        this.logCurrentPoolSize();
    }

    public synchronized void destroy() {
        if (!this.destroyed) {
            LOGGER.logInfo(this + ": destroying pool...");
            for (int i = 0; i < this.connections.size(); ++i) {
                XPooledConnection xpc = this.connections.get(i);
                if (!xpc.isAvailable()) {
                    LOGGER.logWarning(this + ": connection is still in use on pool destroy: " + xpc + " - please check your shutdown sequence to avoid heuristic termination " + "of ongoing transactions!");
                }
                xpc.destroy();
            }
            this.connections = null;
            this.destroyed = true;
            this.maintenanceTimer.stop();
            if (LOGGER.isDebugEnabled()) {
                LOGGER.logDebug(this + ": pool destroyed.");
            }
        }
    }

    private synchronized long waitForAtLeastOneAvailableConnection(long waitTime) throws PoolExhaustedException {
        while (this.availableSize() == 0) {
            long before;
            block6: {
                if (waitTime <= 0L) {
                    throw new PoolExhaustedException("ConnectionPool: pool is empty - increase either maxPoolSize or borrowConnectionTimeout");
                }
                before = System.currentTimeMillis();
                try {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.logDebug(this + ": about to wait for connection during " + waitTime + "ms...");
                    }
                    this.wait(waitTime);
                }
                catch (InterruptedException ex) {
                    InterruptedExceptionHelper.handleInterruptedException((InterruptedException)ex);
                    if (!LOGGER.isDebugEnabled()) break block6;
                    LOGGER.logDebug(this + ": interrupted during wait", (Throwable)ex);
                }
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.logDebug(this + ": done waiting.");
            }
            long now = System.currentTimeMillis();
            waitTime -= now - before;
        }
        return waitTime;
    }

    public synchronized int availableSize() {
        int ret = 0;
        if (!this.destroyed) {
            int count = 0;
            for (int i = 0; i < this.connections.size(); ++i) {
                XPooledConnection xpc = this.connections.get(i);
                if (!xpc.isAvailable()) continue;
                ++count;
            }
            ret = count;
        }
        return ret;
    }

    public synchronized int totalSize() {
        if (this.destroyed) {
            return 0;
        }
        return this.connections.size();
    }

    @Override
    public synchronized void onXPooledConnectionTerminated(XPooledConnection connection) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": connection " + connection + " became available, notifying potentially waiting threads");
        }
        this.notify();
    }

    public String toString() {
        return "atomikos connection pool '" + this.name + "'";
    }
}

