package com.atlassian.xwork.interceptors;

import com.atlassian.util.profiling.ProfilingUtils;
import com.opensymphony.xwork.ActionInvocation;
import com.opensymphony.xwork.ActionProxy;
import com.opensymphony.xwork.interceptor.Interceptor;
import com.opensymphony.xwork.interceptor.PreResultListener;
import org.apache.log4j.Category;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.interceptor.DefaultTransactionAttribute;
import org.springframework.transaction.interceptor.TransactionAttribute;

public abstract class XWorkTransactionInterceptor implements Interceptor
{
    /**
     * Holder to support the currentTransactionStatus() method
     */
    private static ThreadLocal currentTransactionStatus = new ThreadLocal();
    private static Category log = Category.getInstance(XWorkTransactionInterceptor.class);

    /**
     * Return the transaction status of the current method invocation.
     * Mainly intended for code that wants to set the current transaction
     * rollback-only but not throw an application exception.
     *
     * @throws RuntimeException
     *          if the invocation cannot be found, because the method was invoked
     *          outside an AOP invocation context
     */
    public static TransactionStatus currentTransactionStatus() throws RuntimeException
    {
        TransactionStatus status = (TransactionStatus) currentTransactionStatus.get();
        if (status == null)
            throw new RuntimeException("No transaction status in scope");
        return status;
    }

    public void destroy()
    {
    }

    public void init()
    {
    }

    public abstract PlatformTransactionManager getTransactionManager();

    public String intercept(final ActionInvocation invocation) throws Exception
    {
        // if we're not setup yet, just invoke the action
        // If this is just a permissions check, we assume there's already a transaction in effect
        if (!shouldIntercept(invocation))
            return invocation.invoke();

        // If the transaction attribute is null, the method is non-transactional
        final TransactionAttribute transAtt = new DefaultTransactionAttribute(TransactionAttribute.PROPAGATION_REQUIRED);
        final TransactionStatus[] status = new TransactionStatus[1];
        TransactionStatus oldTransactionStatus = null;

        // We need a transaction for this method
        if (log.isDebugEnabled())
        {
            log.debug("Getting transaction for action '" + getDetails(invocation.getProxy()) + "'");
        }

        // The transaction manager will flag an error if an incompatible tx already exists
        status[0] = getTransactionManager().getTransaction(transAtt);

        // Make the TransactionStatus available to callees
        oldTransactionStatus = (TransactionStatus) currentTransactionStatus.get();
        currentTransactionStatus.set(status[0]);

        // Invoke the next interceptor in the chain.
        // This will normally result in a target object being invoked.
        String retVal = null;
        try
        {
            // We'll want to commit the transaction before the result is called.
            if (status[0] != null)
                invocation.addPreResultListener(new PreResultListener()
                {
                    public void beforeResult(ActionInvocation actionInvocation, String s)
                    {
                        if (log.isDebugEnabled())
                            log.debug("Committing transaction for " + getDetails(invocation.getProxy()) + " before result");

                        getTransactionManager().commit(status[0]);

                        if (log.isDebugEnabled())
                            log.debug("Opening new transaction for " + getDetails(invocation.getProxy()) + " result");

                        status[0] = getTransactionManager().getTransaction(transAtt);
                    }
                });

            retVal = invocation.invoke();
        }
        catch (Exception ex)
        {
            // Target invocation exception
            if (status[0] != null)
            {
                onThrowable(invocation, transAtt, status[0], ex);
            }
            throw ex;
        }
        catch (Throwable t)
        {
            // Target invocation exception
            if (status[0] != null)
            {
                onThrowable(invocation, transAtt, status[0], t);
            }
            throw new RuntimeException(t);
        }
        finally
        {
            if (transAtt != null)
            {
                // Use stack to restore old transaction status if one was set
                currentTransactionStatus.set(oldTransactionStatus);
            }
        }
        if (status[0] != null)
        {
            if (log.isDebugEnabled())
            {
                log.debug("Invoking commit for transaction on method '" + getDetails(invocation.getProxy()) + "'");
            }
            getTransactionManager().commit(status[0]);
        }
        return retVal;
    }

    protected abstract boolean shouldIntercept(ActionInvocation invocation);

    private String getDetails(ActionProxy proxy)
    {
        String methodName = proxy.getConfig().getMethodName();

        if (methodName == null)
            methodName = "execute";

        String actionClazz = ProfilingUtils.getJustClassName(proxy.getConfig().getClassName());

        return proxy.getNamespace() + "/" + proxy.getActionName() + ".action (" + actionClazz + "." + methodName + "())";
    }

    /**
     * Handle a throwable.
     * We may commit or roll back, depending on our configuration.
     */
    private void onThrowable(ActionInvocation invocation, TransactionAttribute txAtt, TransactionStatus status, Throwable ex)
    {
        try
        {
            if (txAtt.rollbackOn(ex))
            {
                log.error("Invoking rollback for transaction on action '" + getDetails(invocation.getProxy()) + "' due to throwable: " + ex, ex);
                getTransactionManager().rollback(status);
            }
            else
            {
                if (log.isDebugEnabled())
                    log.debug("Action '" + getDetails(invocation.getProxy()) + "' threw throwable but this does not force transaction rollback: " + ex, ex);
                // Will still roll back if rollbackOnly is true
                getTransactionManager().commit(status);
            }
        }
        catch (Exception e)
        {
            // If an exception occurs during the rollback, there's no point propagating it -- it
            // would just hide the exception that was the root cause, making problem determination
            // much harder
            log.error("Attempted rollback caused exception: " + e, e);
        }
    }
}
