package com.atlassian.velocity.htmlsafe.introspection;

import org.apache.commons.lang3.StringUtils;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.RuntimeServices;
import org.apache.velocity.runtime.log.Log;
import org.apache.velocity.util.introspection.SecureIntrospectorImpl;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

/**
 * SecureIntrospector that support allowlist for classes/packages that
 * override restrictions.
 * <p>
 * Elements on allowlist matches exact class or package without child classes/packages.
 *
 * <p>
 * Example:
 * <code>
 * introspector.restrict.packages = com.example
 * introspector.allow.packages = com.example
 * </code>
 * </p>
 * <p>
 * This configuration will restrict access to all packages under {@code com.example}
 * but classes directly under {@code com.example} will be available.
 * It mean {@code com.example.test.Class} will be blocked but
 * {@code com.example.Class} will be available.
 * </p>
 *
 * @since 3.2.0
 */
public class AllowlistSecureIntrospector extends SecureIntrospectorImpl {
    /**
     * A property containing a comma separated list of packages to allow access to in the SecureIntrospector.
     */
    public static final String INTROSPECTOR_ALLOW_PACKAGES = "introspector.allow.packages";
    /**
     * A property containing a comma separated list of classes to allow access to in the SecureIntrospector.
     */
    public static final String INTROSPECTOR_ALLOW_CLASSES = "introspector.allow.classes";
    private final Set<String> restrictedClasses;
    private final Set<String> allowedClasses;
    private final Set<String> allowedPackages;

    /**
     * Construct new instance of {@code AllowlistSecureIntrospector} using {@code runtimeServices} to read
     * configuration properties.
     *
     * @param log             A Log object to use for the introspector.
     * @param runtimeServices RuntimeServices object used to read configuration of introspector
     */
    public AllowlistSecureIntrospector(Log log, RuntimeServices runtimeServices) {
        this(Optional.ofNullable(runtimeServices.getConfiguration()
                        .getStringArray(RuntimeConstants.INTROSPECTOR_RESTRICT_CLASSES)),
                Optional.ofNullable(runtimeServices.getConfiguration()
                        .getStringArray(RuntimeConstants.INTROSPECTOR_RESTRICT_PACKAGES)),
                Optional.ofNullable(runtimeServices.getConfiguration()
                                .getStringArray(RuntimeConstants.INTROSPECTOR_ALLOWLIST_CLASSES)),
                Optional.ofNullable(runtimeServices.getConfiguration()
                        .getStringArray(INTROSPECTOR_ALLOW_CLASSES)),
                Optional.ofNullable(runtimeServices.getConfiguration()
                        .getStringArray(INTROSPECTOR_ALLOW_PACKAGES)),
                log, runtimeServices);
    }

    /**
     * Construct new instance of {@code AllowlistSecureIntrospector} with manually specified configuration
     *
     * @param restrictedClasses        list of classes which use is restricted in velocity templates. Classes in this list
     *                                 are always restricted by introspector.
     * @param restrictedParentPackages list of packages which use is restricted in velocity templates. It includes subpackages.
     * @param allowedClasses           list of classes which are exception from {@code restrictedParentPackages}. If class is in both
     *                                 {@code allowedClasses} and {@code restrictedClasses} parameters then introspector will
     *                                 treat it as missconfiguration and assume that class should be only in {@code restrictedClasses}
     * @param allowlistClasses         list of classes which are whitelist from {@code restrictedParentPackages}. In this case it will check
     *                                 super classes or interface are in restrict packages.
     * @param allowedPackages          list of packages which are exception from {@code restrictedParentPackages}. It does not include subpackages.
     *                                 Specifying the same package in both {@code allowedPackages} and {@code restrictedParentPackages}
     *                                 will allow use of classes directly in specified package but restrict use of
     *                                 classes from subpackages.
     *                                 If class is in {@code restrictedClasses} then it still will be restricted even if package
     *                                 of class is in {@code allowedPackages}.
     * @param log                      A Log object to use for the introspector.
     * @param runtimeServices          RuntimeServices object used to read configuration of introspector
     */
    public AllowlistSecureIntrospector(Optional<String[]> restrictedClasses, Optional<String[]> restrictedParentPackages, Optional<String[]> allowlistClasses, Optional<String[]> allowedClasses, Optional<String[]> allowedPackages, Log log, RuntimeServices runtimeServices) {
        super(restrictedClasses.orElse(new String[0]), restrictedParentPackages.orElse(new String[0]), allowlistClasses.orElse(new String[0]), log, runtimeServices);
        this.restrictedClasses = mapOptionalToSet(restrictedClasses);
        this.allowedClasses = mapOptionalToSet(allowedClasses);
        this.allowedPackages = mapOptionalToSet(allowedPackages);
    }

    @Override
    public boolean checkObjectExecutePermission(final Class classToCheck, final String methodToCheck) {
        boolean allowedByParent = super.checkObjectExecutePermission(classToCheck, methodToCheck);
        return allowedByParent || isClassListedAsExceptionFromRestrictRule(classToCheck);
    }

    private Set<String> mapOptionalToSet(final Optional<String[]> restrictedClasses) {
        return restrictedClasses
                .map(Arrays::asList)
                .map(HashSet::new)
                .map(Collections::unmodifiableSet)
                .orElseGet(Collections::emptySet);
    }

    private boolean isClassListedAsExceptionFromRestrictRule(final Class classToCheck) {
        String className = classToCheck.getName();
        String packageName = getPackageName(classToCheck, className);
        return (allowedPackages.contains(packageName) || allowedClasses.contains(className))
                && !restrictedClasses.contains(className);
    }

    private String getPackageName(Class classToCheck, String className) {
        if (classToCheck.getPackage() != null) {
            return classToCheck.getPackage().getName();
        } else {
            return StringUtils.substringBeforeLast(className, ".");
        }
    }
}
