package com.atlassian.vcache.internal.memcached;

import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.TransactionControlManager;
import com.atlassian.vcache.internal.core.VCacheCoreUtils;
import com.atlassian.vcache.internal.core.service.AbstractTransactionalExternalCache;
import com.atlassian.vcache.internal.core.service.VersionedExternalCacheRequestContext;
import net.spy.memcached.MemcachedClientIF;
import net.spy.memcached.OperationTimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.core.VCacheCoreUtils.unmarshall;
import static com.atlassian.vcache.internal.memcached.MemcachedUtils.expiryTime;
import static com.atlassian.vcache.internal.memcached.MemcachedUtils.putOperationForPolicy;
import static java.util.Objects.requireNonNull;

/**
 * Implementation that backs onto Memcached.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
class MemcachedTransactionalExternalCache<V>
        extends AbstractTransactionalExternalCache<V> {
    private static final Logger log = LoggerFactory.getLogger(MemcachedTransactionalExternalCache.class);

    private final Supplier<MemcachedClientIF> clientSupplier;
    private final ExternalCacheKeyGenerator keyGenerator;
    private final Marshaller<V> valueMarshaller;
    private final int ttlSeconds;
    private final TransactionControlManager transactionControlManager;

    MemcachedTransactionalExternalCache(
            Supplier<MemcachedClientIF> clientSupplier,
            Supplier<RequestContext> contextSupplier,
            ExternalCacheKeyGenerator keyGenerator,
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings,
            TransactionControlManager transactionControlManager) {
        super(name, contextSupplier);
        this.clientSupplier = requireNonNull(clientSupplier);
        this.keyGenerator = requireNonNull(keyGenerator);
        this.valueMarshaller = requireNonNull(valueMarshaller);
        this.ttlSeconds = VCacheCoreUtils.roundUpToSeconds(settings.getDefaultTtl().get());
        this.transactionControlManager = requireNonNull(transactionControlManager);
    }

    @Override
    public void transactionSync() {
        log.trace("Cache {}: synchronising operations", name);
        final VersionedExternalCacheRequestContext<V> cacheContext = ensureCacheContext();

        if (cacheContext.hasRemoveAll()) {
            cacheContext.updateCacheVersion(
                    MemcachedUtils.incrementCacheVersion(clientSupplier, cacheContext.externalCacheVersionKey()));
        }

        performKeyedOperations(cacheContext);
        cacheContext.forgetAll();
    }

    private void performKeyedOperations(VersionedExternalCacheRequestContext<V> cacheContext) {
        // The approach used to perform the operations is:
        // - schedule all the operations
        // - check the result of all the operations and log a warning is necessary
        // - if any operation failed, wipe the cache

        // Right then, lets submit all the operations.
        final Map<Future<Boolean>, String> futureToFailureMessageMap = new HashMap<>();
        cacheContext.getKeyedOperations().forEach(entry -> {
            final String externalKey = cacheContext.externalEntryKeyFor(entry.getKey());

            if (entry.getValue().isRemove()) {
                log.trace("Cache {}: performing remove on entry {}", name, entry.getKey());
                futureToFailureMessageMap.put(
                        clientSupplier.get().delete(externalKey),
                        "remove entry " + entry.getKey());
            } else {
                log.trace("Cache {}: performing {} on entry {}", name, entry.getValue().getPolicy(), entry.getKey());
                try {
                    final byte[] valueBytes = valueMarshaller.marshall(entry.getValue().getValue());
                    final Future<Boolean> putOp = putOperationForPolicy(
                            entry.getValue().getPolicy(),
                            clientSupplier.get(),
                            externalKey,
                            expiryTime(ttlSeconds),
                            valueBytes);
                    futureToFailureMessageMap.put(
                            putOp,
                            "put using policy " + entry.getValue().getPolicy() + " on entry "
                                    + entry.getKey());
                } catch (MarshallerException mex) {
                    log.warn("Cache {}: Unable to marshall value to perform put operation on entry {}",
                            name, entry.getKey(), mex);
                }
            }
        });

        // Cool, now lets check the result of each operation
        final boolean[] anOperationFailed = new boolean[1];
        futureToFailureMessageMap.entrySet().forEach(entry -> {
            try {
                final Boolean outcome = entry.getKey().get();
                if (outcome) {
                    log.trace("Cache {}: successful deferred operation for {}", name, entry.getValue());
                } else {
                    anOperationFailed[0] = true;
                    log.warn("Cache {}: failed deferred operation for {}", name, entry.getValue());
                }
            } catch (InterruptedException | ExecutionException ex) {
                anOperationFailed[0] = true;
                log.error("Cache {}: had failure getting result for deferred operation {}",
                        name, entry.getValue(), ex);
            }
        });

        // Finally, lets see if an operation failed
        if (anOperationFailed[0]) {
            log.error("Cache {}: an operation failed in transaction sync, so clearing the cache", name);
            cacheContext.updateCacheVersion(
                    MemcachedUtils.incrementCacheVersion(clientSupplier, cacheContext.externalCacheVersionKey()));
        }
    }

    @Nonnull
    @Override
    protected VersionedExternalCacheRequestContext<V> ensureCacheContext() throws OperationTimeoutException {
        final RequestContext requestContext = contextSupplier.get();

        return requestContext.computeIfAbsent(this, () -> {
            transactionControlManager.registerTransactionalExternalCache(requestContext, name, this);

            // Need to build a new context, which involves getting the current cache version, or setting it if it does
            // not exist.
            log.trace("Cache {}: Setting up a new context", name);
            final VersionedExternalCacheRequestContext<V> newCacheContext = new VersionedExternalCacheRequestContext<>(
                    keyGenerator, name, requestContext::partitionIdentifier);
            newCacheContext.updateCacheVersion(
                    MemcachedUtils.obtainCacheVersion(clientSupplier, newCacheContext.externalCacheVersionKey()));
            return newCacheContext;
        });
    }

    @Nonnull
    @Override
    protected Logger getLogger() {
        return log;
    }

    @Nonnull
    @Override
    protected final ExternalCacheException mapException(Exception ex) {
        return MemcachedUtils.mapException(ex);
    }

    @Nonnull
    @Override
    protected final Optional<V> directGet(String externalKey) {
        return unmarshall((byte[]) clientSupplier.get().get(externalKey), valueMarshaller);
    }

    @Nonnull
    @Override
    protected final Map<String, Optional<V>> directGetBulk(Set<String> externalKeys) {
        return MemcachedUtils.directGetBulk(externalKeys, clientSupplier, valueMarshaller);
    }
}
