package com.atlassian.vcache.internal.memcached;

import com.atlassian.vcache.internal.BegunTransactionalActivityHandler;
import com.atlassian.vcache.internal.ExternalCacheExceptionListener;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import com.atlassian.vcache.internal.VCacheSettingsDefaultsProvider;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.Sha1ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.metrics.DefaultMetricsCollector;
import com.atlassian.vcache.internal.core.metrics.MetricsCollector;
import net.spy.memcached.MemcachedClientIF;

import java.time.Duration;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Builder for creating {@link MemcachedVCacheServiceSettings} instances.
 * <p>
 * All settings must be set before {@link #build()} can be called, except for
 * {@link #metricsCollector(MetricsCollector)}, {@link #begunTransactionalActivityHandler(BegunTransactionalActivityHandler)}
 * and {@link #dontExternaliseCache(Function)}. See their Javadoc for the defaults used.
 * </p>
 * <p>
 * Either {@link #externalCacheKeyGenerator(ExternalCacheKeyGenerator)} or {@link #productIdentifier(String)} must be
 * called.
 * </p>
 *
 * @since 1.3.0
 */
public class MemcachedVCacheServiceSettingsBuilder {
    private Supplier<MemcachedClientIF> clientSupplier;
    private Supplier<RequestContext> threadLocalContextSupplier;
    private Supplier<RequestContext> workContextContextSupplier;
    private VCacheSettingsDefaultsProvider defaultsProvider;
    private VCacheCreationHandler creationHandler;
    private MetricsCollector metricsCollector;
    private ExternalCacheKeyGenerator externalCacheKeyGenerator;
    // By default, don't care
    private BegunTransactionalActivityHandler begunTransactionalActivityHandler = context -> {
    };
    // By default, externalise the external caches.
    private Function<String, Boolean> dontExternaliseCache = name -> false;
    // By default, don't use this hack
    private boolean serializationHack;
    // By default, wait a fixed time
    @SuppressWarnings("checkstyle:MagicNumber")
    private Duration lockTimeout = Duration.ofSeconds(30);
    // By default, don't care about listening
    private ExternalCacheExceptionListener externalCacheExceptionListener = (n, ex) -> {};

    /**
     * Sets the client supplier.
     *
     * @param clientSupplier the client supplier.
     * @return the builder
     */
    public MemcachedVCacheServiceSettingsBuilder clientSupplier(Supplier<MemcachedClientIF> clientSupplier) {
        this.clientSupplier = requireNonNull(clientSupplier);
        return this;
    }

    /**
     * Sets the context supplier.
     *
     * @param threadLocalContextSupplier the context supplier.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder threadLocalContextSupplier(Supplier<RequestContext> threadLocalContextSupplier) {
        this.threadLocalContextSupplier = requireNonNull(threadLocalContextSupplier);
        return this;
    }

    /**
     * Sets the context supplier for caches that can be thread safe.
     *
     * @param contextSupplier the context supplier.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder workContextContextSupplier(Supplier<RequestContext> contextSupplier) {
        this.workContextContextSupplier = contextSupplier;
        return this;
    }

    /**
     * Sets the defaults provider.
     *
     * @param defaultsProvider the defaults provider.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder defaultsProvider(VCacheSettingsDefaultsProvider defaultsProvider) {
        this.defaultsProvider = requireNonNull(defaultsProvider);
        return this;
    }

    /**
     * Set the creation handler.
     *
     * @param creationHandler the creation handler.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder creationHandler(VCacheCreationHandler creationHandler) {
        this.creationHandler = requireNonNull(creationHandler);
        return this;
    }

    /**
     * Sets the metrics collector. If not invoked, then the {@link DefaultMetricsCollector} implementation will be
     * used.
     *
     * @param metricsCollector the metrics collector.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder metricsCollector(MetricsCollector metricsCollector) {
        this.metricsCollector = requireNonNull(metricsCollector);
        return this;
    }

    /**
     * Equivalent to calling {@link #externalCacheKeyGenerator(ExternalCacheKeyGenerator)} passing a
     * {@link Sha1ExternalCacheKeyGenerator} instance created using the supplied product identifier.
     *
     * @param productIdentifier the product identifier to use with the {@link Sha1ExternalCacheKeyGenerator}.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder productIdentifier(String productIdentifier) {
        this.externalCacheKeyGenerator = new Sha1ExternalCacheKeyGenerator(productIdentifier);
        return this;
    }

    /**
     * Set the external cache key generator.
     *
     * @param externalCacheKeyGenerator the external cache key generator.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder externalCacheKeyGenerator(ExternalCacheKeyGenerator externalCacheKeyGenerator) {
        this.externalCacheKeyGenerator = requireNonNull(externalCacheKeyGenerator);
        return this;
    }

    /**
     * Sets the begun transaction activity handler. If not invoked, then a default implementation is provided that does
     * nothing.
     *
     * @param handler the begun transaction activity handler.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder begunTransactionalActivityHandler(BegunTransactionalActivityHandler handler) {
        this.begunTransactionalActivityHandler = requireNonNull(handler);
        return this;
    }

    /**
     * Sets the function to determine whether an {@link com.atlassian.vcache.ExternalCache} should really be externalised.
     * If the supplied function returns <tt>true</tt> for a cache, then an in-memory implementation will be used.
     * <p>
     * If not set, then a default implementation is provided that will externalise all {@link com.atlassian.vcache.ExternalCache}'s.
     * nothing.
     * </p>
     * <p>
     * Note: this function is provided to allow for an orderly migration of external caches from being in-memory
     * to be stored in Memcached.
     * </p>
     *
     * @param dontExternaliseCache the function to determine whether an {@link com.atlassian.vcache.ExternalCache}
     *                                 should really be externalised.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder dontExternaliseCache(Function<String, Boolean> dontExternaliseCache) {
        this.dontExternaliseCache = requireNonNull(dontExternaliseCache);
        return this;
    }

    /**
     * Enable the serialization hack, whereby if an {@link com.atlassian.vcache.ExternalCache}'s values are
     * {@link java.io.Serializable}, then the values are not marshalled before they are passed to the delegate
     * Atlassian Cache. This is to allow for performance optimisations.
     */
    public MemcachedVCacheServiceSettingsBuilder enableSerializationHack() {
        serializationHack = true;
        return this;
    }

    /**
     * Change the timeout that is used when acquiring locks.
     * @param lockTimeout the timeout that is used when acquiring locks.
     * @return the current builder
     */
    public MemcachedVCacheServiceSettingsBuilder lockTimeout(Duration lockTimeout) {
        this.lockTimeout = requireNonNull(lockTimeout);
        return this;
    }

    /**
     * Set the external cache exception listener.
     *
     * @param externalCacheExceptionListener the external cache exception listener.
     * @return the builder.
     */
    public MemcachedVCacheServiceSettingsBuilder externalCacheExceptionListener(ExternalCacheExceptionListener externalCacheExceptionListener) {
        this.externalCacheExceptionListener = requireNonNull(externalCacheExceptionListener);
        return this;
    }

    /**
     * Returns a new {@link MemcachedVCacheServiceSettings} instance configured using the supplied settings.
     *
     * @return a new {@link MemcachedVCacheServiceSettings} instance configured using the supplied settings.
     */
    public MemcachedVCacheServiceSettings build() {
        return new MemcachedVCacheServiceSettings(
                requireNonNull(clientSupplier, "missing clientSupplier"),
                requireNonNull(threadLocalContextSupplier, "missing threadLocalContextSupplier"),
                requireNonNull(workContextContextSupplier, "missing thread safe workContextContextSupplier"),
                requireNonNull(defaultsProvider, "missing defaultsProvider"),
                requireNonNull(creationHandler, "missing creationHandler"),
                (metricsCollector != null) ? metricsCollector : new DefaultMetricsCollector(threadLocalContextSupplier),
                requireNonNull(externalCacheKeyGenerator, "missing externalCacheKeyGenerator"),
                begunTransactionalActivityHandler,
                dontExternaliseCache,
                serializationHack,
                lockTimeout,
                externalCacheExceptionListener);
    }
}
