package com.atlassian.vcache.internal.legacy;

import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.internal.BegunTransactionalActivityHandler;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import com.atlassian.vcache.internal.VCacheSettingsDefaultsProvider;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.Sha1ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.metrics.MetricsCollector;
import com.atlassian.vcache.internal.core.service.AbstractVCacheService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Main service that is backed by <tt>Atlassian Cache</tt>.
 *
 * @since 1.0.0
 */
public class LegacyVCacheService extends AbstractVCacheService {
    private static final Logger log = LoggerFactory.getLogger(LegacyVCacheService.class);

    private final Supplier<CacheFactory> cacheFactorySupplier;
    private final LegacyServiceSettings serviceSettings;

    public LegacyVCacheService(
            String productIdentifier,
            Supplier<RequestContext> threadLocalContextSupplier,
            Supplier<RequestContext> workContextContextSupplier,
            VCacheSettingsDefaultsProvider defaultsProvider,
            VCacheCreationHandler creationHandler,
            MetricsCollector metricsCollector,
            Supplier<CacheFactory> cacheFactorySupplier,
            LegacyServiceSettings serviceSettings,
            BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        super(threadLocalContextSupplier,
                workContextContextSupplier,
                defaultsProvider,
                creationHandler,
                metricsCollector,
                new Sha1ExternalCacheKeyGenerator(productIdentifier),
                begunTransactionalActivityHandler,
                serviceSettings.getLockTimeout());
        this.cacheFactorySupplier = requireNonNull(cacheFactorySupplier);
        this.serviceSettings = requireNonNull(serviceSettings);
    }

    public LegacyVCacheService(
            Supplier<RequestContext> threadLocalContextSupplier,
            Supplier<RequestContext> workContextContextSupplier,
            VCacheSettingsDefaultsProvider defaultsProvider,
            VCacheCreationHandler creationHandler,
            MetricsCollector metricsCollector,
            ExternalCacheKeyGenerator externalCacheKeyGenerator,
            Supplier<CacheFactory> cacheFactorySupplier,
            LegacyServiceSettings serviceSettings,
            BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        super(threadLocalContextSupplier,
                workContextContextSupplier,
                defaultsProvider,
                creationHandler,
                metricsCollector,
                externalCacheKeyGenerator,
                begunTransactionalActivityHandler, serviceSettings.getLockTimeout());
        this.cacheFactorySupplier = requireNonNull(cacheFactorySupplier);
        this.serviceSettings = requireNonNull(serviceSettings);
    }

    @Override
    protected Logger log() {
        return log;
    }

    @Override
    protected <K, V> JvmCache<K, V> createJvmCache(String name, JvmCacheSettings settings) {
        final CacheSettings legacySettings = new CacheSettingsBuilder()
                .local()
                .maxEntries(settings.getMaxEntries().get())
                .expireAfterWrite(settings.getDefaultTtl().get().toNanos(), TimeUnit.NANOSECONDS)
                .build();

        return new LegacyJvmCache<>(
                cacheFactorySupplier.get().getCache(name, null, legacySettings),
                serviceSettings.getLockTimeout());
    }

    @Override
    protected <V> TransactionalExternalCache<V> createTransactionalExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable) {
        final CacheSettings legacySettings = buildLegacySettings(settings);

        return new LegacyTransactionalExternalCache<>(
                cacheFactorySupplier.get().getCache(name, null, legacySettings),
                threadLocalContextSupplier,
                externalCacheKeyGenerator,
                (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                transactionControlManager,
                serviceSettings,
                metricsCollector);
    }

    @Override
    protected <V> StableReadExternalCache<V> createStableReadExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable) {
        final CacheSettings legacySettings = buildLegacySettings(settings);

        return new LegacyStableReadExternalCache<>(
                cacheFactorySupplier.get().getCache(name, null, legacySettings),
                workContextContextSupplier,
                externalCacheKeyGenerator,
                (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                serviceSettings,
                metricsCollector);
    }

    @Override
    protected <V> DirectExternalCache<V> createDirectExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable) {
        final CacheSettings legacySettings = buildLegacySettings(settings);

        return new LegacyDirectExternalCache<>(
                cacheFactorySupplier.get().getCache(name, null, legacySettings),
                workContextContextSupplier,
                externalCacheKeyGenerator,
                (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                serviceSettings);
    }

    private CacheSettings buildLegacySettings(ExternalCacheSettings settings) {
        return new CacheSettingsBuilder()
                .remote()
                .maxEntries(settings.getEntryCountHint().get())
                .expireAfterWrite(settings.getDefaultTtl().get().toNanos(), TimeUnit.NANOSECONDS)
                .build();
    }
}
